package com.mapr.db.mapreduce.impl;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.TreeSet;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.io.NullWritable;
import org.apache.hadoop.io.SequenceFile;
import org.ojai.Document;
import org.ojai.DocumentStream;
import org.ojai.Value;
import org.ojai.Value.Type;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.mapr.db.MapRDB;
import com.mapr.db.Table;
import com.mapr.db.TabletInfo;
import com.mapr.db.impl.ConditionNode.RowkeyRange;
import com.mapr.db.impl.IdCodec;
import com.mapr.db.impl.MapRDBTableImpl;
import com.mapr.db.impl.TabletInfoImpl;
import com.mapr.db.rowcol.DBDocumentImpl;
import com.mapr.fs.MapRFileSystem;
import com.mapr.fs.proto.Common.ServiceData;
import com.mapr.org.apache.hadoop.hbase.util.Bytes;
import com.mapr.util.zookeeper.ZKDataRetrieval;

/* utility methods for difftable tools */
public class DiffTableUtils {

  public class ServiceName {
    public static final String HISTORYSERVERNAME = "historyserver";
    public static final String RESOURCEMANAGERNAME = "resourcemanager";
  }

  private static final Logger LOG = LoggerFactory.getLogger(DiffTableUtils.class);

  public static final String KEYPREFIX = "k_";
  public static final String FIRSTKEYNAME = "neginf";
  public static final String LASTKEYNAME = "inf";
  public static final String SucceedFileName = "_SUCCESS";
  public static final String outDirConf = "outdirconf";


  public static Path[] validateAndCreateDirs(FileSystem fs, Path outDir) throws IOException {
    /* validate that the outDir doesn't exist and create a new one */
    try {
      if (fs.exists(outDir)) {
        System.err.println("Output dir " + outDir + " already exists");
        return null;
      }
    } catch (IOException e) {
      System.err.println("Failed to check the status of dir " + outDir);
      throw e;
    }

    try {
      /* create the new dir */
      if (!fs.mkdirs(outDir)) {
        System.err.println("Failed to create dir " + outDir);
        return null;
      }
    } catch (IOException e) {
      System.err.println("Failed to create dir " + outDir);
      throw e;
    }

    Path[] srcDestPaths = new Path[2];
    Path opsForTable1Dir = new Path(outDir, "OpsForSrcTable");
    try {
      /* create the new dir */
      if (!fs.mkdirs(opsForTable1Dir)) {
        System.err.println("Failed to create dir " + opsForTable1Dir);
        return null;
      }
    } catch (IOException e) {
      System.err.println("Failed to create dir " + opsForTable1Dir);
      throw e;
    }

    srcDestPaths[0] = opsForTable1Dir;

    Path opsForTable2Dir = new Path(outDir, "OpsForDstTable");
    try {
      /* create the new dir */
      if (!fs.mkdirs(opsForTable2Dir)) {
        System.err.println("Failed to create dir " + opsForTable2Dir);
        return null;
      }
    } catch (IOException e) {
      System.err.println("Failed to create dir " + opsForTable2Dir);
      throw e;
    }

    srcDestPaths[1] = opsForTable2Dir;

    return srcDestPaths;
  }

  public static Path[] createOutputDirs(FileSystem fs, Path outDir) throws IOException {
    /* we will assume that outputDir already exists. If not, then throw an exception */
    try {
      if (!fs.exists(outDir)) {
        System.err.println("Output dir " + outDir + " does not exists");
        return null;
      }
    } catch (IOException e) {
      System.err.println("Failed to check the status of dir " + outDir);
      throw e;
    }

    Path[] srcDestPaths = new Path[2];
    Path opsForTable1Dir = new Path(outDir, "OpsForSrcTable");
    try {
      /* create the new dir */
      if (!fs.mkdirs(opsForTable1Dir)) {
        System.err.println("Failed to create dir " + opsForTable1Dir);
        return null;
      }
    } catch (IOException e) {
      System.err.println("Failed to create dir " + opsForTable1Dir);
      throw e;
    }

    srcDestPaths[0] = opsForTable1Dir;

    Path opsForTable2Dir = new Path(outDir, "OpsForDstTable");
    try {
      /* create the new dir */
      if (!fs.mkdirs(opsForTable2Dir)) {
        System.err.println("Failed to create dir " + opsForTable2Dir);
        return null;
      }
    } catch (IOException e) {
      System.err.println("Failed to create dir " + opsForTable2Dir);
      throw e;
    }

    srcDestPaths[1] = opsForTable2Dir;

    return srcDestPaths;
  }

  //copy all files from src to dst within the filesystem fs.
  public static boolean copyFileWithPrefix(FileSystem fs, Path srcDir, Path dstDir,
      Configuration conf,
      String prefix, Logger LOG) throws IOException {
    if (!fs.exists(srcDir)) {
      LOG.error("Source " + srcDir + " doesn't exist.");
      return false;
    }
    if (!fs.getFileStatus(srcDir).isDirectory()) {
      LOG.error("Source " + srcDir + " is not a directory.");
      return false;
    }

    if (fs.exists(dstDir)) {
      LOG.warn("Files in destination directory " + dstDir + " will be overwritten.");
      if (!fs.delete(dstDir, true /*recursive*/)) {
        LOG.error("Fail to replace " + dstDir);
        return false;
      }
    }

    if (!fs.mkdirs(dstDir)) {
      LOG.error("Fail to create " + dstDir);
      return false;
    }

    boolean moved = false;
    FileStatus contents[] = fs.listStatus(srcDir);
    System.out.println("util1  "+srcDir.toString()+ " contents "+contents.length+" "+prefix);
    Arrays.sort(contents);
    for (int i = 0; i < contents.length; i++) {
      Path srcPath = contents[i].getPath();
      System.out.println("util 2 "+srcPath.toString()+" "+srcPath.getName()+" prefix "+prefix);
      if (contents[i].isFile() && srcPath.getName().startsWith(prefix)) {
        Path dstPath = new Path(dstDir+"/"+srcPath.getName());
        System.out.println("src and dst "+srcPath.toString()+" "+dstPath.toString());
        if (!fs.rename(srcPath, dstPath)) {
          LOG.error("Fail to move "+srcPath.toString()+" to "+dstPath.toString());
          return false;
        }
        moved = true;
        LOG.debug("Moved "+srcPath.toString()+" to "+dstPath.toString());
      }
    }
    // If no file is moved to the dstDir, remove the empty directory
    if (!moved) {
      if (!fs.delete(dstDir, true /*recursive*/)) {
        LOG.error("Fail to cleanup empty " + dstDir);
        return false;
      }
    }
    return true;
  }

  public static String getTmpDirName(Configuration conf) {
    //The directory is per user, for example, if user is mapr, the directory
    //is /tmp/hadoop-mapr/; if user is root, it becomes /tmp/hadoop-root/
    return conf.get("hadoop.tmp.dir");
  }

  public static String getTabletKeyFileName(String tblName) {
    //Pure file name, directory not included.
    return tblName.replace("/", "").replace(":", "") + ".regionkeys";
  }

  public static String getReSplitedTabletKeyFileName(String tblName) {
    //Pure file name, directory not included.
    return tblName.replace("/", "").replace(":", "") + ".resplitedregionkeys";
  }

  public static String getSegKeyFileName(String tblName) {
    //Pure file name, directory not included.
    return tblName.replace("/", "").replace(":", "") + ".segmentkeys";
  }

  public static String getSegKeyDirName(String tblName) {
    //Directory name from current directory, parent directory not included.
    return tblName.replace("/", "").replace(":", "") + ".startkeys";
  }

  public static String getServiceAddress(Configuration conf, String clusterUri, String serviceName)
      throws IOException, URISyntaxException {

      String zkAddress = null;
      MapRFileSystem mfs = null;
      try {
        mfs = new MapRFileSystem();
        mfs.initialize(new URI(clusterUri), conf);
        zkAddress = mfs.getZkConnectString();
        LOG.info("zkAddress(" + zkAddress + ")");
      } catch (IOException e) {
        LOG.error("Zookeeper address not found from MapRFilesystem from cluster " + clusterUri + ", "+ e);
        zkAddress = null;
      } finally {
        mfs.close();
      }

      if(zkAddress == null) {
        LOG.error("Zookeeper address is null from MapRFilesystem from cluster " + clusterUri);
        return null;
      }

      ZKDataRetrieval zkConnection = new ZKDataRetrieval(zkAddress);
      ServiceData hostInfo = zkConnection.getServiceMasterData(serviceName);

      if(hostInfo == null) {
        LOG.error("Unable to determine service " + serviceName + " address from Zookeeper at " + zkAddress + " from cluster " + clusterUri);
        return null;
      }

      String hostName = hostInfo.getHost();
      int servicePort = hostInfo.getPort();
      LOG.info(serviceName + " address: host name(" + hostName +"), port("+ servicePort +")");
      if (servicePort > 0) {
        return hostName + ":" + servicePort;
      } else {
        return hostName;
      }
  }

  public static boolean checkPathExists(Configuration conf, ClusterTablePath cPath) throws IOException{
    Path path = cPath.getFullPath();
    FileSystem fs = path.getFileSystem(conf);
    // check its existence
    if (fs.exists(path)) {
      return true;
    } else {
      return false;
    }
  }


  static List<ByteBufWritableComparable> getRegionStartKeys(Table table)
      throws IOException {
  TabletInfo[] tabletInfos = table.getTabletInfos();
  ArrayList<ByteBufWritableComparable> ret = new ArrayList<ByteBufWritableComparable>(tabletInfos.length);
  for (TabletInfo tInfo : tabletInfos) {
    byte[] sKey = ((TabletInfoImpl) tInfo).getStartRow();
    ret.add(new ByteBufWritableComparable(ByteBuffer.wrap(sKey)));
  }
  return ret;
}

  public static void printKeyRange(List<ByteBufWritableComparable> startKeys) {
    System.out.println("printKeyRange:");
    if (startKeys == null || startKeys.isEmpty()) {
      System.err.println("Input key list is empty\n");
    } else {
      int i = 0;
      for(ByteBufWritableComparable key : startKeys) {
        System.out.println("StartKey of Range " + i + " : ("  + key + ")");
        ++i;
      }
    }
  }

  public static void printKeyRanges(List<RowkeyRange> keys) {
    LOG.info("StartEndKeys");
    int i=0;
    for(RowkeyRange rkey : keys) {
      LOG.info("Range " + i + " : "  + Bytes.toStringBinary(rkey.getStartRow()) + ",  " + Bytes.toStringBinary(rkey.getStopRow()));
      ++i;
    }
  }

  public static void logKeyRanges(List<RowkeyRange> sekeys) {
    LOG.debug("StartEndKeys");
    int i=0;
    for(RowkeyRange rkey : sekeys) {
      LOG.debug("Range " + i + " : "  + rkey.toString());
      ++i;
    }
  }

  public static List<RowkeyRange> GenStartEndKeys(
      List<ByteBufWritableComparable> startKeys) throws IOException {

      if (startKeys.isEmpty()) {
        throw new IllegalArgumentException("No regions passed");
      }

      List<ByteBufWritableComparable> sklist = new ArrayList<ByteBufWritableComparable>(startKeys.size());
      List<ByteBufWritableComparable> eklist = new ArrayList<ByteBufWritableComparable>(startKeys.size());

      for (ByteBufWritableComparable k : startKeys) {
        sklist.add(k);
      }
      if (startKeys.size() > 0) {
        for (ListIterator<ByteBufWritableComparable> iter = startKeys.listIterator(1); iter.hasNext(); ) {
          ByteBufWritableComparable k = iter.next();
          eklist.add(k);
        }
        byte [] lastEndKey = new byte[0];
        eklist.add(new ByteBufWritableComparable(ByteBuffer.wrap(lastEndKey)));
      }

      List<RowkeyRange> rangeList = new ArrayList<RowkeyRange>(startKeys.size());
      ListIterator<ByteBufWritableComparable> skIter = sklist.listIterator(0);
      ListIterator<ByteBufWritableComparable> ekIter = eklist.listIterator(0);
      while (skIter.hasNext() && ekIter.hasNext()) {
        RowkeyRange keyRange = new RowkeyRange(skIter.next().getBytes(), ekIter.next().getBytes());
        rangeList.add(keyRange);
      }
      if (skIter.hasNext() || ekIter.hasNext()) {
        throw new IllegalArgumentException("startKeys has size " + sklist.size() +", but endKeys has size " + eklist.size());
      }
      return rangeList;
  }

  public static List<ByteBufWritableComparable> readKeyRange(Configuration conf, String partitionsPathName)
      throws IOException {
    return readKeyRange(conf, new Path(partitionsPathName));
  }

  public static List<ByteBufWritableComparable> readKeyRange(Configuration conf, Path partitionsPath)
      throws IOException {

    ArrayList<ByteBufWritableComparable> ret = new ArrayList<ByteBufWritableComparable>();

    FileSystem fs = partitionsPath.getFileSystem(conf);

    @SuppressWarnings("deprecation")
    SequenceFile.Reader reader = new SequenceFile.Reader(fs, partitionsPath, conf);
    ByteBuffer key1 = null;
    ByteBuffer key2 = null;
    ByteBufWritableComparable iKey = new ByteBufWritableComparable();

    NullWritable value = NullWritable.get();
    LOG.debug("readSegKeyRange: " + partitionsPath.toString());
    int i = 0;
    while (reader.next(iKey, value)) {
      key1 = key2;
      key2 = iKey.getByteBuf();
      if (i>0 && (key1.compareTo(key2) >= 0)) {
        reader.close();
        throw new IOException(
              "Current keyrange " + i + " startky(" + Bytes.toStringBinary(key2)
              + ") should be larger than the previous keyrange startkey("
              + Bytes.toStringBinary(key1) + ")");
      }
      LOG.debug("\t (" + Bytes.toStringBinary(key2) + ")");
      ret.add(new ByteBufWritableComparable(key2));
      ++i;
    }
    LOG.debug("Done readSegKeyRange: " + partitionsPath.toString() + ", list length=" + Integer.toString(i));

    reader.close();
    return ret;
  }

  public static void writeKeyRange(Configuration conf, String partitionsPathName,
      List<ByteBufWritableComparable> startKeys) throws IOException {
      writeKeyRange(conf, new Path(partitionsPathName), startKeys);
  }

    // The input startKeys must be increasing order.
  public static void writeKeyRange(Configuration conf, Path partitionsPath,
      List<ByteBufWritableComparable> startKeys) throws IOException {
      if (startKeys.isEmpty()) {
        throw new IllegalArgumentException("No regions passed");
      }

      // Write the actual file
      Path parentPath = partitionsPath.getParent();
      FileSystem fs = partitionsPath.getFileSystem(conf);
      if (!fs.exists(parentPath)) {
        LOG.debug("create directory " + parentPath.toString());
        fs.mkdirs(parentPath);
      }
      @SuppressWarnings("deprecation")
      SequenceFile.Writer writer = SequenceFile.createWriter(fs,
          conf, partitionsPath, ByteBufWritableComparable.class, NullWritable.class);

      LOG.debug("writeSegKeyRange to file " + partitionsPath.toString());
      ByteBufWritableComparable prevKey = null; //the previous start key
      try {
        for (ByteBufWritableComparable startKey : startKeys) {
          if ((prevKey != null) && (prevKey.compareTo(startKey) >= 0)) {
            //Previous key should be less than the current key
              throw new IOException(
                    "Current startky(" + startKey.toString()
                    + ") should be larger than the previous startkey("
                    + prevKey.toString() + ")");
          }

          LOG.debug("\t (" + startKey.toString()+")");
          ByteBuffer startKeyBuf = startKey.getByteBuf();
          writer.append(new ByteBufWritableComparable(startKeyBuf), NullWritable.get());
          prevKey = startKey;
        }

      } finally {
        writer.close();
      }
  }

  public static void copyKeyRangeFile(Configuration srcConf, String srcPathName,
      Configuration dstConf, String dstPathName) throws IOException {
    List<ByteBufWritableComparable> startKeys = readKeyRange(srcConf, srcPathName);
    writeKeyRange(dstConf, dstPathName, startKeys);
  }

  public static void copyKeyRangeFile(Configuration srcConf, Path srcPath,
      Configuration dstConf, Path dstPath) throws IOException {
    List<ByteBufWritableComparable> startKeys = readKeyRange(srcConf, srcPath);
    writeKeyRange(dstConf, dstPath, startKeys);
  }

  public static void writeTabletKeyRange(String tblKeyPathName, String tblName)
      throws IOException {

  Configuration conf = new Configuration();

  Table table = MapRDB.getTable(tblName);
  // Get the tablet start keys.
  List<ByteBufWritableComparable> tblKeys = getRegionStartKeys(table);
  TreeSet<ByteBufWritableComparable> sortedTblKeys =
      new TreeSet<ByteBufWritableComparable>(tblKeys);

  ByteBufWritableComparable first = sortedTblKeys.first();
  if (!first.equals(ByteBufWritableComparable.EMPTY_BYTEBUFFER)) {
    throw new IllegalArgumentException(
          "First region of table should have empty start key. Instead has("
          + first.toString() + ") with size " + first.getLength());
  }

  ByteBufWritableComparable[] sortedList = sortedTblKeys.toArray(new ByteBufWritableComparable[sortedTblKeys.size()]);
  LOG.info("write region start keys to " + tblKeyPathName + " for table " + tblName);
  writeKeyRange(conf, tblKeyPathName, Arrays.asList(sortedList));

}


  // return the number of segments.
  public static int writeSegKeyRange(String segKeyPathName,
      String tblName) throws IOException {

  Configuration conf = new Configuration();
  LOG.debug("writeSegKeyRange to file " + segKeyPathName +" for table " + tblName);
  //Prepare the file
  Path partitionsPath = new Path(segKeyPathName);
  FileSystem fs = partitionsPath.getFileSystem(conf);
  @SuppressWarnings("deprecation")
  SequenceFile.Writer writer = SequenceFile.createWriter(fs,
      conf, partitionsPath, ByteBufWritableComparable.class, NullWritable.class);

  Table table = MapRDB.getTable(tblName);
  DocumentStream rs = ((MapRDBTableImpl) table).segmentKeyScan();
  Iterator<Document> itrs = rs.iterator();

  int segnum = 0;
  boolean isEmpty = false;
  Document res = null;
  if (!itrs.hasNext()) {
    //No segments
    isEmpty = true;
  } else {
    //Have a segment, but it is empty
    res = itrs.next();
    ++segnum;
    if (res == null) {
      isEmpty = true;
    }
  }
  if (isEmpty) {
    // Empty table is a valid candidate for table comparison.
    table.close();
    if (LOG.isInfoEnabled()) {
      LOG.info("No segment key range returned from server. Write only -INF(EMPTY_BYTE_ARRAY) segment start key to " + partitionsPath.toString() + " for empty table " + tblName);
    }
    LOG.debug("\t (" + ByteBufWritableComparable.EMPTY_BYTEBUFFER +")");
    writer.append(ByteBufWritableComparable.EMPTY_BYTEBUFFER, NullWritable.get());
    writer.close();
    return segnum;
  }

  //When we have rowkey other than binary or string, we need to change this to getId.
  Value firstId = ((DBDocumentImpl) res).getId();
  Type idType = firstId.getType();
  boolean firstIsEmpty = true;
  if ( idType == Type.STRING) {
    if (firstId.getString() != "") {
      // The first segment is empty, the segment scan skipped it
      LOG.info("First region of table does not have any data. We will add the empty start key "
        + " to cover these beginning empty regions. The first region with data has start key ("
        + firstId.getString() + ")");
      firstIsEmpty = false;
    }
  } else if (idType == Type.BINARY) {
    // The first segment is empty, the segment scan skipped it
    if (!(firstId.getBinary().equals(ByteBuffer.wrap(new byte[0])))) {
      // The first segment is empty, the segment scan skipped it
      LOG.info("First region of table does not have any data. We will add the empty start key "
        + " to cover these beginning empty regions. The first region with data has start key ("
        + Bytes.toStringBinary(firstId.getBinary())  + ")");
      firstIsEmpty = false;
    }
  } else {
    table.close();
    throw new IOException(
        "Only String and Binary Document IDs are supported, and this ID has type "
        + idType + ")");
  }

  //Write the actual file
  LOG.debug("write segment start keys to " + partitionsPath.toString() + " for table " + tblName);
  LOG.debug("\t (" + ByteBufWritableComparable.EMPTY_BYTEBUFFER.toString() + ")");
  writer.append(ByteBufWritableComparable.EMPTY_BYTEBUFFER, NullWritable.get());


  byte[] rKey1 = Bytes.getBytes(IdCodec.encode(((DBDocumentImpl) res).getId()));
  byte[] rKey2 = rKey1;

  ByteBufWritableComparable iKey = null;
  if (!firstIsEmpty) {
    //Need to write this non-empty key
    iKey = new ByteBufWritableComparable(ByteBuffer.wrap(rKey2));
    LOG.debug("\t (" + Bytes.toStringBinary(rKey2)+")");
    writer.append(iKey, NullWritable.get());
  }

  //Rest of the keys should be ordered.
  while (itrs.hasNext()) {
    res = itrs.next();
    ++segnum;

    rKey1 = rKey2;
    rKey2 = Bytes.getBytes(IdCodec.encode(((DBDocumentImpl) res).getId()));;
    if (Bytes.compareTo(rKey1,rKey2) >= 0) {
      // previous key should be smaller than the current key.
      table.close();
      writer.close();
      throw new IOException(
            "Current keyrange startky(" + Bytes.toStringBinary(rKey2)
            + ") should be larger than the previous keyrange startkey("
            + Bytes.toStringBinary(rKey1) + ")");
    }
    iKey = new ByteBufWritableComparable(ByteBuffer.wrap(rKey2));
    LOG.debug("\t (" + Bytes.toStringBinary(rKey2)+")");
    writer.append(iKey, NullWritable.get());

  }
  table.close();
  writer.close();
  LOG.debug("Total " + segnum + " segement startKey written.");
  return segnum;
}

  /* Read the segment keys from inputSegKeyFile, and split the input segment keys into small files, which
   * will be written to outputSegKeyDir, based on the start keys in the inputTblKeyFile. And write the
   * start keys of each split into file outputSplitKeyFile.
   */
  public static void splitSubRegionKeysbyRegionKeys(String inputRegionKeyFileName, String inputSubRegionKeyFileName,
        String outputSplitKeyFileName, String outputSubRegionKeyDir)
        throws IOException {
    Configuration conf = new Configuration();
    // tblKeys has been sorted before written to the file.
    List<ByteBufWritableComparable> tblKeys = readKeyRange(conf, inputRegionKeyFileName);
    splitSubRegionKeysbyRegionKeys(tblKeys, inputSubRegionKeyFileName, outputSplitKeyFileName, outputSubRegionKeyDir);
  }

  static void splitSubRegionKeysbyRegionKeys(List<ByteBufWritableComparable> tblKeys, String inputSubRegionKeyFileName,
        String outputSplitKeyFileName, String outputSubRegionKeyDir)
        throws IOException {

    Configuration conf = new Configuration();
    Iterator<ByteBufWritableComparable> itTblKeys = tblKeys.iterator();
    //Current tablet start and end key
    ByteBufWritableComparable curTblStartKey = itTblKeys.next();
    ByteBufWritableComparable curTblEndKey = null;
    if (itTblKeys.hasNext()) {
      curTblEndKey = itTblKeys.next();
    }
    //The startkeys for the current tablet/region
    ArrayList<ByteBufWritableComparable> curTblSegKeys = new ArrayList<ByteBufWritableComparable>();
    //The adjusted tablet startkeys based on the segment keys.
    ArrayList<ByteBufWritableComparable> tblSplitKeys = new ArrayList<ByteBufWritableComparable>();

    //Since the keys may be large, we do not use readKeyRange to load the keys to save memory.
    Path partitionsPath = new Path(inputSubRegionKeyFileName);
    FileSystem fs = partitionsPath.getFileSystem(conf);

    @SuppressWarnings("deprecation")
    SequenceFile.Reader reader = new SequenceFile.Reader(fs, partitionsPath, conf);
    ByteBufWritableComparable iKey = new ByteBufWritableComparable();
    NullWritable value = NullWritable.get();
    LOG.debug("readSegKeyRange: " + partitionsPath.toString());

    if (!reader.next(iKey, value)) {
      reader.close();
      throw new IOException ("No  key in sub-region file " + inputSubRegionKeyFileName);
    }

    // First segment key should be empty
    ByteBuffer rKey1 = iKey.getByteBuf();
    ByteBuffer rKey2 = rKey1;
    if (iKey == null || !iKey.equals(ByteBufWritableComparable.EMPTY_BYTEBUFFER)) {
      reader.close();
      throw new IOException(
          "First segment of the table should have empty start key. Instead has: "
          + iKey);
    }
    curTblSegKeys.add(new ByteBufWritableComparable(rKey2));
    LOG.debug("\t0 (" + Bytes.toStringBinary(rKey2) + ")");

    //Rest of the keys should be ordered.
    int cnt = 1;
    while (reader.next(iKey, value)) {
      rKey1 = rKey2;
      rKey2 = iKey.getByteBuf();

      if (rKey1.compareTo(rKey2) >= 0) {
        // previous key should be smaller than the current key.
        reader.close();
        throw new IOException(
            "Current keyrange startky(" + Bytes.toStringBinary(rKey2)
            + ") should be larger than the previous keyrange startkey("
            + Bytes.toStringBinary(rKey1) + ")");
      }
      if ((curTblEndKey == null) || (rKey2.compareTo(curTblEndKey.getByteBuf()) < 0)) {
        //Current tablet is the last tablet or iKey is in current tablet
        curTblSegKeys.add(new ByteBufWritableComparable(rKey2));
        LOG.debug("\t" + Integer.toString(cnt) + " (" + Bytes.toStringBinary(rKey2) + ")");
      } else {
        //write current keys to the file with name as the startkey of this tablet.
        String tblSegKeysFileName = KEYPREFIX + curTblSegKeys.get(0);
        if (curTblSegKeys.get(0).equals(ByteBufWritableComparable.EMPTY_BYTEBUFFER)) {
          //First Tablet use key name as negative infinity.
          tblSegKeysFileName = FIRSTKEYNAME;
        }
        LOG.debug(Integer.toString(curTblSegKeys.size())+" segments to write.");
        writeKeyRange(conf, outputSubRegionKeyDir+"/"+tblSegKeysFileName, curTblSegKeys);

        //write down each region's first key into the split file.
        tblSplitKeys.add(curTblSegKeys.get(0));

        //clear the curTblSegKeys, and move to next tablet
        curTblSegKeys.clear();
        curTblStartKey = curTblEndKey;
        if (itTblKeys.hasNext()) {
          curTblEndKey = itTblKeys.next();
        } else {
          curTblEndKey = null;
        }

        curTblSegKeys.add(new ByteBufWritableComparable(rKey2));
        LOG.debug("\t" + Integer.toString(cnt) + " (" + Bytes.toStringBinary(rKey2) + ")");
      }
      ++cnt;
    }
    //Write the last one
    if (curTblSegKeys.size() > 0) {
      String tblSegKeysFileName = KEYPREFIX + curTblSegKeys.get(0);
      if (curTblSegKeys.get(0).equals(ByteBufWritableComparable.EMPTY_BYTEBUFFER)) {
        //First Tablet use key name as negative infinity.
        tblSegKeysFileName = FIRSTKEYNAME;
      }
      writeKeyRange(conf, outputSubRegionKeyDir+"/"+tblSegKeysFileName, curTblSegKeys);

      //write down last region's first key into the split file.
      tblSplitKeys.add(curTblSegKeys.get(0));
    }

    reader.close();
    writeKeyRange(conf, outputSplitKeyFileName, tblSplitKeys);
    LOG.debug("Total " + cnt + " segement startKey written into " + tblSplitKeys.size() + " tablet key range files.");
  }

  //If filePathName exists, overwrite it with msg.
  public static void writeStringToFile(Configuration conf, String msg, String filePathName)
      throws IOException {

    Path filePath = new Path(filePathName);
    FileSystem fs = filePath.getFileSystem(conf);
    BufferedWriter br = new BufferedWriter(new OutputStreamWriter(fs.create(filePath,true)));
    LOG.debug("Write "+msg+" to file "+filePathName);
    br.write(msg);
    br.close();
  }

  public static String[] diffTablesWithCrcArgToDiffTablesArg(
      final String diffTableOutdir,
      final String splitKRFile,
      final String includeKRFile,
      final String[] diffArgs) {

    if (diffArgs.length == 0 && splitKRFile == null && includeKRFile == null) {
          return new String[0];
    }
    if (includeKRFile != null && splitKRFile == null) {
      throw new IllegalArgumentException("Input key range file is missing.");
    }

    ArrayList<String> rangeArgs = new ArrayList<String>();
    for (int i = 0; i < diffArgs.length; ++i) {
      if (diffArgs[i].equalsIgnoreCase("-outdir")) {
        diffArgs[i+1] = diffTableOutdir;
        LOG.debug("diffTables -outdir =(" + diffArgs[i+1] + ")");
      }
      rangeArgs.add(diffArgs[i]);
    }
    if (splitKRFile != null) {
      rangeArgs.add("-split_keyrange");
      rangeArgs.add(splitKRFile);
    }
    if (includeKRFile != null) {
      rangeArgs.add("-keyrange_included");
      rangeArgs.add(includeKRFile);
    }
    rangeArgs.add("-mapreduce");
    rangeArgs.add("true");
    String[] retArr = new String[rangeArgs.size()];
    retArr = rangeArgs.toArray(retArr);
    LOG.info(rangeArgs.toString());
    return retArr;
  }

}
