package com.mapr.db.mapreduce.impl;

import java.net.URI;
import java.net.URISyntaxException;
import org.apache.hadoop.fs.Path;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ClusterTablePath {

  private static final Logger LOG = LoggerFactory.getLogger(ClusterTablePath.class);

  //check the CLUSTERPREFIX before CLUSTERPREFIX3, and check WRONGCLUSTERPREFIX after CLUSTERPREFIX3
  public static final String CLUSTERPREFIX = "maprfs:///mapr/";
  public static final String CLUSTERPREFIX2 = "/mapr/";
  public static final String CLUSTERPREFIX3 = "maprfs://";
  public static final String WRONGCLUSTERPREFIX = "maprfs:";  // begin with maprfs:, but it is neither CLUSTERPREFIX, nor CLUSTERPREFIX3

  public final String clusterName;
  public final String tablePathName;

  public ClusterTablePath(String cName, String tPath) {

    if (cName == null && tPath == null) {
      throw new java.lang.NullPointerException("cluster name and table path name cannot both be null.");
    }
    this.tablePathName = tPath;
    if (cName == null || cName.equals("")) {
      this.clusterName = null;
    } else {
      this.clusterName = cName;
    }
  }

  public String toString() {
      return getFullPathName();
  }

  public URI getClusterUri() {
    URI uri = null;
    try {
      if (clusterName == null) {
        //local table
        uri = new URI(CLUSTERPREFIX3 + "/");
      } else {
        uri = new URI(CLUSTERPREFIX3 + clusterName + "/");
      }
    } catch (URISyntaxException e) {
      LOG.error("Fail to generate cluster uri. Reason:");
      e.printStackTrace();
    }
    return uri;
  }

  public String getFullPathName() {
    if (this.clusterName  == null) {
      return this.tablePathName;
    } else {
      return (CLUSTERPREFIX3 + this.clusterName + this.tablePathName );
    }
  }

  public Path getFullPath() {
    return new Path(getFullPathName());
  }


  public static boolean equal(ClusterTablePath ctp1, ClusterTablePath ctp2) {

    boolean clusterIsSame = false;
    if (ctp1.clusterName == null && ctp2.clusterName == null) {
      clusterIsSame = true;
    } else if (ctp1.clusterName == null && ctp2.clusterName != null) {
      clusterIsSame = false;
    } else if (ctp1.clusterName != null && ctp2.clusterName == null) {
      clusterIsSame = false;
    } else if (ctp1.clusterName.equals(ctp2.clusterName)) {
      clusterIsSame = true;
    } else {
      clusterIsSame = false;
    }
    if (!clusterIsSame) {
      LOG.debug(ctp1 + "'s cluster is different from " + ctp2 + "'s cluster");
      return false;
    }
    boolean tablePathIsSame = false;
    if (ctp1.tablePathName == null && ctp2.tablePathName == null) {
      tablePathIsSame = true;
    } else if (ctp1.tablePathName == null && ctp2.tablePathName != null) {
      tablePathIsSame = false;
    } else if (ctp1.tablePathName != null && ctp2.tablePathName == null) {
      tablePathIsSame = false;
    } else if (ctp1.tablePathName.equals(ctp2.tablePathName)) {
      tablePathIsSame = true;
    } else {
      tablePathIsSame = false;
    }
    if (!tablePathIsSame) {
      LOG.debug(ctp1 + "'s tablePath is different from " + ctp2 + "'s tablePath");
      return false;
    }
    LOG.debug(ctp1 + " is same as " + ctp2);
    return true;
  }

  public boolean equal(ClusterTablePath ctp) {
    return equal(this, ctp);
  }

  public static ClusterTablePath parse(String tablePathName) {
    String cName = null;
    String tPath = null;
    String sub = null;
    //check the CLUSTERPREFIX before CLUSTERPREFIX3
    if (tablePathName.startsWith(CLUSTERPREFIX)) {
      sub = tablePathName.substring(CLUSTERPREFIX.length());
    } else if (tablePathName.startsWith(CLUSTERPREFIX2)) {
      sub = tablePathName.substring(CLUSTERPREFIX2.length());
    } else if (tablePathName.startsWith(CLUSTERPREFIX3)) {
      sub = tablePathName.substring(CLUSTERPREFIX3.length());
    } else if (tablePathName.startsWith(WRONGCLUSTERPREFIX)) {
      sub = tablePathName.substring(WRONGCLUSTERPREFIX.length());
      LOG.error("'/' is missing from input path(" + tablePathName+"), it should start with " + CLUSTERPREFIX2 + ", or " + CLUSTERPREFIX3);
      return null;
    } else {
      //local table
      return new ClusterTablePath(null, tablePathName);
    }
    //sub is the part of tablePathName where cluster prefix is removed, it can be empty string, but not null.
    if (sub.equals("")) {
      // the tablePathName only contains the cluster prefix, this is the root directory of local cluster
      return new ClusterTablePath(null, "/");
    }
    String[] splits = sub.split("/", 2);
    if (splits.length <= 0) {
      LOG.error("cluster name is missing from input path(" + tablePathName+"), it should start with " + CLUSTERPREFIX2 + "clustername/tablepathname");
      return null;
    } else if (splits.length <= 1) {
      LOG.error("tablepathname is missing from input path(" + tablePathName+"), to get the root directory of cluster "+splits[0]
              +", use "+CLUSTERPREFIX3 + splits[0] + "/");
      return null;
    } else {
      LOG.debug("cluster(" + splits[0] + "), table path(/" + splits[1]+")" );
      return new ClusterTablePath(splits[0], "/"+splits[1]);
    }
  }
}
