/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db.mapreduce.impl;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;
import java.nio.ByteBuffer;

import org.apache.hadoop.io.WritableComparable;

import com.mapr.db.util.ByteBufs;
import com.mapr.org.apache.hadoop.hbase.util.Bytes;

/**
 * A byte sequence that can be used as a key in JSON DB table. It implements WritableComparable
 * interface. When a new instance is created, underlying byte buffer is only referenced.
 *
 */

public class ByteBufWritableComparable implements WritableComparable<ByteBufWritableComparable> {

  public static ByteBufWritableComparable EMPTY_BYTEBUFFER
      = new ByteBufWritableComparable(ByteBuffer.wrap(new byte[0]));

  private ByteBuffer key;

  public ByteBufWritableComparable() {}

  public ByteBufWritableComparable(ByteBuffer buff) { key = buff;}

  public int getLength() {
    return key.limit();
  }

  public byte[] getBytes() {
    if (!key.hasArray()) {
      return null;
    }
    return key.array();
  }

  public ByteBuffer getByteBuf() {
    return key;
  }

  public void setByteBuf(ByteBuffer b) {
    key = b.slice();
  }

  /**
   * Deserialize the fields of this object from input stream <code>in</code>.
   * @param in Input stream which contains serialized object.
   * @see org.apache.hadoop.io.Writable#readFields(java.io.DataInput)
   */
  @Override
  public void readFields(DataInput in) throws IOException {
    int length = in.readInt();
    key = ByteBufs.allocatePreferred(length);
    in.readFully(key.array());
  }

  /**
   * Write serialized object to output stream <code>out</code>.
   * @param out Outputstream where serialized object is to be written.
   */
  @Override
  public void write(DataOutput out) throws IOException {
    out.writeInt(key.remaining());
    if (key.hasArray()) {
      out.write(key.array(), key.position(), key.limit());
    } else {
      byte[] data = new byte[key.remaining()];
      key.duplicate().get(data);
      out.write(data, 0, data.length);
    }
  }

  /**
   * Compares current object with <code>b</code>.
   * @param b Object to be compared with.
   * @return true/false
   * @see java.lang.Object#equals(java.lang.Object)
   */
  @Override
  public boolean equals(Object b) {
    if (b instanceof ByteBufWritableComparable) {
      return key.equals(((ByteBufWritableComparable) b).getByteBuf());
    }
    return false;
  }

  /**
   * compares current object with another object of type ByteBufWritableComparable.
   * @param o the object to be compared with current object.
   * @return a negative integer, zero, or a positive integer as this object is less
   *         than, equal to, or greater than the specified object.
   */
  @Override
  public int compareTo(ByteBufWritableComparable o) {
    return key.compareTo(o.getByteBuf());
  }

  /**
   * Returns the string version of current object.
   * @return string representation of current object.
   */
  @Override
  public String toString() {
    if (key == null) {
      return null;
    } else {
      ByteBuffer dup = key.duplicate();
      return Bytes.toStringBinary(dup);
    }
  }

}
