package com.mapr.db.mapreduce;

import java.io.DataInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;

import org.apache.hadoop.conf.Configured;
import org.apache.hadoop.io.serializer.Deserializer;
import org.apache.hadoop.io.serializer.Serialization;
import org.apache.hadoop.io.serializer.Serializer;
import org.ojai.Document;
import org.ojai.Value;

import com.mapr.db.impl.IdCodec;
import com.mapr.db.rowcol.RowcolCodec;
import com.mapr.db.util.ByteBufs;
import com.mapr.org.apache.hadoop.hbase.util.Bytes;

public class ValueSerialization extends Configured implements Serialization<Value> {

  @Override
  public boolean accept(Class<?> c) {
    return ( Value.class.isAssignableFrom(c) && (!Document.class.isAssignableFrom(c)));
  }

  @Override
  public Serializer<Value> getSerializer(Class<Value> c) {
    return new ValueSerializer();
  }

  @Override
  public Deserializer<Value> getDeserializer(Class<Value> c) {
    return new ValueDeserializer();
  }

  private static class ValueDeserializer implements Deserializer<Value> {

    DataInputStream d;

    @Override
    public void open(InputStream in) throws IOException {
      d = new DataInputStream(in);
    }

    @Override
    public Value deserialize(Value t) throws IOException {
      final int lenSize = (Integer.SIZE/Byte.SIZE);
      byte[] recSize = new byte[lenSize];
      d.read(recSize, 0, lenSize);
      int recordSize = Bytes.toInt(recSize);
      if (recordSize < 0) {
        throw new IllegalArgumentException("Invalid record size " + recordSize);
      }

      /* extract record  and deserialize */
      byte[] b = new byte[recordSize];
      d.read(b, 0, recordSize);
      ByteBuffer buffer = ByteBufs.wrap(b);

      return IdCodec.decode(buffer);
    }

    @Override
    public void close() throws IOException {
      d.close();
    }
  }

  private static class ValueSerializer implements Serializer<Value> {

    OutputStream out;

    @Override
    public void open(OutputStream out) throws IOException {
      this.out = out;
    }

    @Override
    public void serialize(Value t) throws IOException {
      ByteBuffer b = IdCodec.encode(t);
      out.write(Bytes.toBytes(b.remaining()));
      int remaining;
      final int maxRead = 128;
      byte[] buffer = new byte[maxRead];
      while((remaining = Math.min(b.remaining(), maxRead)) > 0) {
        b.get(buffer, 0, remaining);
        out.write(buffer, 0, remaining);
      }
    }

    @Override
    public void close() throws IOException {
      out.close();
    }
  }

}
