/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db.mapreduce;

import java.io.IOException;

import org.apache.hadoop.mapreduce.RecordWriter;
import org.apache.hadoop.mapreduce.TaskAttemptContext;
import org.ojai.Document;
import org.ojai.Value;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.mapr.db.Table;
import com.mapr.db.impl.MapRDBTableImpl;
import com.mapr.db.impl.MapRDBTableImpl.TablePrivateOption;

/**
 * Receives a document from Map/Reduce phase and writes to a JSON table.
 * @param table MapR-DB table where the document is to be written.
 */
public class TableRecordWriter extends RecordWriter<Value, Document>
   {

  static final Logger LOG = LoggerFactory.getLogger(TableRecordWriter.class);

  private Table jTable;

  /** Job parameter that specifies the output table. */
  public static final String OUTPUT_TABLE = "maprdb.mapreduce.outputtable";


  public TableRecordWriter(Table table) {
    jTable = table;
    ((MapRDBTableImpl)jTable).setPrivateOption(TablePrivateOption.PRESERVE_TIMESTAMP, true);
  }

  /**
   * Closes the writer, in this case flush table commits.
   *
   * @param context  The context.
   * @throws IOException When closing the writer fails.
   * @see org.apache.hadoop.mapreduce.RecordWriter#close(org.apache.hadoop.mapreduce.TaskAttemptContext)
   */
  @Override
  public void close(TaskAttemptContext context)
  throws IOException {
    jTable.flush();
  }

  /**
   * Writes a key/value pair into the table.
   *
   * @param key  The key.
   * @param value  The value.
   * @throws IOException When writing fails.
   * @see org.apache.hadoop.mapreduce.RecordWriter#write(java.lang.Object, java.lang.Object)
   */
  @Override
  public void write(Value key, Document value)
  throws IOException {
    if (key == null) {
      jTable.insertOrReplace(value);
    } else {
      jTable.insertOrReplace(key, value);
    }
  }

}
