/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db.mapreduce;

import java.io.IOException;
import java.util.Iterator;

import org.apache.hadoop.mapreduce.InputSplit;
import org.apache.hadoop.mapreduce.TaskAttemptContext;
import org.ojai.Document;
import org.ojai.DocumentStream;
import org.ojai.Value;
import org.ojai.store.QueryCondition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.mapr.db.Table;

/**
 * Iterates over MapR-DB JSON table data. Returns key, value pair as
 * ByteBufWritableComparable and Document respectively.
 */
public class TableRecordReaderImpl {

  static final Logger LOG = LoggerFactory.getLogger(TableRecordReaderImpl.class);

  private Value key = null;
  private Document value = null;
  private Table jTable = null;
  private TaskAttemptContext context = null;
  DocumentStream rs = null;
  Iterator<Document> recordIterator = null;
  private QueryCondition cond = null;

  /* The list of paths for projection */
  private String[] fieldPaths = null;

  /**
   * Close current split.
   */
  public void close() {
    try {
      jTable.close();
    } catch (Exception e) {
      LOG.warn("Error closing table "+e.getMessage());
    }
  }

  /**
   * Returns current key.
   * @return current key.
   * @throws IOException
   * @throws InterruptedException
   */
  public Value getCurrentKey() throws IOException, InterruptedException {
    return key;
  }

  /**
   * Returns current value as Document.
   * @return current document.
   * @throws IOException
   * @throws InterruptedException
   */
  public Document getCurrentValue() throws IOException, InterruptedException {
    return value;
  }

  /**
   * Positions reader/iterator to the next row.
   * @return <code>true</code> if there is a row.
   * @throws IOException
   * @throws InterruptedException
   */
  public boolean nextKeyValue() throws IOException, InterruptedException {

    key = null;
    value = null;
    if (recordIterator.hasNext()) {
      value = recordIterator.next();
      key = value.getId();
      return true;
    }
    return false;
  }

  /**
   * Initialize reader/iterator.
   * @param inputsplit split on which the iterator is to be initialized.
   * @param cntxt current task context.
   * @throws IOException
   * @throws InterruptedException
   */
  public void initialize(InputSplit inputsplit,
      TaskAttemptContext cntxt) throws IOException,
      InterruptedException {

    if (cntxt != null) {
      this.context = cntxt;
    }

    if (cond != null) {
      if (fieldPaths != null) {
        rs = jTable.find(cond, fieldPaths);
      } else {
        rs = jTable.find(cond);
      }
    } else {
      if (fieldPaths != null) {
        rs = jTable.find(fieldPaths);
      } else {
        rs = jTable.find();
      }
    }

    if (rs != null) {

     recordIterator = rs.iterator();
    }
  }

  /**
   * Sets table name.
   * @param t input table.
   */
  public void setTable(Table t) {
    this.jTable = t;
  }

  public void setCondition(QueryCondition c) {
    cond = c;
  }

  /* This function is used to set field paths to be used for projection.
   * For simplicity, we assume for now that ',' is the separator and there are
   * no commas in individual field name strings.
   */
  public void setFieldPath(String fieldPath) {
    //fieldPaths = TableMapReduceUtil.getFieldPaths(fieldPath);
    fieldPaths = fieldPath.split(",");
  }


}

