/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db.mapreduce;

import java.io.IOException;

import org.apache.hadoop.mapreduce.InputSplit;
import org.apache.hadoop.mapreduce.RecordReader;
import org.apache.hadoop.mapreduce.TaskAttemptContext;
import org.ojai.Document;
import org.ojai.Value;
import org.ojai.store.QueryCondition;

import com.mapr.db.Table;
import com.mapr.db.mapreduce.impl.ByteBufWritableComparable;

/**
 * Top level wrapper class which iterates over rows in a JSON table and returns (Key, Value) pairs,
 * where value is a document.
 * @param inputSplit split which will be scanned by this iterator.
 * @param context current job context.
 */
public class TableRecordReader extends RecordReader<Value, Document> {

  private TableRecordReaderImpl recordReaderImpl = new TableRecordReaderImpl();
  /**
   * Closes the split.
   *
   * @see org.apache.hadoop.mapreduce.RecordReader#close()
   */
  @Override
  public void close() {
    recordReaderImpl.close();
  }

  /**
   * Returns the current key.
   *
   * @return The current key.
   * @throws IOException
   * @throws InterruptedException When the job is aborted.
   * @see org.apache.hadoop.mapreduce.RecordReader#getCurrentKey()
   */
  @Override
  public Value getCurrentKey() throws IOException,
      InterruptedException {
    return this.recordReaderImpl.getCurrentKey();
  }

  /**
   * Returns the current value.
   *
   * @return The current value.
   * @throws IOException When the value is faulty.
   * @throws InterruptedException When the job is aborted.
   * @see org.apache.hadoop.mapreduce.RecordReader#getCurrentValue()
   */
  @Override
  public Document getCurrentValue() throws IOException, InterruptedException {
    return this.recordReaderImpl.getCurrentValue();
  }

  /**
   * Initializes the reader.
   *
   * @param inputsplit  The split to work with.
   * @param context  The current task context.
   * @throws IOException When setting up the reader fails.
   * @throws InterruptedException When the job is aborted.
   * @see org.apache.hadoop.mapreduce.RecordReader#initialize(
   *   org.apache.hadoop.mapreduce.InputSplit,
   *   org.apache.hadoop.mapreduce.TaskAttemptContext)
   */
  @Override
  public void initialize(InputSplit inputsplit,
      TaskAttemptContext context) throws IOException,
      InterruptedException {
    this.recordReaderImpl.initialize(inputsplit, context);
  }

  /**
   * Positions the record reader to the next record.
   *
   * @return <code>true</code> if there was another record.
   * @throws IOException When reading the record failed.
   * @throws InterruptedException When the job was aborted.
   * @see org.apache.hadoop.mapreduce.RecordReader#nextKeyValue()
   */
  @Override
  public boolean nextKeyValue() throws IOException, InterruptedException {
    return this.recordReaderImpl.nextKeyValue();
  }

  /**
   * The current progress of the record reader through its data.
   *
   * @return A number between 0.0 and 1.0, the fraction of the data read.
   * @see org.apache.hadoop.mapreduce.RecordReader#getProgress()
   */
  @Override
  public float getProgress() {
    return 0;
  }

  public void setTable(Table jTable) {
    this.recordReaderImpl.setTable(jTable);
  }

  public void setCondition(QueryCondition c) {
    this.recordReaderImpl.setCondition(c);
  }

  public void setFieldPath(String fieldPath) {
    this.recordReaderImpl.setFieldPath(fieldPath);
  }


}
