package com.mapr.db.mapreduce;

import java.io.IOException;

import org.apache.hadoop.conf.Configurable;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.mapreduce.JobContext;
import org.apache.hadoop.mapreduce.OutputCommitter;
import org.apache.hadoop.mapreduce.OutputFormat;
import org.apache.hadoop.mapreduce.RecordWriter;
import org.apache.hadoop.mapreduce.TaskAttemptContext;
import org.ojai.Value;
import org.ojai.store.DocumentMutation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.mapr.db.Table;
import com.mapr.db.Table.TableOption;
import com.mapr.db.impl.MapRDBTableImpl;
import com.mapr.db.mapreduce.impl.TableOutputCommitter;

/**
 * Writes DocumentMutation from Map/Reduce phase to JSON tables. The key is of type Value and the
 * Value is a DocumentMutation.
 */

public class TableMutationOutputFormat extends OutputFormat<Value, DocumentMutation> implements Configurable {

  private static final Logger LOG = LoggerFactory.getLogger(TableMutationOutputFormat.class);

  /** Job parameter that specifies the output table. */
  public static final String OUTPUT_TABLE = "maprdb.mapred.mutation.outputtable";

  private Table jTable = null;

  /** The configuration. */
  private Configuration conf = null;
  private String tableName = null;


  /**
   * This function is used to set parameters in the configuration. This is used to
   * set relevant parameters for scanning JSON tables.
   *
   * @param arg0 Configuration object with parameters for TableOutputFormat.
   *  @see org.apache.hadoop.conf.Configurable#setConf(
   *  org.apache.hadoop.conf.Configuration)
   */
  @Override
  public void setConf(Configuration arg0) {
    conf = arg0;
    tableName = conf.get(OUTPUT_TABLE);

    if ((tableName == null) || (tableName.length() <= 0)) {
      throw new IllegalArgumentException("Must specify output table name");
    }

    jTable = new MapRDBTableImpl(new Path(tableName), conf);
    jTable.setOption(TableOption.BUFFERWRITE, true);
  }

  /**
   * Returns the current configuration.
   *
   * @return : Current Configuration.
   * @see org.apache.hadoop.conf.Configurable#getConf()
   */
  @Override
  public Configuration getConf() {
    return conf;
  }

  /**
   * Creates a new DocumentMutation record writer.
   *
   * @param context  The current task context.
   * @return The newly created writer instance.
   * @throws IOException When creating the writer fails.
   * @throws InterruptedException When the jobs is cancelled.
   */
  @Override
  public RecordWriter<Value, DocumentMutation> getRecordWriter(TaskAttemptContext context)
      throws IOException, InterruptedException {
    return new TableMutationRecordWriter(jTable);
  }

  /**
   * Checks if the output target exists.
   *
   * @param context  The current context.
   * @throws IOException When the check fails.
   * @throws InterruptedException When the job is aborted.
   * @see org.apache.hadoop.mapreduce.OutputFormat#checkOutputSpecs(org.apache.hadoop.mapreduce.JobContext)
   */
  @Override
  public void checkOutputSpecs(JobContext context) throws IOException, InterruptedException {
    if (jTable == null) {
      throw new IOException("Destination table "
                             + tableName + " does not exist");
    }
  }

  /**
   * Returns the output committer.
   *
   * @param context  The current context.
   * @return The committer.
   * @throws IOException When creating the committer fails.
   * @throws InterruptedException When the job is aborted.
   * @see org.apache.hadoop.mapreduce.OutputFormat#getOutputCommitter(org.apache.hadoop.mapreduce.TaskAttemptContext)
   */
  @Override
  public OutputCommitter getOutputCommitter(TaskAttemptContext context) throws IOException,
      InterruptedException {
    return new TableOutputCommitter();
  }

}
