package com.mapr.db.mapreduce;

import java.io.DataInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;

import org.apache.hadoop.conf.Configured;
import org.apache.hadoop.io.serializer.Deserializer;
import org.apache.hadoop.io.serializer.Serialization;
import org.apache.hadoop.io.serializer.Serializer;
import org.ojai.Document;

import com.mapr.db.rowcol.SequenceFileRowColCodec;
import com.mapr.db.util.ByteBufs;
import com.mapr.org.apache.hadoop.hbase.util.Bytes;

/**
 * Converts a JSON document from MapR-DB format to binary SequenceFile format.
 *
 */

public class DBDocumentSerialization extends Configured implements Serialization<Document> {

  /**
   * Allows clients to test whether this serialization supports the given class.
   */
  @Override
  public boolean accept(Class<?> arg0) {
    return Document.class.isAssignableFrom(arg0);
  }

  /**
   * Returns a deserializer of the document.
   * @see org.apache.hadoop.io.serializer.Serialization#getDeserializer(java.lang.Class)
   */
  @Override
  public Deserializer<Document> getDeserializer(Class<Document> arg0) {
    return new DocumentDeserializer();
  }

  /**
   * Returns a serializer of document.
   * @see org.apache.hadoop.io.serializer.Serialization#getSerializer(java.lang.Class)
   */
  @Override
  public Serializer<Document> getSerializer(Class<Document> arg0) {
    return new DocumentSerializer();
  }

  /**
   * Implements the Deserializer of MapR-DB document from binary sequence file format.
   * @param d DataInputStream which contains serialized document.
   */
  private static class DocumentDeserializer implements Deserializer<Document> {

    DataInputStream d;

    /**
     * @see org.apache.hadoop.io.serializer.Deserializer#close()
     */
    @Override
    public void close() throws IOException {
      d.close();
    }

    /**
     * Deserialize document from input stream
     * @return Deserialized document from input stream.
     */
    @Override
    public Document deserialize(Document record) throws IOException {
      /* extract serialized length of record*/
      final int lenSize = (Integer.SIZE/Byte.SIZE);
      byte[] recSize = new byte[lenSize];
      d.read(recSize, 0, lenSize);
      int recordSize = Bytes.toInt(recSize);
      if (recordSize < 0) {
        throw new IllegalArgumentException("Invalid record size " + recordSize);
      }

      /* extract record  and deserialize */
      byte[] b = new byte[recordSize];
      d.read(b, 0, recordSize);
      ByteBuffer buffer = ByteBufs.wrap(b);

      return SequenceFileRowColCodec.decode(buffer);
    }

    /**
     * @see org.apache.hadoop.io.serializer.Deserializer#open(java.io.InputStream)
     */
    @Override
    public void open(InputStream in) throws IOException {
      d = new DataInputStream(in);

    }

  }

  /**
   * Implements the Serializer of MapR-DB document into binary sequence file format.
   */
  private static class DocumentSerializer implements Serializer<Document> {

    OutputStream out;

    /**
     * @see org.apache.hadoop.io.serializer.Serializer#close()
     */
    @Override
    public void close() throws IOException {
      out.close();
    }

    /**
     * @see org.apache.hadoop.io.serializer.Serializer#open(java.io.OutputStream)
     */
    @Override
    public void open(OutputStream out) throws IOException {
      this.out = out;
    }

    /**
     * Serialize MapR-DB document into binary sequence file format and write to the
     * output stream.
     * @param record Document to be serialized.
     */
    @Override
    public void serialize(Document record) throws IOException {
      ByteBuffer outBuffer;
      outBuffer = SequenceFileRowColCodec.encode(record);

      /* length of encoded bytebuffer */
      out.write(Bytes.toBytes(outBuffer.remaining()));
      int remaining;
      final int maxRead = 128;
      byte[] buffer = new byte[maxRead];
      while((remaining = Math.min(outBuffer.remaining(), maxRead)) > 0) {
        outBuffer.get(buffer, 0, remaining);
        out.write(buffer, 0, remaining);
      }
    }

  }

}
