/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.db.mapreduce;


import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.mapreduce.RecordWriter;
import org.apache.hadoop.mapreduce.TaskAttemptContext;
import org.ojai.Document;
import org.ojai.FieldPath;
import org.ojai.Value;
import org.ojai.store.DocumentMutation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.mapr.db.exceptions.TableNotFoundException;
import com.mapr.db.impl.IdCodec;
import com.mapr.db.impl.MapRDBTableImpl;
import com.mapr.db.mapreduce.impl.ByteBufWritableComparable;
import com.mapr.db.rowcol.KeyValue;
import com.mapr.db.rowcol.IdValueComparator;
import com.mapr.db.rowcol.MutationImpl;
import com.mapr.db.rowcol.RowcolCodec;
import com.mapr.db.rowcol.SerializedFamilyInfo;
import com.mapr.fs.jni.MapRJSONPut;
import com.mapr.org.apache.hadoop.hbase.util.Bytes;

/**
 * Receives a document from Map/Reduce phase and writes to a JSON table using bulkload mode.
 *
 * @param tablePath path to the table.
 * @param conf job configuration.
 */

public class BulkLoadRecordWriter extends RecordWriter<Value, Document> {
  private static final Logger LOG = LoggerFactory.getLogger(BulkLoadRecordWriter.class);
  private final MapRDBTableImpl maprDbTable;
  private long bulkLoaderId;
  private MapRJSONPut prevRecord;
  private int curNumFamilies;
  private Value prevKey;
  private Value curKey;
  private Document curRecord;
  private ByteBuffer curRecordEncoded;

  private List<MapRJSONPut> bufferedPuts;
  private long totalBufferedLen;
  private byte[] tableUuid;
  private Map<FieldPath, Integer> cfIdPathMap = null;
  private SerializedFamilyInfo[] famInfo;
  private IdValueComparator valueComparator;

  public BulkLoadRecordWriter(Configuration conf, Path tablePath)
    throws IOException,TableNotFoundException {

    //Init the table
    maprDbTable = new MapRDBTableImpl(tablePath, conf);

    String uuid = conf.get("maprdb.bulkload.uuid");
    if (uuid != null) {
      tableUuid = Bytes.toBytesBinary(uuid);
    } else {
      tableUuid = null;
    }
    cfIdPathMap = maprDbTable.idPathMap();
    assert cfIdPathMap != null;

    totalBufferedLen = 0;
    bufferedPuts = new ArrayList<MapRJSONPut>();
    curNumFamilies = 0;
    curKey = null;
    prevKey = null;
    curRecord = null;
    curRecordEncoded = null;
    prevRecord = null;
    famInfo = null;
    valueComparator = new IdValueComparator();
  }

  private void open() throws
    IOException {
    if (this.bulkLoaderId == 0) {
      this.bulkLoaderId = maprDbTable.maprTable().getBulkLoader(this.tableUuid);
    }
  }

  /**
   * Writes a record mutation to the table.
   * @param rowBuf rowkey in the table.
   * @param mutation mutation to be updated in the table.
   */
  public void write(Value idValue, DocumentMutation mutation)
    throws IOException, InterruptedException {
    open();

    if (idValue == null && mutation == null) {
      flush();
      return;
    }

    MutationImpl rmi = ((MutationImpl)mutation);

    //Check for unsorted or duplicate keys
    if (prevKey != null) {
      int cmp = valueComparator.compareTo(prevKey, idValue);
      if (cmp == 0) {
        LOG.warn("Ignoring duplicate key-record");
        return;
      } else if (cmp > 0) {
        throw new IOException("Received unsorted key-value.");
      }
    }

    if (rmi.needsReadOnServer()) {
      throw new IOException("BulkLoadRecordWriter not supported for " +
                            "RecordMutation that needs read on server.");
    }

    if (curKey != null && (valueComparator.compareTo(curKey, idValue) != 0)) {
      potentiallyFlush();
    }

    if (curKey == null) {
      curKey = idValue;
    }

    //Serialize the mutation
    assert mutation != null;
    famInfo = rmi.rowcolSerialize(cfIdPathMap, true /*isBulkLoad*/);
    assert famInfo != null;

    MapRJSONPut mput = new MapRJSONPut(IdCodec.encodeAsBytes(curKey), famInfo.length);
    prevKey = curKey;

    for (SerializedFamilyInfo i : famInfo) {
      switch(i.getAction()) {
        case NO_ACTION:
          continue;

        case DELETE_FAMILY:
          mput.addCFEntry(i.getFamilyId(), (ByteBuffer)null);
          break;

        case SET:
          mput.addCFEntry(i.getFamilyId(), i.getByteBuffer());
          break;

        default:
          assert "Invalid Serialization access" == null;
      }
    }
    totalBufferedLen = mput.recordTotalBytes;
    bufferedPuts.add(mput);
  }

  /**
   * Writes a document to the table.
   * @param rowBuf rowkey to be written to the table.
   * @param record document to be written.
   */
  @Override
  public void write(Value rowKey, Document record)
    throws IOException, InterruptedException {
    open();

    // null input == user explicitly wants to flush
    if (rowKey == null && record == null) {
      flush();
      return;
    }


    //Check for unsorted or duplicate keys
    if (prevKey != null) {
      int cmp = valueComparator.compareTo(prevKey, rowKey);
      if (cmp == 0) {
        LOG.warn("Ignoring duplicate key-record");
        return;
      } else if (cmp > 0) {
        throw new IOException("Received unsorted key-value: prev " + prevKey.toString() + " cur " + rowKey.toString());
      }
    }

    if (curKey != null && (valueComparator.compareTo(curKey, rowKey) != 0)) {
      potentiallyFlush();
    }

    if (curKey == null) {
      curKey = rowKey;
    }
    assert record != null;

    //Serialize the record
    // We should use the stored timestamp if record was created with flag to store the ts
    famInfo = RowcolCodec.encode(record, cfIdPathMap, true /*isBulkLoad*/, true /*useEncoded*/);
    assert famInfo != null;

    MapRJSONPut mput = new MapRJSONPut(IdCodec.encodeAsBytes(curKey), famInfo.length);
    prevKey = curKey;

    for (SerializedFamilyInfo i : famInfo) {
      switch(i.getAction()) {
        case NO_ACTION:
          continue;

        case DELETE_FAMILY:
          mput.addCFEntry(i.getFamilyId(), (ByteBuffer)null);
          break;

        case SET:
          mput.addCFEntry(i.getFamilyId(), i.getByteBuffer());
          break;

        default:
          assert "Invalid Serialization access" == null;
      }
    }
    totalBufferedLen = mput.recordTotalBytes;
    bufferedPuts.add(mput);
  }

  private void potentiallyFlush() throws IOException {
      if (bufferedPuts.size() >= 100 ||
        totalBufferedLen >= 1024 * 1024) {
        flush();
      }
      curKey = null;
      curRecord = null;
      curRecordEncoded = null;
      prevRecord = null;
  }

  private void flush() throws IOException {
    if (bulkLoaderId == 0 || bufferedPuts.size() == 0) {
      return;
    }

    if (LOG.isDebugEnabled()) {
      LOG.debug("flushing " + bufferedPuts.size() + " records");
    }

    MapRJSONPut[] mputs = bufferedPuts.toArray(new MapRJSONPut[bufferedPuts.size()]);
    maprDbTable.maprTable().bulkLoaderAppendEncoded(bulkLoaderId, mputs);
    bufferedPuts.clear();
    totalBufferedLen = 0;
  }

  /*
   *
   */
 @Override
public void close(TaskAttemptContext c)
  throws IOException, InterruptedException {
  if (bulkLoaderId == 0) {
    return;
  }

  flush();
  maprDbTable.maprTable().bulkLoaderClose(bulkLoaderId);
  bulkLoaderId = 0;
 }
}
