/* Copyright (c) 2015 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.db.mapreduce;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.List;

import org.apache.hadoop.conf.Configurable;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.mapreduce.JobContext;
import org.apache.hadoop.mapreduce.OutputCommitter;
import org.apache.hadoop.mapreduce.OutputFormat;
import org.apache.hadoop.mapreduce.RecordWriter;
import org.apache.hadoop.mapreduce.TaskAttemptContext;
import org.ojai.Document;
import org.ojai.Value;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.mapr.db.Admin;
import com.mapr.db.MapRDB;
import com.mapr.db.Table;
import com.mapr.db.TabletInfo;
import com.mapr.db.exceptions.TableNotFoundException;
import com.mapr.db.impl.ConditionImpl;
import com.mapr.db.impl.ConditionNode.RowkeyRange;
import com.mapr.db.mapreduce.impl.TableOutputCommitter;

/**
 * This class defines the OutputFormat to be used when Ojai documents have to be imported
 * into MapR Document-DB using Bulkload mode.
 * @param OUTPUT_TABLE Configuration parameter for providing output table.
 * @param tableName Name of the output table.
 *
 */
public class BulkLoadOutputFormat extends OutputFormat<Value, Document>
    implements Configurable {
  private static final Logger LOG = LoggerFactory.getLogger(BulkLoadOutputFormat.class);
  public static final String OUTPUT_TABLE = "maprdb.mapreduce.bulkloadrecordwriter.outputTable";
  private static Table table = null;
  private static String tableName = null;
  private Configuration conf = null;

  /**
   * Sets parameters in current configuration.
   * @param conf Current configuration.
   * @see org.apache.hadoop.conf.Configurable#setConf(org.apache.hadoop.conf.Configuration)
   */
  @Override
  public void setConf(Configuration conf) {
    this.conf = conf;
    if (tableName == null) {
      tableName = conf.get(OUTPUT_TABLE);
    } else {
      assert (tableName == conf.get(OUTPUT_TABLE));
    }
    if ((tableName == null) || (tableName.length() == 0)) {
      throw new IllegalArgumentException("Complete Output Table path must be specified");
    }
  }

  /**
   * Returns current configuration object.
   * @return current configuration.
   * @see org.apache.hadoop.conf.Configurable#getConf()
   */
  @Override
  public Configuration getConf() {
    return conf;
  }

  /**
   * Returns a bulkload RecordWriter object which can write a row to a table. The record
   * writer works with key type as ByteBufWritableComparable and value type as a Document.
   * @param context current job context.
   * @return RecordWriter object.
   */
  @Override
  public RecordWriter<Value, Document> getRecordWriter(
      TaskAttemptContext context) throws IOException, InterruptedException {
    return createRecordWriter(context);
  }

  private RecordWriter<Value, Document> createRecordWriter(
      TaskAttemptContext context) throws IOException, InterruptedException {
    table = getTable(tableName);
    Path tablePath = new Path(tableName);
    return new BulkLoadRecordWriter(conf, tablePath);
  }

  @Override
  public void checkOutputSpecs(JobContext context) throws IOException,
      InterruptedException {

  }

  /**
   * Return the outputCommitter object.
   * @param context current job context.
   * @return OutputCommitter .
   * @see org.apache.hadoop.mapreduce.OutputFormat#getOutputCommitter(org.apache.hadoop.mapreduce.TaskAttemptContext)
   */
  @Override
  public OutputCommitter getOutputCommitter(TaskAttemptContext context)
      throws IOException, InterruptedException {
    return new TableOutputCommitter();
  }

  private static Table getTable(String tabName)
    throws IOException {
    tableName = tabName;
    try (Admin admin = MapRDB.newAdmin()) {
      if (!admin.tableExists(tableName)) {
        table = admin.createTable(
            MapRDB.newTableDescriptor(tabName)
              .setBulkLoad(true));
      }
      table = MapRDB.getTable(tableName);
    } catch (TableNotFoundException e) {
      LOG.info("Table not found. Creating table " + tableName + " with bulkload=true.");
    }
    return table;
  }

  /**
   * Returns a list of keys which indicate the starting of each input partition.
   * @param tabName output table Name.
   * @return List of partitioning split points for TotalOrderPartitioner
   * @throws IOException
   */
  public static List<ByteBuffer> getPartitionSplitPoints(String tabName)
      throws IOException {
    TabletInfo[] tabletInfo = getTable(tabName).getTabletInfos();
    assert tabletInfo.length >= 1;

    List<ByteBuffer> startKeys = new ArrayList<ByteBuffer>();
    for (TabletInfo ti : tabletInfo) {
      List<RowkeyRange> range = ((ConditionImpl)ti.getCondition()).getRowkeyRanges();
      startKeys.add(ByteBuffer.wrap(range.get(0).getStartRow()));
    }
    return startKeys;
  }

}
