#!/bin/bash
# Copyright (c) 2018 & onwards. MapR Tech, Inc., All rights reserved
#
# This script configures various settings for MapR scripts
#
this="${BASH_SOURCE-$0}"
while [ -h "$this" ]; do
  ls=`ls -ld "$this"`
  link=`expr "$ls" : '.*-> \(.*\)$'`
  if expr "$link" : '.*/.*' > /dev/null; then
    this="$link"
  else
    this=`dirname "$this"`/"$link"
  fi
done

# Convert relative path to absolute path
bin=`dirname "$this"`
script=`basename "$this"`
home=`cd "$bin/..">/dev/null; pwd`
this="$home/bin/$script"

# Test for *nix flavor
OS_FLAVOR=""
is_cygwin=false
case "`uname -s`" in
  CYGWIN*) is_cygwin=true; OS_FLAVOR="windows";;
  Darwin*) OS_FLAVOR="mac";;
  Linux*)
    case "`lsb_release -si`" in
      RedHat*|CentOS*) OS_FLAVOR="rhel";;
      Ubuntu*) OS_FLAVOR="ubuntu";;
    esac
    ;;
esac

# Prints files matching the specified pattern from the
# specified folder separated by PATH separator ':'
#
# $1:       Folder to include files from
# $2[...]:  Files (wildcards allowed) from the folder.
#           If you use wildcards, make sure to "quote" them.
get_files_in_folder() {
  FOLDER=$1
  shift

  PATH_FILES=
  until [ -z "$1" ]; do
    FOUND_FILES=`echo $(ls ${FOLDER}/$1 2> /dev/null) | sed -E "s/[[:space:]]+/:/g"`
    if [ "${FOUND_FILES}" != "" ]; then
      PATH_FILES=${PATH_FILES}:${FOUND_FILES}
    fi
    shift
  done
  # Remove any additional ':' from the tail
  PATH_FILES="${PATH_FILES#:}"
  echo $PATH_FILES
}

#MapR DAG core jars
get_mapr_dag_jars() {
  get_files_in_folder $MAPR_DAG_JARS \
    "*.jar"
}

#MapR DAG bootclasspath jars
get_mapr_dag_boot_jars() {
  get_files_in_folder $MAPR_DAG_JARS/boot \
    "*.jar"
}

# Root of the MapR installation
MAPR_HOME=${MAPR_HOME:=/opt/mapr}

# Various MapR folders
MAPR_DAG_HOME="$home" # determined from the location of this script
MAPR_DAG_BIN="$MAPR_DAG_HOME/bin"
MAPR_DAG_CONF="$MAPR_DAG_HOME/conf"
MAPR_DAG_JARS="$MAPR_DAG_HOME/jars"
MAPR_DAG_LIB="$MAPR_DAG_HOME/lib"
MAPR_DAG_LOGS="$MAPR_DAG_HOME/logs"
MAPR_DAG_CP="$MAPR_DAG_CONF"
MAPR_DAG_CP=${MAPR_DAG_CP}:$(get_mapr_dag_jars)
MAPR_DAG_LOGIN_CONF="$MAPR_HOME/conf/mapr.login.conf"

# Create necessary folders
mkdir -p ${MAPR_DAG_LOGS}

# Translate CygWin paths
if $is_cygwin ; then
  # files
  MAPR_DAG_HOME=`cygpath -w "$MAPR_DAG_HOME"`
  MAPR_DAG_BIN=`cygpath -w "$MAPR_DAG_BIN"`
  MAPR_DAG_CONF=`cygpath -w "$MAPR_DAG_CONF"`
  MAPR_DAG_LIB=`cygpath -w "$MAPR_DAG_LIB"`
  MAPR_DAG_LOGS=`cygpath -w "$MAPR_DAG_LOGS"`
  MAPR_DAG_LOGIN_CONF=`cygpath -w "$MAPR_DAG_LOGIN_CONF"`
  # paths
  MAPR_DAG_CP=`cygpath -w -p "$MAPR_DAG_CP"`
fi

# Set the JRE Options for the DAG process
MAPR_DAG_OPTS="${MAPR_DAG_OPTS} -Xmx512m -Xms256m"

# SSL Options
MAPR_DAG_OPTS="${MAPR_DAG_OPTS} -Ddata-access-gateway.grpc.service.ssl.enabled=cluster" # cluster|true|false

# Java Security Auth Config
MAPR_DAG_OPTS="${MAPR_DAG_OPTS} -Djava.security.auth.login.config=${MAPR_DAG_LOGIN_CONF}"

# Java library path
if [ -n "$OS_FLAVOR" ] ; then
  MAPR_DAG_OPTS="${MAPR_DAG_OPTS} -Djava.library.path=${MAPR_DAG_LIB}/${OS_FLAVOR}"
elif [ -d ${MAPR_HOME}/lib ] ; then
  MAPR_DAG_OPTS="${MAPR_DAG_OPTS} -Djava.library.path=${MAPR_HOME}/lib"
fi

if [ -n "$DEBUG_DAG" ] ; then
  MAPR_DAG_OPTS="${MAPR_DAG_OPTS} -agentlib:jdwp=transport=dt_socket,server=y,address=5677,suspend=y"
fi

if [ -n "$DEBUG_DAG_SSL" ] ; then
  MAPR_DAG_OPTS="${MAPR_DAG_OPTS} -Djavax.net.debug=ssl:handshake:verbose:keymanager:trustmanager -Djava.security.debug=access:stack"
fi

# Logger options
if [ -n "$DEBUG_DAG_LOGS" ] ; then
  MAPR_DAG_OPTS="${MAPR_DAG_OPTS} -Droot.log.level=debug -Ddrill.log.level=debug -Dhadoop.log.level=debug"
else
  MAPR_DAG_OPTS="${MAPR_DAG_OPTS} -Droot.log.level=info -Ddrill.log.level=error -Dhadoop.log.level=error"
fi

# Export the necessary variables to the caller
export is_cygwin
export MAPR_DAG_HOME
export MAPR_DAG_BIN
export MAPR_DAG_CONF
export MAPR_DAG_JARS
export MAPR_DAG_LIB
export MAPR_DAG_LOGS
export MAPR_DAG_CP
export MAPR_DAG_OPTS
export MAPR_IMPERSONATION_ENABLED=1
