package com.mapr.security.maprsasl;

import java.io.File;
import java.io.IOException;
import java.security.Principal;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import javax.security.auth.Subject;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.login.LoginException;
import javax.security.auth.spi.LoginModule;

import org.apache.log4j.Logger;

import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.baseutils.BaseUtilsHelper;
import com.mapr.fs.proto.Security.Key;
import com.mapr.fs.proto.Security.ServerKeyType;
import com.mapr.fs.proto.Security.TicketAndKey;
import com.mapr.security.ClusterServerTicketGeneration;
import com.mapr.security.MutableInt;
import com.mapr.security.Security;
import com.mapr.security.MapRPrincipal;
import com.mapr.login.client.MapRLoginClient;
import com.mapr.login.client.MapRLoginHttpsClient;

public class MaprSecurityLoginModule implements LoginModule {
  public static final String USER_TICKET_FILE_LOCATION = "MAPR_TICKETFILE_LOCATION";

  private static final Logger LOG = Logger.getLogger(MaprSecurityLoginModule.class);

  private static final String maprHome = BaseUtilsHelper.getPathToMaprHome();

  private Subject subject;
  private CallbackHandler callbackHandler;
  private Map<String, ?> options;
  private Map<String, ?> sharedState;
  private boolean succeeded = false;
  private boolean commitSucceeded = false;
  private MapRPrincipal principal = null;
  private boolean useServerKey;

  private static boolean useMaprServerTicket;
  private static boolean generatedServerKey = false;
  private static String cldbkeylocation;
  private static TicketAndKey maprServerTicketAndKey;

  private boolean checkUGI = true;
  static {
    // absolutely crazy figuring out, as Zookeeper prepends all the property names with "zookeeper."
    // which causes problems with parsing
    Set<Object> propsKeys = System.getProperties().keySet();
    for ( Object propKey : propsKeys ) {
      String propKeyStr = (String) propKey;
      if ( propKeyStr.endsWith("mapr.usemaprserverticket")) {
        useMaprServerTicket = Boolean.getBoolean(propKeyStr);
      }
      if ( propKeyStr.endsWith("mapr.cldbkeyfile.location")) {
        cldbkeylocation = System.getProperty(propKeyStr);
      }
    }

    if ( cldbkeylocation != null ) {
      MutableInt err = new MutableInt();
      int errCode = Security.SetKeyFile(ServerKeyType.CldbKey, cldbkeylocation);
      if (errCode != 0) {
        LOG.error("Failed to set cldb key file " + cldbkeylocation + " err " + err);
      } else {
        if (LOG.isInfoEnabled()) {
          LOG.info("Set the cldb key file to " + cldbkeylocation);
        }
      }
      Key cldbKey = Security.GetKey(ServerKeyType.CldbKey, err);
      if (cldbKey == null) {
        LOG.error("Cldb key can not be obtained: " + err.GetValue());
      }
      Key serverKey = Security.GetServerKey(cldbKey, 0);

      if ( serverKey == null ) {
        LOG.error("Server key can not be obtained");
      }

      errCode = Security.SetKey(ServerKeyType.ServerKey, serverKey);
      if ( errCode != 0 ) {
        LOG.error("Failed to set Server key with error: " + err);
      }
    }

  }

  private synchronized static void generateClusterServerTicket(String clusterName)
      throws IOException {
    if (!generatedServerKey) {
      ClusterServerTicketGeneration.getInstance().generateTicketAndSetServerKey(clusterName);
      generatedServerKey = true;
    }
  }

  private synchronized static TicketAndKey getMaprServerTicketAndKey(
      String clusterName) throws LoginException {
    if (maprServerTicketAndKey == null) {
      // "login" using maprserverticket
      String ticketPath = maprHome + "/conf/maprserverticket";
      if (!new File(ticketPath).exists()) {
        // actually if this happens it is some internal issue, since if file is not there
        // it would not pass login
        throw new LoginException("Security is enabled, but userTicketFile can not be found.");
      }
      Security.SetTicketAndKeyFile(ticketPath);

      MutableInt err = new MutableInt();
      TicketAndKey tk = Security.GetTicketAndKeyForCluster(ServerKeyType.CldbKey, clusterName, err);
      if ((tk != null) && err.GetValue() == 0) {
        maprServerTicketAndKey = tk;
        try {
          generateClusterServerTicket(clusterName);
        } catch (Throwable t) {
          LOG.warn("Unable to generate the server key.");
          LOG.debug(t.getMessage(), t);
        }
      } else {
        throw new LoginException("MapR user ticket not available! error = " + err);
      }
    }

    return maprServerTicketAndKey;
  }

  @Override
  public boolean abort() throws LoginException {
    if (succeeded == false) {
      return false;
    } else if (succeeded == true && commitSucceeded == false) {

      // Clean out state
      succeeded = false;
      principal = null;
    } else {
      // overall authentication succeeded and commit succeeded,
      // but someone else's commit failed
      logout();
    }
    return true;
  }

  @Override
  public boolean commit() throws LoginException {
    if ( !succeeded ) {
      return false;
    }

    if (subject.isReadOnly()) {
      throw new LoginException("Commit Failed: Subject is Readonly");
    }
    subject.getPrincipals().add(principal);
    commitSucceeded = true;

    return true;
  }

  @Override
  public void initialize(Subject subject, CallbackHandler callbackHandler,
      Map<String, ?> sharedState, Map<String, ?> options) {
    this.subject = subject;
    this.callbackHandler = callbackHandler;
    this.sharedState = sharedState;
    this.options = options;
    this.useServerKey = "true".equalsIgnoreCase((String)options.get("useServerKey"));
    if ("false".equalsIgnoreCase((String) options.get("checkUGI")))
      this.checkUGI = false;
    else this.checkUGI = true;
  }

  @Override
  public boolean login() throws LoginException {
    //TODO
    //at this point we really have no idea what cluster we are contacting
    //so we just use the default cluster as a proxy. That's not ideal
    //but the Hadoop APIs for this upon us as they ask "who are you?"
    //before making an outbound connection. With MapR your identity is
    //cluster specific, hence potential ambiguity.  We can do several things
    //that might help: 1) put in a MapRPrincipal for every cluster ticket
    //although that complicates principal retrieval.
    //2) have a way to update this object when/if the cluster is
    //determined. Not clear if (2) is possible.
    //3) put in a principal for the default cluster if available or any
    //cluster if there isn't one for the default. That should cover 90%.
    String cluster = CLDBRpcCommonUtils.getInstance().getCurrentClusterName();
    if ( cluster == null ) {
      throw new LoginException("Current cluster name is not found");
    }

    MutableInt err = new MutableInt();
    TicketAndKey tk = null;
    if (useMaprServerTicket) {
      tk = getMaprServerTicketAndKey(cluster);
    } else {
      try {
        boolean needToTryPlainTicket = false;
        if (useServerKey) {
          // server login a la KeyTabs
          try {
            generateClusterServerTicket(cluster);
          } catch (IOException e) {
            if ( LOG.isDebugEnabled() ) {
              LOG.debug("Unable to obtain MapR credentials", e);
            }
            needToTryPlainTicket = true;
          }
        } 
        if (!useServerKey || needToTryPlainTicket) {
          if ( LOG.isDebugEnabled()) {
            LOG.debug("Need to addplainticket: " + needToTryPlainTicket);
          }
          // client login a la kinit (maprlogin)
          //login in case I'm not yet authenticated to MapR
          MapRLoginClient c = new MapRLoginHttpsClient();
          c.setCheckUGI(checkUGI);
          c.authenticateIfNeeded(cluster);
        }
      } catch (IOException e) {
        if ( LOG.isDebugEnabled() ) {
          LOG.debug("Unable to obtain MapR credentials", e);
        }
        throw (LoginException)
            new LoginException("Unable to obtain MapR credentials").initCause(e);
      }

      tk = Security.GetTicketAndKeyForCluster(
          ServerKeyType.ServerKey, cluster, err);
    }

    if ((tk != null) && err.GetValue() == 0) {
      // have ticket without error, has it expired??
      String user = tk.getUserCreds().getUserName();
      principal = new MapRPrincipal(user, cluster);
    } else {
      throw new LoginException("MapR user ticket not available! error = " + err);
    }

    succeeded = true;
    return succeeded;
  }

  @Override
  public boolean logout() throws LoginException {
    Iterator<Principal> itr = subject.getPrincipals().iterator();
    while ( itr.hasNext()) {
      Object object = itr.next();
      if ( object instanceof MapRPrincipal ) {
        itr.remove();
      }
    }
    return true;
  }

  public static boolean isUseMaprServerTicket() {
    return useMaprServerTicket;
  }

}
