package com.mapr.security.maprauth;

import org.apache.hadoop.security.authentication.client.Authenticator;
import org.apache.hadoop.security.authentication.client.AuthenticationException;
import org.apache.hadoop.security.authentication.client.AuthenticatedURL;
import org.apache.hadoop.security.authentication.client.KerberosAuthenticator;
import org.apache.hadoop.security.authentication.client.PseudoAuthenticator;

import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.List;
import java.util.Map;

import org.apache.commons.codec.binary.Base64;

import com.google.protobuf.ByteString;
import com.mapr.fs.ShimLoader;
import com.mapr.fs.proto.Security.AuthenticationReqFull;
import com.mapr.fs.proto.Security.AuthenticationResp;
import com.mapr.fs.proto.Security.Key;
import com.mapr.fs.proto.Security.TicketAndKey;
import com.mapr.login.client.MapRLoginClient;
import com.mapr.login.client.MapRLoginHttpsClient;
import com.mapr.security.JNISecurity;
import com.mapr.security.MutableInt;
import com.mapr.security.Security;
import com.mapr.baseutils.JVMProperties;
import org.apache.hadoop.security.authentication.client.ConnectionConfigurator;


public class MaprAuthenticator implements Authenticator  {

    /**
     * HTTP header used by the MAPR server endpoint during an authentication sequence in case of error
     */
    public static final String WWW_ERR_AUTHENTICATE = "WWW-MAPR-Err-Authenticate";

    /**
     * HTTP header prefix used by the MAPR client/server endpoints during an authentication sequence.
     */
    public static final String NEGOTIATE = "MAPR-Negotiate";

    /**
     * HTTP connection endpoint.
     */
    private HttpURLConnection conn;

    /**
     * Oozie server URL
     */
    private URL               url;

    /**
     * key used by client to encrypt its messages
     */
    private Key               userkey;
    
    private ConnectionConfigurator connConfigurator;

    /**
     * authenticate function is invoked when the client tries to open a HTTP connection with the server
     * Based on the header returned by the server we will either do a Mapr handshake or
     * simply fallback to the PseudoAuthenticator
     * @param url   - Server url
     * @param token - Once authenticated this token can be used for future connections without
     *                having to go through the negotiation protocol
     * @throws IOException
     * @throws AuthenticationException
     */
    @Override
    public void authenticate(URL url, AuthenticatedURL.Token token) throws IOException, AuthenticationException {
        //ensure VM properties related to HTTP are set
        JVMProperties.init();

        this.url = url;

        conn = (HttpURLConnection) url.openConnection();
        if (connConfigurator != null) {
        	conn = connConfigurator.configure(conn);
        }
        conn.setRequestMethod("OPTIONS");
        conn.connect();

        if (isNegotiate())
        {
            /* Server set the HTTP header to initiate negotiation
             * Begin the Mapr negotiation handshake
             */
            ShimLoader.load();
            doMaprHandshake(token);
        }
        else
        {
          // TODO - not sure what to do here
            // Authentication is not enabled, fallback to simple authenticator 
            getFallbackAuthenticator().authenticate(url, token);
        }
    }

    public void setConnectionConfigurator(ConnectionConfigurator configurator) {
      connConfigurator = configurator;
    }

    /**
     * This function does the brunt of the work for HTTP authentication on the client side.
     * It completes the authentication in two passes.
     * In the first pass, we get the ticket and key information from the local machine.
     * Within this we have the userkey, encrypted ticket etc.
     * We generate a random number which will be our challenge request and encrypt it using
     * the userkey.
     * We send the challenge request (encrypted with userkey) and encrypted ticket to the server
     * The server should be able to decrypt this message and respond with the appropriate challenge
     * response. At this point we are done with our first pass
     *
     * During the next pass we verify the challenge response sent by the server and simply send an
     * acknowledgement to the server indicating the challenge response is validated and client has
     * completed the authentication sequence.
     *
     * @param token -  Once authenticated this token can be used for future connections without
     *                having to go through the negotiation protocol
     * @throws IOException
     * @throws AuthenticationException
     */
    private void doMaprHandshake(AuthenticatedURL.Token token) throws IOException, AuthenticationException {

        try
        {
            MapRLoginClient client = new MapRLoginHttpsClient();
            TicketAndKey ticketKey = client.authenticateIfNeeded();

            if (ticketKey == null) {
                throw new AuthenticationException("ServerTicketKey was not set");
            }

            MutableInt err = new MutableInt();
            userkey = ticketKey.getUserKey();
            long randomSecret = JNISecurity.GenerateRandomNumber();
            randomSecret =  Math.abs(randomSecret);

            byte[] writeBuffer = new byte[8];

            writeBuffer[0] = (byte)(randomSecret >>> 56);
            writeBuffer[1] = (byte)(randomSecret >>> 48);
            writeBuffer[2] = (byte)(randomSecret >>> 40);
            writeBuffer[3] = (byte)(randomSecret >>> 32);
            writeBuffer[4] = (byte)(randomSecret >>> 24);
            writeBuffer[5] = (byte)(randomSecret >>> 16);
            writeBuffer[6] = (byte)(randomSecret >>>  8);
            writeBuffer[7] = (byte)(randomSecret >>>  0);

            AuthenticationReqFull.Builder bld = AuthenticationReqFull.newBuilder();
            byte [] secretBytesEncrypted = Security.Encrypt(userkey, writeBuffer, err);

            if (err.GetValue() != 0) {
                throw new AuthenticationException("Error while encrypting data: " + err.GetValue());
            }

            bld.setEncryptedRandomSecret(ByteString.copyFrom(secretBytesEncrypted));
            bld.setEncryptedTicket(ticketKey.getEncryptedTicket());
            byte [] authRequestBytes = bld.build().toByteArray();

            /* Send the authentication request bytes we created */
            sendBytes(authRequestBytes);

            AuthenticationResp authResponse = readResponse();

            if (authResponse.hasChallengeResponse()) {
                long responseSecret = authResponse.getChallengeResponse();
                if (responseSecret != (randomSecret + 1))
                {
                    throw new AuthenticationException("Incorrect challenge response");
                }
            }
            else
            {
                throw new AuthenticationException("No response secret");
            }

            /* Extract the authentication token from the connection
             * and stash it into the local variable - token. If token
             * is set we will not perform the handshake again
             */
            AuthenticatedURL.extractToken(conn, token);
        } catch (Throwable t) {

            t.printStackTrace();

            if (t instanceof AuthenticationException) {
                throw (AuthenticationException) t;
            }

            throw new AuthenticationException("Exception while getting ticket and key", t);
        }
    }

    /**
     * Function sets the HTTP header AUTHORIZATION with the passed in string
     * and connects to the server
     * @param authorizationToken - HTTP authorization header value to be set
     * @throws IOException
     * @throws AuthenticationException
     */
    private void sendToken(String authorizationToken) throws IOException, AuthenticationException {
        conn = (HttpURLConnection) url.openConnection();
        conn.setRequestMethod("OPTIONS");
        conn.setRequestProperty(KerberosAuthenticator.AUTHORIZATION, NEGOTIATE + " " + authorizationToken);
        conn.connect();
    }

    /**
     * Function converts the set of bytes to string using Base64 encoding and invokes
     * the method to set the string in the HTTP header and connect to the server
     * @param authRequestBytes - Authentication request bytes containing encrypted ticket and random secret
     * @throws IOException
     * @throws AuthenticationException
     */
    private void sendBytes(byte[] authRequestBytes) throws IOException, AuthenticationException {
        Base64 base64 = new Base64(0);
        String token  = base64.encodeToString(authRequestBytes);

        /* Send the base64 encoded string */
        sendToken(token);
    }

    /**
     * Function reads the response once client connects to the server.
     * Decodes the Base64 encoded string into bytes and constructs the
     * AuthenticationResp class from these bytes and returns this to caller
     * @return
     * @throws IOException
     * @throws AuthenticationException
     */
    private AuthenticationResp readResponse() throws IOException, AuthenticationException {

        int status = conn.getResponseCode();

        /* Check to see if the error header is set
         * If its set, throw an exception and get out
         */
        String authorizationError = conn.getHeaderField(WWW_ERR_AUTHENTICATE);
        if (authorizationError != null)
        {
            String err = authorizationError.trim();
            throw new AuthenticationException("Exception in server: " + err);
        }

        /* Verify that the server responded with 200. If it did it would mean
         * that the server has authenticated the client and generated the
         * authentication token. Client still has to make sure that it
         * verifies the server and whether to continue communication to the
         * server or not
         */
        if (status == HttpURLConnection.HTTP_OK)
        {
          String authHeader = null;
          Map<String, List<String>> headers = conn.getHeaderFields();
          List<String> wwwAuthHeaders = headers.get(KerberosAuthenticator.AUTHORIZATION);
          if ( wwwAuthHeaders == null ) {
            throw new AuthenticationException("No header : " +  KerberosAuthenticator.AUTHORIZATION + " is present");
          }
          for ( String header : wwwAuthHeaders ) {
            if ( header != null && header.trim().startsWith(NEGOTIATE) ) {
              authHeader = header;
            }
          }

          if (authHeader == null ) {
              throw new AuthenticationException("Invalid sequence, incorrect header" + wwwAuthHeaders);
          }

            String negotiation = authHeader.trim().substring((NEGOTIATE + " ").length()).trim();
            Base64 base64 = new Base64(0);
            byte[] base64Bytes = base64.decode(negotiation);

            MutableInt err = new MutableInt();

            byte[] decodedResponse = Security.Decrypt(userkey, base64Bytes, err);

            if (err.GetValue() != 0) {
                throw new AuthenticationException("Error while decrypting response " + err.GetValue());
            }

            AuthenticationResp authResponse = null;

            authResponse = AuthenticationResp.parseFrom(decodedResponse);

            if (authResponse == null)
                throw new AuthenticationException("Response is null");

            return authResponse;
        }
        else
        {
            /* Invalid status */
            throw new AuthenticationException("Incorrect status" + status);
        }
    }

    private Authenticator getFallbackAuthenticator() {
        return new PseudoAuthenticator();
    }

    private boolean isNegotiate() throws IOException {
        boolean negotiate = false;
        if (conn.getResponseCode() == HttpURLConnection.HTTP_UNAUTHORIZED) {
            Map<String, List<String>> headers = conn.getHeaderFields();
            List<String> wwwAuthHeaders = headers.get(KerberosAuthenticator.WWW_AUTHENTICATE);
            if ( wwwAuthHeaders != null ) {
              for ( String authHeader : wwwAuthHeaders ) {
                if ( authHeader != null && authHeader.trim().startsWith(NEGOTIATE) ) {
                  return true;
                }
              }
            }
        }
        return negotiate;
    }

}

