package com.mapr.security;

import java.io.BufferedReader;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.ByteBuffer;
import java.util.List;

import com.google.protobuf.ByteString;
import com.google.protobuf.InvalidProtocolBufferException;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.fs.proto.Security.CredentialsMsg;
import com.mapr.fs.proto.Security.Key;
import com.mapr.fs.proto.Security.ServerKeyType;
import com.mapr.fs.proto.Security.TicketAndKey;
import com.mapr.fs.proto.Security.Ticket;
import com.mapr.security.JNISecurity.MutableErr;

class Errno {
  public static final int EINVAL = 22;
}

public class Security {
/* Some callers are outside the MapR realm, so we must ensure that the client
   library is loaded.  Use ShimLoader for this to ensure that the library is
   only loaded once
*/
  static {
    com.mapr.fs.ShimLoader.load();
  }

  private static int KeySizeInBytes = 32;
  // Create private constructor to ensure that
  // no object is created for Security class
  // all the methods are static
  private Security() {
  }

  //----------------------------------------------------------------//
  //----------------------PUBLIC FUNCTIONS--------------------------//
  //----------------------------------------------------------------//
  // Initialise the security class 
  // should be called before using any of the methods
  //public static long MAX_EXPIRY_TIME = Long.MAX_VALUE; //0x7ffffffffffffffff
  public static long MAX_EXPIRY_TIME = Long.MAX_VALUE / 10000; //0x7ffffffffffffffff
  // returns a randomly generated key
  public static Key GenerateRandomKey() {
    Key.Builder key = Key.newBuilder();
    byte [] keyBuf = new byte[KeySizeInBytes];
    JNISecurity.GenerateRandomBlock(keyBuf);
    key.setKey(ByteString.copyFrom(keyBuf));
    return key.build();
  }  

  // Encrypts data in inBuf by key "key".
  // On failure the error is returned in err parameter
  // On success the return value indicates contains encrypted data
  public static byte[] Encrypt(Key key, byte [] inBuf, MutableInt erri) {
    MutableErr err = new MutableErr();
    
    byte [] encrypted = JNISecurity.Encrypt(key.getKey().toByteArray(), inBuf, err);
    
    if (err.GetValue() != 0) {
      erri.SetValue(err.GetValue());
    }
    
    return encrypted;
  }
  

  // Decrypts data in inBuf by key "key"
  // On failure the error is returned in err parameter
  // On success the return byte array has decrypted data
  public static byte[]
      Decrypt(Key key, byte[] inBuf, MutableInt erri) {
    MutableErr err = new MutableErr();
    
    byte [] decrypted = JNISecurity.Decrypt(key.getKey().toByteArray(), inBuf, err);
    
    if (err.GetValue() != 0) {
      erri.SetValue(err.GetValue());
    }
    
    return decrypted;
  }

  // Encodes the serializedProtobuf data for writing to file. 
  // on failure error is returned in err parameter
  // on success the encoded data is returned
  public static byte[] 
    EncodeDataForWritingToKeyFile(byte[] serializedProtobuf, MutableInt erri) {
    MutableErr err = new MutableErr();
    
    byte [] encoded = 
        JNISecurity.EncodeDataForWritingToKeyFile(serializedProtobuf, err);
    
    if (err.GetValue() != 0) {
      erri.SetValue(err.GetValue());
    }
    
    return encoded;
  }

  // Decodes the encoded data read from file.
  // on failure error is returned in err parameter
  // on success the return byte array has decoded data
  public static byte[]
  DecodeDataFromKeyFile(byte[] encodedData, MutableInt erri) {
    MutableErr err = new MutableErr();
    
    byte [] decoded = 
        JNISecurity.DecodeDataFromKeyFile(encodedData, err);
    
    if (err.GetValue() != 0) {
      erri.SetValue(err.GetValue());
    }
    
    return decoded;
  }

  // Sets the keyfile. Security module shall read the key and treat it of the 
  // given keyType. Returns any error in reading the file
  public static int SetKeyFile(ServerKeyType keyType, String path) {
    return JNISecurity.SetKeyFileInternal(keyType.getNumber(), path);
  }

  // API for setting the key of given type 
  // returns error on failure
  public static int SetKey(ServerKeyType keyType, Key key) {
    return JNISecurity.SetKeyInternal(keyType.getNumber(), key.getKey().toByteArray());
  }

  // Returns the key of given type which would have been set earlier
  // by SetKeyFile or SetKey. On failure return value is null and
  // error is set in SecurityError
  public static Key GetKey(ServerKeyType keyType, MutableInt erri) {
    MutableErr err = new MutableErr();
      
    byte [] keyBuf = JNISecurity.GetKeyInternal(keyType.getNumber(), err);
    if (keyBuf != null) {
      Key.Builder key = Key.newBuilder();
      key.setKey(ByteString.copyFrom(keyBuf));
      return key.build();
    }
    else if (err.GetValue() != 0) {
      erri.SetValue(err.GetValue());
    }
        
    return null;
  }
  
  // Generates TicketAndKey protobufs using key of given type
  // UserName, uid, gids and expirytime is used to initialise
  // the ticket. On success TicketAndKey object is returned
  // on failure TicketAndKey is null and error is in err parameter
  public static TicketAndKey GenerateTicketAndKey (
      ServerKeyType keyType, String userName,
      int uid, int [] gids, long expiryTime,
      long maxRenewalTimeSec, boolean isExternal,
      boolean canUserImpersonate, MutableInt erri)  {

    MutableErr err = new MutableErr();
    
    byte [] serializedTicketAndKey =
    	JNISecurity.GenerateTicketAndKeyInternal(keyType.getNumber(), userName, uid,
                                                 gids, expiryTime, maxRenewalTimeSec, 
                                                 isExternal, canUserImpersonate, err);
    if (serializedTicketAndKey != null) {
      try {
        return TicketAndKey.parseFrom(serializedTicketAndKey);
      } catch (InvalidProtocolBufferException e) {
        e.printStackTrace();
        return null;
      }
    }
    else if (err.GetValue() != 0) {
      erri.SetValue(err.GetValue());
    }
    
    return null;
  }
 
  /**
   * Used by JT to generate a ticket for user for a specific job
   * @param userName 
   * @param expiration - will be configurable on JT
   * @param outTicket - DataOutputStream where ticket will be written
   * @throws IOException
   */
  public static void GenerateAndWriteTicketAndKey(String userName, 
      long expiration, DataOutputStream outTicket) throws IOException {
    UnixUserGroupHelper uugh = new UnixUserGroupHelper();
    int uid = uugh.getUserId(userName);
    int [] gids = uugh.getGroups(userName);
    MutableInt mutableerr = new MutableInt();
    
    // Generate the MapRUserTicket now
    TicketAndKey ticketAndKey = Security.GenerateTicketAndKey(
        ServerKeyType.ServerKey, 
        userName, 
        uid, 
        gids,
        expiration/1000, // convert to seconds 
        0 /*not-renewable*/,
        true /*isExternal*/,
        false /*canUserImpersonate*/,
        mutableerr);
    
    if ( ticketAndKey != null && mutableerr.GetValue() == 0 ) {
      byte [] encodedticket = 
        Security.EncodeDataForWritingToKeyFile(
            ticketAndKey.toByteArray(), mutableerr);
      if ( mutableerr.GetValue() != 0 || encodedticket == null ) {
        throw new IOException("Security Ticket for user: "
            + userName + " can not be encoded with error: " + mutableerr.GetValue());
      }

      try {
        String currentClusterName = CLDBRpcCommonUtils.getInstance().getCurrentClusterName();
        outTicket.write(currentClusterName.getBytes("UTF-8"));
        outTicket.write(new String(" ").getBytes("UTF-8"));
        outTicket.write(encodedticket);
        outTicket.write(new String("\n").getBytes("UTF-8"));
      } finally {
        outTicket.close();
      }
    } else {
      throw new IOException("Security Ticket for user: "
          + userName + " can not be generated with error: " + mutableerr.GetValue());
    }
  }

  /**
   * Given an input stream for ticket file, clone the current cluster
   * ticket with given expiration. It generates a non-renewable ticket.
   *
   * Parameters that will be cloned are uid, gids, canImpersonate
   *
   * @param fsin - DataInputStream for input ticket
   * @param expiration - expiration of the cloned ticket.
   * @param outTicket - DataOutputStream where ticket will be written
   * @throws IOException where input ticket cannot be parsed
   *                     for current cluster.
   */
  public static void CloneAndGenerateTicketFile(DataInputStream fsin,
      long expiration, DataOutputStream outTicket)
      throws IOException {
    if (fsin == null) {
      throw new IOException("Security Ticket can not be cloned "
          + "as DataInputStream is null");
    }
    
    MergeAndGenerateTicketFile(fsin, null /*userName*/, expiration,
                               outTicket, true /*cloneTicket*/);
  }

  public static void MergeAndGenerateTicketFile(DataInputStream fsin, String userName, 
      long expiration, DataOutputStream outTicket) throws IOException {
    if ( fsin == null ) {
      // just generate ticket and be done
      GenerateAndWriteTicketAndKey(userName, 
          expiration, outTicket);
      return;
    }

    MergeAndGenerateTicketFile(fsin, userName, expiration, outTicket,
                               false /*cloneTicket*/);
  }

  private static void MergeAndGenerateTicketFile(DataInputStream fsin,
      String userName, long expiration, DataOutputStream outTicket,
      boolean cloneTicket) throws IOException {
    boolean isTicketGenerated = false;
    MutableInt err = new MutableInt();
    String currentClusterName = CLDBRpcCommonUtils.getInstance().getCurrentClusterName();
    // looks like source file may exist
    BufferedReader in = new BufferedReader(new InputStreamReader(fsin, "UTF-8"));
    try {
      String line = null;
      while ((line = in.readLine()) != null) {
        String [] splits = line.split(" ");
        if ( splits == null || splits.length < 2 ) {
          // something is malformed in the ticket
          // skip this line
          continue;
        }
        String cluster = splits[0];
        if ( currentClusterName.equals(cluster) ) {
          String encodedTicket = splits[1];
          byte[] ticketBytes = Security.DecodeDataFromKeyFile(encodedTicket.getBytes(), err);
          TicketAndKey tk = TicketAndKey.parseFrom(ticketBytes);
          TicketAndKey ticketAndKey = null;
          MutableInt mutableerr = new MutableInt();
          // this is ticket for current cluster - which we are going to regen anyway
          // Generate the MapRUserTicket now
          if ( tk != null && 
               err.GetValue() == 0 && 
               tk.getUserCreds() != null) {
              boolean canImpersonate = (cloneTicket) ?
                        tk.getCanUserImpersonate() : false;
              List<Integer> gidList = tk.getUserCreds().getGidsList();
              int [] gids = new int[gidList.size()];
              for ( int i = 0; i < gidList.size(); i++) {
                gids[i] = gidList.get(i);
              }
              ticketAndKey = Security.GenerateTicketAndKey(
              ServerKeyType.ServerKey, 
              tk.getUserCreds().getUserName(), 
              tk.getUserCreds().getUid(), 
              gids,
              expiration/1000, // convert to seconds 
              0 /*not-renewable*/,
              true /*isExternal*/,
              canImpersonate,
              mutableerr);
          } else if (cloneTicket) {
            // This is not expected in clone case. Need current cluster ticket.
              throw new IOException("Security Ticket "
                  + " can not be cloned as parameters to "
                  + "be cloned cannot be retreived from ticket");
          } else {
            // gen completely new
            UnixUserGroupHelper uugh = new UnixUserGroupHelper();
            int uid = uugh.getUserId(userName);
            int [] gids = uugh.getGroups(userName);
            
            // Generate the MapRUserTicket now
            ticketAndKey = Security.GenerateTicketAndKey(
                ServerKeyType.ServerKey, 
                userName, 
                uid, 
                gids,
                expiration/1000, // convert to seconds 
                0 /*not-renewable*/,
                true /*isExternal*/,
                false /*canUserImpersonate*/,
                mutableerr);
         }
          // write it out
          if ( ticketAndKey != null && mutableerr.GetValue() == 0 ) {
            byte [] encodedticket = 
              Security.EncodeDataForWritingToKeyFile(
                  ticketAndKey.toByteArray(), mutableerr);
            if ( mutableerr.GetValue() != 0 || encodedticket == null ) {
              throw new IOException("Security Ticket for user: "
                  + userName + " can not be encoded with error: " + mutableerr.GetValue());
            }
            outTicket.write(currentClusterName.getBytes("UTF-8"));
            outTicket.write(new String(" ").getBytes("UTF-8"));
            outTicket.write(encodedticket);
            outTicket.write(new String("\n").getBytes("UTF-8"));
            isTicketGenerated = true;
           } else {
            throw new IOException("Security Ticket for user: "
                + userName + " can not be generated with error: " + mutableerr.GetValue());
           }
        } else {    
          outTicket.write(line.getBytes("UTF-8"));
          outTicket.write(new String("\n").getBytes("UTF-8"));
        }
      }
      if ( !isTicketGenerated && !cloneTicket) {
        // just in case
        GenerateAndWriteTicketAndKey(userName,
           expiration, outTicket);
       return;
      }
    } finally {
      outTicket.close();
    }
  }
  
  // Same as above except that it takes the key to generate ticket also
  // as input.
  public static TicketAndKey GenerateTicketAndKeyUsingServerKey(
    ServerKeyType keyType, Key key, String userName,
    int uid, int [] gids, long expiryTime,
    long maxRenewalTimeSec, boolean isExternal,
    MutableInt erri) {

    MutableErr err = new MutableErr();
    
    byte [] serializedTicketAndKey =
      JNISecurity.GenerateTicketAndKeyUsingServerKeyInternal(
             keyType.getNumber(), key.getKey().toByteArray(), 
             userName, uid, gids, expiryTime, maxRenewalTimeSec,
             isExternal, err);
    
    if (serializedTicketAndKey != null) {
      try {
        return TicketAndKey.parseFrom(serializedTicketAndKey);
      } catch (InvalidProtocolBufferException e) {
        e.printStackTrace();
        return null;
      }
    }
    else if (err.GetValue() != 0) {
      erri.SetValue(err.GetValue());
    }
    
    return null;
  }

  // Renews the input ticket and key and returns the renwed
  // ticket and key. It returns error if the input ticket 
  // is already expired or if the ticket is not renewable
  // or the ticket has crossed max renewable time
  public static TicketAndKey RenewTicketAndKey(
      ServerKeyType keyType,
      TicketAndKey ticketAndKey, long expiryTime,
      MutableInt erri)  {

    MutableErr err = new MutableErr();

    byte[] serializedInTicketAndKey = ticketAndKey.toByteArray();
    
    byte [] serializedTicketAndKey =
    	JNISecurity.RenewTicketAndKeyInternal(
                        keyType.getNumber(), serializedInTicketAndKey,
                        expiryTime, err);
    if (serializedTicketAndKey != null) {
      try {
        return TicketAndKey.parseFrom(serializedTicketAndKey);
      } catch (InvalidProtocolBufferException e) {
        e.printStackTrace();
        return null;
      }
    }
    else if (err.GetValue() != 0) {
      erri.SetValue(err.GetValue());
    }
    return null;
  }
  

  // Decrypts the input ticket using the server key. Server key is 
  // set by the server using SetKeyFile/SetKey calls earlier.
  // On success returns the Ticket object. On failure, the ticket
  // object is null and error is set in err.
  public static Ticket 
    DecryptTicket(byte[] encryptedTicket, MutableInt erri) {
    
    MutableErr err = new MutableErr();
    
    byte [] serializedTicket = JNISecurity.DecryptTicketInternal(encryptedTicket, err);
    if (serializedTicket != null) {
      try {
        return Ticket.parseFrom(serializedTicket);
      } catch (InvalidProtocolBufferException e) {
        e.printStackTrace();
        return null;
      }
    }
    else if (err.GetValue() != 0) {
      erri.SetValue(err.GetValue());
    }
    
    return null;
  }

  // Sets the ticket file path. The Security module shall read the tickets
  // for different cluster from this file. On failure error code is returned
  public static int SetTicketAndKeyFile(String path) {
    return JNISecurity.SetTicketAndKeyFileInternal(path);
  }

  // API mechanism to set the TicketAndKey from given keytype and clustername
  // on failure error code is returned by function
  public static int SetTicketAndKey(ServerKeyType keyType,
      String clusterName, TicketAndKey ticketAndKey) {
    return 
       JNISecurity.SetTicketAndKeyInternal(keyType.getNumber(), clusterName, 
                                           ticketAndKey.toByteArray());
  }

  // API to copy all ClusterTickets to ServerTickets in KeyStore
  public static void UseClusterTicketAsServerTicket() {
    JNISecurity.UseClusterTicketAsServerTicketInternal();
  }

  // Returns the TicketAndKey for given keytype and clustername.
  // TicketAndKey are set earlier by calling SetTicketFile/SetTicket API.
  // on success TicketAndKey object is returned on failure the returned object
  // is null and error code is set in err
  public static TicketAndKey
      GetTicketAndKeyForCluster(ServerKeyType keyType,
                                String clusterName, MutableInt erri) {
    MutableErr err = new MutableErr();
    
    byte [] serializedTicketAndKey = 
    	JNISecurity.GetTicketAndKeyForClusterInternal(keyType.getNumber(), 
    			                                      clusterName, err);
    if (serializedTicketAndKey != null) {
      try {
        return TicketAndKey.parseFrom(serializedTicketAndKey);
      } catch (InvalidProtocolBufferException e) {
        e.printStackTrace();
        return null;
      }
    }
    else if (err.GetValue() != 0) {
      erri.SetValue(err.GetValue());
    }
    
    return null;
  }

  // returns the Server key based on the CLdbKey and time
  public static Key GetServerKey(Key CldbKey, long time) {
    ByteBuffer bBuffer = ByteBuffer.allocate(KeySizeInBytes + 8);
    // for the time being use 0 as the time always
    // once we have renewal of ServerKey then based on the 
    // time we will select the time which is to be used here
    time = 0;
    bBuffer.putLong(time);
    bBuffer.put(CldbKey.getKey().toByteArray());
    byte [] hash = JNISecurity.GetHash(0, bBuffer.array());
    Key.Builder key = Key.newBuilder();
    key.setKey(ByteString.copyFrom(hash));
    return key.build();
  }

  // returns the cluster key based on the CLdbKey and time
  public static Key GetClusterKey(Key CldbKey, long time) {
    ByteBuffer bBuffer = ByteBuffer.allocate(KeySizeInBytes + 8);
    // for the time being use 0 as the time always
    // once we have renewal of ServerKey then based on the 
    // time we will select the time which is to be used here
    time = 97;  //some number.
    bBuffer.putLong(time);
    bBuffer.put(CldbKey.getKey().toByteArray());
    byte [] hash = JNISecurity.GetHash(0, bBuffer.array());
    Key.Builder key = Key.newBuilder();
    key.setKey(ByteString.copyFrom(hash));
    return key.build();
  }

  // Helper functions for testing and debugging
  public static String ByteArrayToString(String preFix, byte []bArray) {
    StringBuffer buf = new StringBuffer();
    buf.append(preFix + "\n");
    for (byte b : bArray) {
      buf.append(String.format("%02X", b));
    }
    return buf.toString();
  }
  
  public static String UserCredsToString(String prefix, CredentialsMsg c) {
    StringBuffer buf = new StringBuffer();
    buf.append(prefix + "\n");
    buf.append("UserName " + c.getUserName() + "\n");
    buf.append("UID " + c.getUid() + "\n");
    buf.append("GIDS " + c.getGids(0) + "\n");
    return buf.toString();
  }
  
  public static String TicketAndKeyToString(String prefix, TicketAndKey t) {
    StringBuffer buf = new StringBuffer();
    buf.append(prefix + "\n");
    buf.append(ByteArrayToString("EncryptedTicket", 
        t.getEncryptedTicket().toByteArray()) + "\n");
    buf.append(UserCredsToString("UserCreds ", t.getUserCreds()) + "\n");
    buf.append(ByteArrayToString("UserKey ", t.getUserKey().getKey().toByteArray()) + "\n");
    return buf.toString();
  }

  public static String TicketToString(String prefix, Ticket t) {
    StringBuffer buf = new StringBuffer();
    buf.append(prefix + "\n");
    buf.append(UserCredsToString("UserCreds ", t.getUserCreds()) + "\n");
    buf.append(ByteArrayToString("UserKey ", t.getUserKey().getKey().toByteArray()) + "\n");
    return buf.toString();
  }


  // Validate that the ticket and key is a valid combo and is decryptable 
  // from the server key
  public static Ticket ValidateTicketAndKey(TicketAndKey t, MutableInt err) {
    Ticket ticket = Security.DecryptTicket(t.getEncryptedTicket().toByteArray(), err);
    if (ticket == null) {
      err.SetValue(Errno.EINVAL);
      return null;
    }

    // Able to decrypt the ticket now match the UserKey in the ticket and
    // userkey in the ticketandkey sent by the caller. 

    byte [] userKey = t.getUserKey().getKey().toByteArray();
    byte [] keyInTicket = ticket.getUserKey().getKey().toByteArray();

    if (userKey.length != keyInTicket.length) {
      err.SetValue(Errno.EINVAL);
      return null;
    }
    for (int i = 0; i < userKey.length; ++i) {
      if (userKey[i] != keyInTicket[i]) {
        err.SetValue(Errno.EINVAL);
        return null;
      }
    }
    return ticket;
  }

  // Returns true if the ticket and key are yet to expire else false
  public static boolean IsTicketAndKeyUsable(TicketAndKey ticketAndKey) {
    if (ticketAndKey != null) {
      long currentTimeSec = System.currentTimeMillis() / 1000;
      long expiryTimeSec = ticketAndKey.getExpiryTime();
      return currentTimeSec < expiryTimeSec;
    }
    return false;
  }

  public static void BlacklistAndCloseConnections(int[] uids,
                                                 long[] blacklistTimes,
                                                 boolean reinitList) {
    if (uids == null || uids.length == 0) {
      System.out.println("Incorrect uids specified for blacklisting");
      return;
    }

    if (blacklistTimes == null || blacklistTimes.length == 0) {
      System.out.println("Incorrect uids specified for blacklisting");
      return;
    }

    if (uids.length != blacklistTimes.length) {
      System.out.println("Number of uids passed does not match " + 
                         "number of blacklist times passed");
      return;
    }

    JNISecurity.BlacklistAndCloseConnections(uids, 
                                             blacklistTimes,
                                             reinitList);
  }

  public static void RemoveFromBlacklist(int[] uids) {
     if (uids == null || uids.length == 0) {
      System.out.println("No uids to be cleaned up from blacklist");
      return;
    }

    JNISecurity.RemoveFromBlacklist(uids);
  }

  public static void TestSecurity() {

    // SetTicketAndKeyFile
    // GetTicketAndKey for server
    // DecryptTicket 
    try {
    // Generate a random key. Put the key in key file.
    MutableInt err = new MutableInt();
    Key k = Security.GenerateRandomKey();
    byte [] fileData = Security.EncodeDataForWritingToKeyFile(k.toByteArray(), err);
    FileOutputStream fout = new FileOutputStream("/tmp/key-file");
    fout.write(fileData);
    fout.write(new String("\n").getBytes("UTF-8"));
    fout.close();
    System.out.println(ByteArrayToString("Key", k.getKey().toByteArray()));

    // Set the key file on server
    Security.SetKeyFile(ServerKeyType.CldbKey, "/tmp/key-file");

    // Generate TicketAndKey using the key
    int []gids = new int[1];
    gids[0] = 0;

    TicketAndKey ticketAndKey = Security.GenerateTicketAndKey(ServerKeyType.CldbKey, "root", 0, gids,
                                                    Security.MAX_EXPIRY_TIME, 0 /*non-renewal*/, true,
                                                    false /*canUserImpersonate*/, err);

    // Write TicketAndKey to a file
    fileData = Security.EncodeDataForWritingToKeyFile(ticketAndKey.toByteArray(), err);
    String clusterName = "default ";
    fout = new FileOutputStream("/tmp/ticket-file");
    fout.write(clusterName.getBytes("UTF-8"));
    fout.write(fileData);
    fout.write(new String("\n").getBytes("UTF-8"));
    fout.close();
    Security.SetTicketAndKeyFile("/tmp/ticket-file");
    System.out.println("Calling GetTicketAndKeyForCluster");    

    Key cldbKey = Security.GetKey(ServerKeyType.CldbKey, err);
    Key serverKey = Security.GetServerKey(cldbKey, System.currentTimeMillis());
    System.out.println(ByteArrayToString("CLDB Key", cldbKey.getKey().toByteArray()));
    System.out.println(ByteArrayToString("Server Key", serverKey.getKey().toByteArray()));

    Ticket t = ValidateTicketAndKey(ticketAndKey, err);
    System.out.println(TicketToString(" Anurag - Validated Ticket ", t));

    TicketAndKey ticketAndKey2 = Security.GetTicketAndKeyForCluster(ServerKeyType.CldbKey, "default", err);
    System.out.println(TicketAndKeyToString("default", ticketAndKey2));
    System.out.println("\n");

    Ticket ticket = Security.DecryptTicket(ticketAndKey2.getEncryptedTicket().toByteArray(), err); 
    System.out.println(TicketToString("Ticket ", ticket));
    System.out.println("\n");
    } catch (Exception e) {
      System.out.println( "Exception " + e);
    }
/*
    System.out.println("Encoded data is " + new String(fileData, "UTF-8"));

    byte[] data = new byte[32];
    Arrays.fill(data, (byte) 0x30);
    MutableInt err = new MutableInt();
    byte [] enryptedData = Security.Encrypt(k, data, err);
    byte [] decryptedData = Security.Decrypt(k, enryptedData, err);
    System.out.println(ByteArrayToString("Key", k.getKey().toByteArray()));
    System.out.println(ByteArrayToString("Plain Data ", data));
    System.out.println(ByteArrayToString("EncryptedData ", enryptedData));
    System.out.println(ByteArrayToString("DecryptedData ", decryptedData));

    // Set the key
    TicketAndKey ticketAndKey2 = Security.GenerateTicketAndKeyUsingServerKey(ServerKeyType.CldbKey, k, "mapr", 100, gids, Security.MAX_EXPIRY_TIME, err);

    System.out.println(TicketAndKeyToString("TicketAndKey2", ticketAndKey2));
    System.out.println("\n");

    Ticket ticket = Security.DecryptTicket(ticketAndKey.getEncryptedTicket().toByteArray(), err);
    System.out.println(TicketToString("Ticket ", ticket));
    System.out.println("\n");

    try {
    byte [] fileData = Security.EncodeDataForWritingToKeyFile(ticketAndKey.toByteArray(), err);
    System.out.println("Encoded data is " + new String(fileData, "UTF-8"));
    String s = new String(fileData, "UTF-8");
    byte [] newData = s.getBytes(Charset.forName("UTF-8"));
    byte [] decodedData = Security.DecodeDataFromKeyFile(newData, err);
    TicketAndKey t1 = TicketAndKey.parseFrom(decodedData);
    System.out.println(TicketAndKeyToString("TicketAndKey", t1));
    } catch (Exception e) {
      System.out.println("Exception " + e);
    }
*/    
  }

  public static void main(String[] args) { 
    System.out.println("Hello, World");
    TestSecurity();
  }
}  
