package com.mapr.security;

import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.security.maprauth.MaprAuthenticator;
import org.apache.hadoop.security.authentication.client.AuthenticatedURL;
import org.apache.hadoop.security.authentication.client.AuthenticatedURL.Token;
import org.apache.hadoop.security.authentication.client.AuthenticationException;
import org.apache.log4j.Logger;

import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.URL;

/**
 * 1. Returns a usable http(s) connection.
 * 2. Caches a authentication token and re-uses it for connections long as it is valid.
 * 3. Performs authentication via MaprAuthenticator only if the token becomes invalid.
 * <p/>
 * TODO(Santosh):
 * 1. This code is not cluster aware (it always watches if security is enabled or not on the current cluster)
 * 2. This code is not user aware (MaprAuthenticator will pickup creds of the process uid that's running this code)
 * <p/>
 * Author: smarella
 */
public class MaprHttpURL {
  private static final Logger LOG = Logger.getLogger(MaprHttpURL.class);

  private static Token token = new Token();

  public static HttpURLConnection openConnection(URL url) throws IOException {
    if (JNISecurity.IsSecurityEnabled(CLDBRpcCommonUtils.getInstance().getCurrentClusterName())) {
      synchronized (MaprHttpURL.class) {
        HttpURLConnection conn = null;
        if (token.isSet()) { // try the existing token
          conn = (HttpURLConnection) url.openConnection();
          AuthenticatedURL.injectToken(conn, token);
          conn.connect();
          if (conn.getResponseCode() == HttpURLConnection.HTTP_UNAUTHORIZED) {
            token = new Token(); // refresh the token on auth failure
            LOG.debug("Received HTTP " + conn.getResponseCode() + ". Created a new token.");
          } else {
            LOG.debug("Received HTTP " + conn.getResponseCode() + ". Token still good..");
          }
        }

        if (!token.isSet()) { // authenticate only if the token is not set
          LOG.debug("Token not set. Peforming authentication..");
          AuthenticatedURL authenticatedURL = new AuthenticatedURL(new MaprAuthenticator());
          try {
            conn = authenticatedURL.openConnection(url, token);
          } catch (AuthenticationException e) {
            LOG.error("Authentication failed while connecting to URL: " + url.toString());
          }
        }
        return conn;
      }
    }
    return (HttpURLConnection) url.openConnection();
  }
}
