package com.mapr.security;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.Reader;
import java.util.List;
import java.util.Properties;
import java.util.Iterator;

import org.apache.log4j.Logger;

import com.mapr.baseutils.Errno;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtilsException;
import com.mapr.baseutils.BaseUtilsHelper;
import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.cldb.proto.CLDBProto.BlacklistedAeMsg;
import com.mapr.fs.cldb.proto.CLDBProto.GetServerKeyRequest;
import com.mapr.fs.cldb.proto.CLDBProto.GetServerKeyResponse;
import com.mapr.fs.proto.Common;
import com.mapr.fs.proto.Security.CredentialsMsg;
import com.mapr.fs.proto.Security.Key;
import com.mapr.fs.proto.Security.ServerKeyType;
import com.mapr.fs.proto.Security.TicketAndKey;

/**
 * Class to allow generation of local to the code ticket
 * based on ServerKey received from CLDB for authentication of the services
 * owned by the same cluster User.
 * Also need to set ServerKey (Security.SetKey()) so it can be used
 * for authentication of clients
 * 
 *  This class is a singleton
 *
 */
public class ClusterServerTicketGeneration {
  
  private static final Logger LOG = Logger.getLogger(ClusterServerTicketGeneration.class);
  private static String maprHome = BaseUtilsHelper.getPathToMaprHome();
  private String maprUser;
  private String maprGroup;

  private static  ClusterServerTicketGeneration s_instance = new ClusterServerTicketGeneration();
  
  private ClusterServerTicketGeneration() {
    Properties daemonConfProperties = new Properties();
    String daemonConfName = maprHome + "/conf/daemon.conf";
    Reader dataReader;
    try {
      dataReader = new FileReader(daemonConfName);
      daemonConfProperties.load(dataReader);
      dataReader.close();
      maprUser = daemonConfProperties.getProperty("mapr.daemon.user");
      maprGroup = daemonConfProperties.getProperty("mapr.daemon.group");
    } catch (FileNotFoundException nfe) {
      LOG.warn("File not found: " + daemonConfName);
    } catch (IOException e) {
      LOG.warn("Exception while trying to read properties from: " + daemonConfName , e);
    } catch (Throwable t) {
      LOG.warn("Exception while trying to read properties from: " + daemonConfName, t);
    }
    
  }
  
  public static ClusterServerTicketGeneration getInstance() {
    return s_instance;
  }
  
  /*
   * Actual method to
   * 1. Authenticate to CLDB
   * 2. receive from CLDB ServerKey
   * 3. use Security.SetKey(ServerKeyType.ServerKey, ReceivedKey)
   */
  public void generateTicketAndSetServerKey() throws IOException {
    String currentClusterName = CLDBRpcCommonUtils.getInstance().getCurrentClusterName();
    if ( currentClusterName == null ) {
      throw new IOException("Current cluster name is not found"); 
    }
    generateTicketAndSetServerKey(currentClusterName);
  }

  public void generateTicketAndSetServerKey(String clusterName) throws IOException {
    String ticketPath = maprHome + "/conf/maprserverticket"; 
    if (!new File(ticketPath).exists() ) {
      // actually if this happens it is some internal issue, since if file is not there
      // it would not pass login
      throw new FileNotFoundException("Security is enabled, but userTicketFile can not be found.");
    }

    Security.SetTicketAndKeyFile(ticketPath);

    MutableInt err = new MutableInt();
    TicketAndKey ticketKey = Security.GetTicketAndKeyForCluster(ServerKeyType.CldbKey, clusterName, err);
    if ( err.GetValue() != 0 ) {
      throw new IOException("Could not get the ticket: " + err.GetValue());
    }

    // to make it in ms
    long ticketExpTime = ticketKey.getExpiryTime();
    if ( ticketExpTime  * 1000L < System.currentTimeMillis() ) {
      throw new IOException("My Ticket Expired");
    } 
    
   CredentialsMsg userCreds = ticketKey.getUserCreds();
   CredentialsMsg.Builder newCreds = CredentialsMsg.newBuilder(userCreds);
   if ( userCreds.getUserName() == null || userCreds.getUserName().isEmpty() ) {
      // need to have username filled in
      // TODO until correctly done from maprserverticket
      // get it from daemon.conf file
      if ( maprUser != null ) {
        newCreds.setUserName(maprUser);
      } else {
        LOG.warn("UserName is not known and will create issues later on");
      }
    }
    // CLDB RPC here to auth
    // should return ServerKey to create own ticket
    GetServerKeyRequest.Builder bldSK = GetServerKeyRequest.newBuilder();
    bldSK.setCreds(userCreds);
    bldSK.setSendBlacklistInfo(true);
    byte [] data = null;
    
    try {
      data = CLDBRpcCommonUtils.getInstance().sendRequest(clusterName, Common.MapRProgramId
        .CldbProgramId.getNumber(),
        CLDBProto.CLDBProg.GetServerKeyProc.getNumber(), 
        bldSK.build(), GetServerKeyResponse.class, ServerKeyType.CldbKey);
    } catch (Throwable t) {
      throw new IOException("RPC Request to get ServerKey. No data returned", t);
    }
    if ( data == null ) {
      throw new IOException("userKey is null from CLDB!");    
    }

    GetServerKeyResponse keyResp = GetServerKeyResponse.parseFrom(data);
    if ( keyResp.getStatus() != 0 ) {
      LOG.error("Request to get ServerKey failed with error: " + Errno.toString(keyResp.getStatus()));
      throw new IOException("Request to get ServerKey failed with error: " + Errno.toString(keyResp.getStatus()));
    } 
    if (! keyResp.hasServerKey() ) {
      throw new IOException("key from CLDB does not contain a server key!"); 
    }

    Key serverKey = keyResp.getServerKey(); 
    int[] gids = new int[newCreds.getGidsCount()];
    for ( int i = 0; i < newCreds.getGidsCount(); i++) {
      gids[i] = newCreds.getGids(i);
    }
    Security.SetKey(ServerKeyType.ServerKey, serverKey);
    // Set max expiry time same as original ticket
    TicketAndKey localTicketAndKey = Security.GenerateTicketAndKey(
        ServerKeyType.ServerKey, newCreds.getUserName(), 
        newCreds.getUid(), gids, ticketExpTime, 0 /*not-renewable*/, true /*isExternal*/,
        false /*canUserImpersonate*/, err);
    if ( err.GetValue() != 0 ) {
      throw new IOException("Could not generate ticket: " + err.GetValue());
    }
    Security.SetTicketAndKey(ServerKeyType.ServerKey, clusterName, localTicketAndKey);
    
    /*long[] cldbBindings = null;

    try {
      cldbBindings =  CLDBRpcCommonUtils.getInstance()
                                            .getCldbBindings(clusterName);
    } catch (CLDBRpcCommonUtilsException exception) {
      throw new IOException(exception.getMessage(), exception);
    }

    if (cldbBindings == null || cldbBindings.length == 0) {
      throw new IOException ("Cannot get Cldb bindings for cluster " + 
                             clusterName);
    }

    int err = JNISecurity.PopulateServerKeyAndTicket(cldbBindings, clusterName);
    
    if (err != 0) {
      throw new IOException("Error " +
                            err +  
                            " returned during populate server ticket and key ");
    }*/
  
    updateBlacklistInfo(keyResp);

    LOG.info("Server key was cached for cluster: " + clusterName);
    return;
  }
  
  private void updateBlacklistInfo(GetServerKeyResponse keyResp) {
    int numOfBlacklistedAes = keyResp.getBlacklistedaesCount();

    if (numOfBlacklistedAes > 0 ) {
      List<BlacklistedAeMsg> blacklistedAes = keyResp.getBlacklistedaesList();

      int[] uids = new int[numOfBlacklistedAes];
      long[] blacklistTimes = new long[numOfBlacklistedAes];
      Iterator<BlacklistedAeMsg> iterator = blacklistedAes.iterator();
      int index = 0;
      BlacklistedAeMsg blacklistedAe = null;
      while(iterator.hasNext()) {
        blacklistedAe = iterator.next();
        uids[index] = blacklistedAe.getId();
        blacklistTimes[index] = blacklistedAe.getBlacklistTime();
        index++;
      }

      Security.BlacklistAndCloseConnections(uids, blacklistTimes, true);
    }
  }
}
