package com.mapr.login.client;

import com.google.common.collect.Maps;
import com.google.protobuf.InvalidProtocolBufferException;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.JVMProperties;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils.IpPort;
import com.mapr.fs.proto.Security.TicketAndKey;
import com.mapr.fs.proto.Security.ServerKeyType;
import com.mapr.login.MapRLoginException;
import com.mapr.login.common.*;
import com.mapr.login.common.GenTicketTypeRequest.TicketType;
import com.mapr.security.JNISecurity;
import com.mapr.security.MutableInt;
import com.mapr.security.Security;
import java.lang.reflect.Method;


import org.apache.log4j.Logger;
import org.json.JSONException;

import java.io.*;

import javax.net.ssl.HttpsURLConnection;

import java.security.PrivilegedExceptionAction;
import java.security.PrivilegedActionException;
import java.security.cert.X509Certificate;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.Collections;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.security.auth.Subject;
import javax.security.auth.login.*;
import javax.xml.bind.DatatypeConverter;
import javax.security.auth.kerberos.KerberosPrincipal;
import java.security.AccessControlContext;
import java.security.AccessController;




import org.ietf.jgss.GSSContext;
import org.ietf.jgss.GSSManager;
import org.ietf.jgss.GSSName;
import org.ietf.jgss.Oid;

/**
 * Author: smarella
 */
public class MapRLoginHttpsClient implements MapRLoginClient  {
  private static Logger LOG = Logger.getLogger(MapRLoginHttpsClient.class);

  private Map<String, List<IpPort>> clustersMap = CLDBRpcCommonUtils.getInstance().getClusterMap();
  public static final String SPNEGO_OID = "1.3.6.1.5.5.2";
  public static final String NT_GSS_KRB5_PRINCIPAL = "1.2.840.113554.1.2.2.1";
  public static final String MAPR_CLIENT_KERBEROS = "MAPR_CLIENT_KERBEROS";

  private String currentClusterName =
      CLDBRpcCommonUtils.getInstance().getCurrentClusterName();

  private static boolean kerberosRuntimeAvailable = false;
  private static boolean kerberosCheckDone = false;
  private boolean checkUGI = true;

  private static final boolean windows = System.getProperty("os.name").startsWith("Windows");


  static {
    JVMProperties.init();
  }

  /*
  check for existing MapR credentials for the cluster in question. If they
  are there use them. If they are not there, try to load the ticket file.
  If that doesn't work, try obtaining via Kerberos.

  If you think about it, this logic flow is a bit like a login module
  sequence where the first step (use mapr creds) is sufficient - meaning
  you'd only proceed to kerberos if it failed.

  For now, not bothering to make this JAAS based as that would be more work
  and I'm not seeing immediate value.

   */
  public void quietAuthenticateIfNeeded() {
    quietAuthenticateIfNeeded(currentClusterName);
  }

  public void quietAuthenticateIfNeeded(String cluster) {
    try {
      authenticateIfNeeded(cluster);
    } catch (MapRLoginException e) {
      LOG.debug("Exception in authentication suppressed. May cause issues later.");
      LOG.debug(e.getMessage(), e);
    }
  }

  @Override
  public TicketAndKey authenticateIfNeeded() throws MapRLoginException {
    return authenticateIfNeeded(currentClusterName);
  }

  /**
   * @throws NullPointerException if the {@code cluster} is null
   */
  @Override
  public TicketAndKey authenticateIfNeeded(String cluster)
      throws MapRLoginException {
    LOG.debug("Entering authenticate if needed.");
    return authenticateIfNeeded(cluster, isKerberosAnOption(cluster));
  }


  private TicketAndKey authenticateIfNeeded(boolean hasKerberosOption)
      throws MapRLoginException {
    return authenticateIfNeeded(currentClusterName, hasKerberosOption);
  }

  /**
   * @throws NullPointerException if the {@code cluster} is null
   */
  private TicketAndKey authenticateIfNeeded(String cluster, boolean hasKerberosOption)
      throws MapRLoginException {
    // bail if security not enabled
    if (!isSecurityEnabled(cluster)) {
      LOG.debug("security appears to be off");
      return null;
    }

    // have we already loaded a ticket for the cluster??
    if (doesSecurityHaveGoodKey(cluster, false)) {
      LOG.debug("Already have good ticket, done");
      return getTicket(cluster);
    }

    // no good ticket. so now try to find one by loading keyfile
    LOG.debug("Try reloading the ticket file");
    File file = new File(JNISecurity.GetUserTicketAndKeyFileLocation());
    if (file.exists()) {
      int errno = Security.SetTicketAndKeyFile(file.toString());
      if (errno != 0) {
        //no good, do I error out?
        if (!hasKerberosOption) {
          throw new MapRLoginException("Unable to load ticket file '"
              + file + "', error = " + errno);
        }
      } else {
        // file good, does it have the ticket I need?
        if (doesSecurityHaveGoodKey(cluster, ! hasKerberosOption))
          return getTicket(cluster);
      }
    } else {

      // Get the server key from CLDB and generate the ticket.
      /*try {
        long[] cldbbindings = CLDBRpcCommonUtils.getInstance()
                                                .getCldbBindings(cluster);
        int err = JNISecurity.PopulateServerKeyAndTicket(cldbbindings,
                                                         cluster);
        if (err == 0)
          return getTicket(cluster);
      } catch (Exception exception) {
        LOG.debug("Exception during populating server and ticket" + 
                  exception.getMessage());
      }*/

      if (!hasKerberosOption) {
        throw new MapRLoginException("Unable to authenticate as ticket " +
                                     "is not available");
      }
    }

    // I've gotten to this point meaning there are no valid tickets and
    // kerberos is an option, so try that...
    LOG.debug("Try kerberos");
    TicketAndKey tk = getMapRCredentialsViaKerberos(cluster, null);
    //update Security object with new key
    Security.SetTicketAndKey(ServerKeyType.ServerKey, cluster, tk);

    return tk;
  }

  /**
   * Returns true if the security is enabled for default cluster
   */
  @Override
  public boolean isSecurityEnabled() throws MapRLoginException {
    return isSecurityEnabled(currentClusterName);
  }

  /**
   * Returns true if the security is enabled for specified cluster
   * @throws NullPointerException if the {@code cluster} is null
   */
  @Override
  public boolean isSecurityEnabled(String cluster) {
    return JNISecurity.IsSecurityEnabled(cluster);
  }

  private TicketAndKey getTicket(String cluster) {
    MutableInt err = new MutableInt();
    TicketAndKey tk = Security.GetTicketAndKeyForCluster(
      ServerKeyType.ServerKey, cluster, err);
    return tk;
  }

  /**
   * check to see if there is a ticket cached for the cluster and if it has
   * expired if cached and not expired, we are good.
   *
   * @param cluster
   * @return
   */
  private boolean doesSecurityHaveGoodKey(String cluster, boolean bailOnError)
      throws MapRLoginException {
    MutableInt err = new MutableInt();

    TicketAndKey tk = Security.GetTicketAndKeyForCluster(
      ServerKeyType.ServerKey, cluster, err);
    if ((tk != null) && err.GetValue() == 0) {
      // have ticket without error, is it still good?
      if (Security.IsTicketAndKeyUsable(tk)) {
        // ticket still good
        LOG.debug("found existing MapR ticket");
        return true;
      }
      if (bailOnError) {
        throw new MapRLoginException("Found ticket for cluster '"
            + cluster + "' but it has expired.");
      }
    }

    // something went wrong with getting ticket
    if (bailOnError) {
      throw new MapRLoginException("Failed to find ticket for cluster '"
          + cluster + "', error = " + err);
    } else {
      return false;
    }

  }

  private synchronized boolean isKerberosAnOption(String cluster) {
    if (! JNISecurity.IsKerberosEnabled(cluster)) {
      //no kerberos for this cluster....
      LOG.debug("Kerberos not configured for this cluster.");
      return false;
    }

    LOG.debug("Kerberos available for this cluster. Determining if kerberos authentication supported in runtime");

    //looks like this cluster supports kerberos, does our client runtime?
    //these checks are potentially expensive so we defer if we can
    if (!kerberosCheckDone) {
      kerberosCheckDone = true;
      try {
        //is there even a kerberos jaas configuration for us?
        if (Configuration.getConfiguration().getAppConfigurationEntry(MAPR_CLIENT_KERBEROS) != null) {

          LOG.debug("Found JAAS configuration MAPR_CLIENT_KERBEROS");
          //is there a valid kerberos configuration on this host?
          // this second check is very expensive
          KerberosPrincipal kn = new KerberosPrincipal("anything");
          LOG.debug("Found Kerberos runtime");
          kerberosRuntimeAvailable = true;
        } else {
          LOG.debug("No Kerberos JAAS configuration");
        }
      } catch (Exception e) {
        //failed meaning there is no kerberos configuration
        LOG.debug("Failed to detect kerberos runtime. No Kerberos usage." + e.getMessage());
      }

      // would prefer this was info level but the hadoop fs command
      //echo's info output to the display which is needless. other commands
      //send such output to a file....
      if (kerberosRuntimeAvailable) {
        LOG.debug("Kerberos is an option");
      } else {
        LOG.debug("Kerberos is not an option");
      }
    }

    return kerberosRuntimeAvailable;
  }

  @Override
  public TicketAndKey getMapRCredentialsViaPassword(String username,
      String password, Long desiredTicketDurInSecs) throws MapRLoginException {
    return getMapRCredentialsViaPassword(
      currentClusterName, username, password,
      desiredTicketDurInSecs, null /*ticketLocation*/);
  }

  /**
   * @throws NullPointerException if the {@code cluster} is null
   */
  @Override
  public TicketAndKey getMapRCredentialsViaPassword(String cluster, String username,
      String password, Long desiredTicketDurInSecs, String ticketLocation) throws MapRLoginException {

    AuthResponse response = authenticateWithMapRCluster(
      cluster,
      getPasswordAuthRequestString(username, password, desiredTicketDurInSecs),
      AuthSchemes.PASSWORD);

    return processResponse(cluster, response, ticketLocation);
  }

  @Override
  public TicketAndKey getMapRCredentialsViaKerberos(Long desiredTicketDurInSecs)
      throws MapRLoginException {
    return getMapRCredentialsViaKerberos(currentClusterName, desiredTicketDurInSecs);
  }


  @Override
  public void setCheckUGI(boolean b) {
    checkUGI=b;
  }
  /**
   * @throws NullPointerException if the {@code cluster} is null
   */
  @Override
  public TicketAndKey getMapRCredentialsViaKerberos(String cluster,
      final Long desiredTicketDurInSecs) throws MapRLoginException {
if (JNISecurity.IsReplayDetectionDisabled(cluster)) {
  LOG.info("REPLAY DETECTION DISABLED");
}

    AuthResponse response;
    try {
      //is there already a kerberos identity available??
      AccessControlContext context = AccessController.getContext();
      Subject subject = Subject.getSubject(context);
      if (subject == null || subject.getPrincipals(KerberosPrincipal.class).isEmpty()) {
        if (LOG.isDebugEnabled()) {
          if (subject == null)
            LOG.debug("No subject found");
          else LOG.debug("Subject found but no Kerberos principal in it");
         }
        if (checkUGI) {
          LOG.debug("Will attempt to check UGI object for Kerberos creds");
          try {
            //In some cases the user is authenticated via the UGI and
            //we need to pick up that identity. Not normal JAAS behavior,
            //but we have to adapt....
            //this is a potential circular dependency to behavior
            //can be disabled and we use reflection to avoid build time
            //dependency
              Class aclass = Class.forName("org.apache.hadoop.security.UserGroupInformation");
              Method m = aclass.getMethod("getCurrentUser",(Class [])  null);
              Object ugi = m.invoke(null, (Object []) null);
              m = aclass.getMethod("getSubject", (Class []) null);
              subject = (Subject) m.invoke(ugi, (Object[]) null);
          } catch (ClassNotFoundException e) {
            //not a real error
            LOG.debug("Couldn't find UGI object. Doesn't matter.");
          }
        } else {
          LOG.debug("UGI checking disabled.");
        }
        if ( subject == null || subject.getPrincipals(KerberosPrincipal.class).isEmpty()) {
          LOG.debug("No subject with Kerberos found even after (optionally) checking Hadoop UGI object");
          // Create a LoginContext with a callback handler
          LoginContext lcontext = new LoginContext(MAPR_CLIENT_KERBEROS);

          // Perform authentication
          lcontext.login();
          subject = lcontext.getSubject();
         }
        }

       // Perform action as authenticated user
       LOG.debug("Client kerberos identity: " + subject.getPrincipals());

      final String clusterName = cluster;
      response =
          Subject.doAs(subject, new PrivilegedExceptionAction<AuthResponse>() {
            public AuthResponse run() throws Exception {
              String serverPrinc = getCLDBKerberosName(clusterName);
              GSSManager manager = GSSManager.getInstance();

              /*
               * Create a GSSName out of the server's name.
               */
              LOG.debug("Attempting to connect to kerberos server that has the identity '"
                  + serverPrinc + "'");
              GSSName serverName = manager.createName(serverPrinc,
                new Oid(NT_GSS_KRB5_PRINCIPAL), new Oid(SPNEGO_OID));

              /*
               * Create a GSSContext for mutual authentication with the
               * server.
               * Note: Passing in null for the credentials asks GSS-API to
               * use the default credentials. This means that the mechanism
               * will look among the credentials stored in the current Subject
               * to find the right kind of credentials that it needs.
               */
              GSSContext context = manager.createContext(serverName,
                new Oid(SPNEGO_OID),
                null,
                GSSContext.DEFAULT_LIFETIME);

              if (JNISecurity.IsReplayDetectionDisabled(clusterName)) {
                if (LOG.isDebugEnabled()) {
                  LOG.debug("Replay detection disabled in mapr-clusters.conf");
                }
                context.requestReplayDet(false);
              }
              context.requestMutualAuth(true);  // Mutual authentication

              // obtain kerberos handshake info. Note that this assumes a single
              // pass authentication. Which is true for kerberos but not true in
              // in general.

              byte[] intoken = new byte[0];
              byte[] outtoken;
              byte[] credBytes = null;

              // intoken is ignored on the first call
              outtoken = context.initSecContext(intoken, 0, intoken.length);

              // Send a token to the server if one was generated by
              // initSecContext
              if (outtoken == null) {
                LOG.error("No kerberos identity to use.");
                throw new MapRLoginException("No kerberos identity.");
              }

              String base64 = DatatypeConverter.printBase64Binary(outtoken);
              String request = getKerberosAuthRequestString(base64, desiredTicketDurInSecs);
              String authScheme = AuthSchemes.KERBEROS;
              AuthResponse response = authenticateWithMapRCluster(clusterName, request, authScheme);
              if (response.getError() != null)
                return response;

              base64 = response.getToken();
              intoken = DatatypeConverter.parseBase64Binary(base64);
              outtoken = context.initSecContext(intoken, 0, intoken.length);

              // If the client is done with context establishment
              // then there will be no more tokens to read and we are safe
              if (!context.isEstablished()) {
                //should never happen
                LOG.error("During kerberos authentication another auth round expected. Unable to complete authentication.");
                throw new MapRLoginException("Unable to complete Kerberos authentication.");
              }
              if (LOG.isDebugEnabled()) {
                LOG.debug("Context Established! Client principal is " + context.getSrcName());
                LOG.debug("Server principal is " + context.getTargName());
              }

              return response;
            }
          });

    } catch (Exception e) {
      if (e instanceof PrivilegedActionException)
        e = ((PrivilegedActionException) e).getException();

      String error = "Failure during kerberos authentication. " + e.getMessage();
      LOG.debug(error, e);
      throw new MapRLoginException(error, e);
    }
    return processResponse(cluster, response);
  }


  @Override
  public void logOut() throws MapRLoginException {
    String ticketFilePath = JNISecurity.GetUserTicketAndKeyFileLocation();
    File ticketFile = new File(ticketFilePath);
    if (ticketFile.exists() && ticketFile.delete()) {
      LOG.info("All tickets deleted.");
    } else {
      throw new MapRLoginException("Logout failed. Unable to delete the ticket file: "
          + ticketFile.getAbsolutePath());
    }
  }

  /**
   * Logout of the specified cluster.
   * @throws NullPointerException if the {@code cluster} is null
   */
  @Override
  public void logOut(String cluster) throws MapRLoginException {
    String ticketFilePath = JNISecurity.GetUserTicketAndKeyFileLocation();
    File tempFile = getTempFile(ticketFilePath + ".tmp");
    try {
      File ticketFile = new File(ticketFilePath);
      Map<String, String> clusterToCredentials = getExistingCredsFromTicketFile(ticketFile);

      if (clusterToCredentials.containsKey(cluster)) {
        clusterToCredentials.remove(cluster);
        FileOutputStream fos = new FileOutputStream(tempFile);
        for (String clusterName : clusterToCredentials.keySet()) {
          String clusterTicket = clusterToCredentials.get(clusterName);
          fos.write(clusterName.getBytes("UTF-8"));
          fos.write(" ".getBytes("UTF-8"));
          fos.write(clusterTicket.getBytes("UTF-8"));
          fos.write("\n".getBytes("UTF-8"));
        }
        fos.close();

        renameFile(tempFile, ticketFile);
        LOG.info("Deleted the ticket for cluster '" + cluster + "' from " + ticketFilePath);
      } else {
        LOG.info("No ticket present for cluster '" + cluster + "' in " + ticketFilePath + ". Nothing to delete.");
        tempFile.delete();
      }
    } catch (Exception e) {
      String error = "Error writing mapr credentials to file. File path: " + ticketFilePath;
      LOG.error(error, e);
      throw new MapRLoginException(error, e);
    }
  }

  /**
   * Renew the ticket for default cluster.
   */
  @Override
  public TicketAndKey renew(Long desiredTicketDurInSecs)
      throws MapRLoginException {
    return renew(currentClusterName, desiredTicketDurInSecs,
                 null /*inTicketFile*/, null /*ticketLocation*/);
  }

  /**
   * Renew the ticket for the specified cluster.
   * @throws NullPointerException if the {@code cluster} is null
   */
  @Override
  public TicketAndKey renew(String cluster, Long desiredTicketDurInSecs,
                            String inTicketFile, String ticketLocation)
      throws MapRLoginException {
    MutableInt err = new MutableInt();
    if (inTicketFile != null) {
      File file = new File(inTicketFile);
      if (!file.exists())
        throw new MapRLoginException("keyfile: " + inTicketFile + " not found");
    }

    byte[] ticketBytes = getTicketBytes(cluster, inTicketFile, err);
    if (err.GetValue() != 0 || ticketBytes == null || ticketBytes.length == 0) {
      String error = "Cannot renew the ticket for cluster '" + cluster +
          "'. Error decoding the existing ticket. Error code: " + err.GetValue();
      LOG.error(error);
      throw new MapRLoginException(error);
    }

    try {
      TicketAndKey ticketAndKey = TicketAndKey.parseFrom(ticketBytes);
      RenewRequest req = new RenewRequest();
      req.setTicketAndKeyString(DatatypeConverter.printBase64Binary(ticketAndKey.toByteArray()));
      req.setTicketDurInSecs(desiredTicketDurInSecs);
      return processResponse(cluster, authenticateWithMapRCluster(
        cluster, RenewRequest.toJSON(req), AuthSchemes.RENEW), ticketLocation);
    } catch (InvalidProtocolBufferException e) {
      String error = "Cannot renew the ticket for cluster '" + cluster +
          "'. Error building the TicketAndKey object.";
      LOG.error(error, e);
      throw new MapRLoginException(error, e);
    } catch (JSONException e) {
      String error = "Cannot renew the ticket for cluster '" + cluster +
          "'. Error in JSON serialization of the TicketAndKey object.";
      LOG.error(error, e);
      throw new MapRLoginException(error, e);
    }
  }

  @Override
  public TicketAndKey generateTicket(TicketType ticketType,
                                     String targetUserName,
                                     String cluster,
                                     Long desiredTicketDurInSecs,
                                     Long renewalTicketDurInSecs,
                                     String ticketFileLocation)
                                     throws MapRLoginException {
    return genTicketType(ticketType, targetUserName, cluster,
                         desiredTicketDurInSecs,
                         renewalTicketDurInSecs,
                         ticketFileLocation);
  }

  private TicketAndKey genTicketType(TicketType ticketType,
                                     String targetUserName,
                                     String cluster,
                                     Long desiredTicketDurInSecs,
                                     Long renewalTicketDurInSecs,
                                     String ticketFileLocation)
                                     throws MapRLoginException {
    MutableInt err = new MutableInt();
    byte[] ticketBytes = getTicketBytes(cluster, null /*inTicketFile*/, err);
    if (err.GetValue() != 0 || ticketBytes == null || ticketBytes.length == 0) {
      String error = "Cannot generate service ticket for cluster '" + cluster +
          "'. Error decoding the existing ticket. Error code: " + err.GetValue();
      LOG.error(error);
      throw new MapRLoginException(error);
    }

    try {
      TicketAndKey ticketAndKey = TicketAndKey.parseFrom(ticketBytes);
      GenTicketTypeRequest req = new GenTicketTypeRequest();
      req.setTicketAndKeyString(DatatypeConverter.printBase64Binary(ticketAndKey.toByteArray()));
      req.setTicketDurInSecs(desiredTicketDurInSecs);
      req.setTicketRenewInSecs(renewalTicketDurInSecs);
      req.setTargetUserName(targetUserName);
      req.setTicketType(ticketType);

      return processResponse(cluster, authenticateWithMapRCluster(
        cluster, GenTicketTypeRequest.toJSON(req),
        AuthSchemes.GEN_TICKET_TYPE), ticketFileLocation);
    } catch (InvalidProtocolBufferException e) {
      String error = "Cannot generate service ticket for cluster '" + cluster +
          "'. Error building the TicketAndKey object.";
      LOG.error(error, e);
      throw new MapRLoginException(error, e);
    } catch (JSONException e) {
      String error = "Cannot generate service ticket for cluster '" + cluster +
          "'. Error in JSON serialization of the TicketAndKey object.";
      LOG.error(error, e);
      throw new MapRLoginException(error, e);
    } 
  }

  private byte[] getTicketBytes(String cluster, String inTicketFile,
                                MutableInt err) throws MapRLoginException {
    String ticketFilePath = (inTicketFile != null ? inTicketFile : JNISecurity.GetUserTicketAndKeyFileLocation());
    File ticketFile = new File(ticketFilePath);
    Map<String, String> clusterToCredentials = getExistingCredsFromTicketFile(ticketFile);
    if (!ticketFile.exists() || !clusterToCredentials.containsKey(cluster)) {
      String error = "Operation failed. " +
          "User has no established credentials on the cluster: " + cluster;
      LOG.error(error);
      throw new MapRLoginException(error);
    }
    
    return Security.DecodeDataFromKeyFile(
      clusterToCredentials.get(cluster).getBytes(), err);
  }

  private TicketAndKey processResponse(String cluster,
      AuthResponse response) throws MapRLoginException {
    String ticketFilePath = JNISecurity.GetUserTicketAndKeyFileLocation();
    return processResponse(cluster, response, ticketFilePath);
  }

  private TicketAndKey processResponse(String cluster,
      AuthResponse response, String ticketFilePath) throws MapRLoginException {
    if (response.getStatus() == Errno.SUCCESS) {
      if (ticketFilePath == null || ticketFilePath.isEmpty())
        ticketFilePath = JNISecurity.GetUserTicketAndKeyFileLocation();

      TicketAndKey ticketAndKey;
      ticketAndKey = extractCredentialsFromResponse(response);
      writeMapRCredentialsToFile(cluster, ticketAndKey, ticketFilePath);
      return ticketAndKey;
    } else {
      String error = "Error obtaining mapr credentials for cluster : " +
          cluster + ". Error message from cldb: " + response.getError();
      LOG.error(error);
      throw new MapRLoginException(response.getError());
    }
  }

  private AuthResponse authenticateWithMapRCluster(String cluster,
      String jsonRequest, String authScheme) throws MapRLoginException {
    if (LOG.isDebugEnabled()) {
      try {
        String requestString = null;
        if (authScheme.equals(AuthSchemes.PASSWORD)) {
            requestString = PasswordAuthRequest.fromJSON(jsonRequest).toString();
        } else if (authScheme.equals(AuthSchemes.KERBEROS)) {
            requestString = KerberosAuthRequest.fromJSON(jsonRequest).toString();
        } else if (authScheme.equals(AuthSchemes.RENEW)) {
            requestString = RenewRequest.fromJSON(jsonRequest).toString();
        } else if (authScheme.equals(AuthSchemes.GEN_TICKET_TYPE)) {
          requestString = GenTicketTypeRequest.fromJSON(jsonRequest).toString();
        }
        LOG.debug("Attempting authentication with cluster - " + cluster + ". Request - "
            + (requestString != null ? requestString : "null") + ", auth scheme: " + authScheme);
      } catch (JSONException e) {
        LOG.debug(e);
      }
    }

    if (!clustersMap.containsKey(cluster)) {
      String error = "Cluster name '" + cluster + "' is not found in " +
          CLDBRpcCommonUtils.getInstance().getPathToClustersConfFile();
      LOG.error(error);
      throw new MapRLoginException(error);
    }

    StringBuilder cldbs = new StringBuilder();
    boolean first = true;
    int cldbHttpsPort = JNISecurity.GetCldbHttpsPort(currentClusterName);

    // make a copy of cldbHosts for this cluster
    List<IpPort> cldbHosts = new ArrayList(clustersMap.get(cluster));

    // shuffle the cldb hosts for randomness
    Collections.shuffle(cldbHosts);
    
    for (IpPort cldb : cldbHosts) {
      for (String cldbAddr : cldb.getOriginalAddr()) {
        if (!first) {
          cldbs.append(", ");
        }
        first = false;
        cldbs.append(cldbAddr);
        cldbs.append(":");
        cldbs.append(cldbHttpsPort);

        HttpURLConnection cldbConn = getCLDBConnection(cldbAddr, cldbHttpsPort, authScheme);
        if (cldbConn != null) { // found a right cldb to communicate to
          AuthResponse response = parseAuthResponse(sendRequest(jsonRequest, cldbConn));
          cldbConn.disconnect();
          if (LOG.isDebugEnabled()) {
            LOG.debug("Obtained auth response " + response.toString() +
              " from cldb @ " + cldbAddr + ":" + cldbHttpsPort);
          }
          if (response.getStatus() == Errno.EROFS) {
            LOG.info("Message from cldb " + cldbAddr + ":"
                + cldbHttpsPort + " - '" + response.getError() + "'. Trying another cldb..");
            continue;
          }
          return response;
        } else {
          LOG.warn("Couldn't connect to cldb " + cldbAddr + ":"
              + cldbHttpsPort + ". Trying another cldb..");
        }
      }
    }
    String error = "Unable to connect to any of the cluster's CLDBs. CLDBs tried: "
        + cldbs.toString() + ". Please check your cluster configuration.";
    LOG.error(error);
    throw new MapRLoginException(error);
  }

  private String getCLDBKerberosName(String clustername) {
    String prin = JNISecurity.GetCldbPrincipal(clustername);
    if (prin.trim().isEmpty()) {
      prin = "mapr/" + clustername;
    }
    return prin;
  }

  private AuthResponse parseAuthResponse(String response)
      throws MapRLoginException {
    try {
      return AuthResponse.fromJSON(response);
    } catch (JSONException jsone) {
      String error = "JSON parse error while parsing auth response.";
      LOG.error(error, jsone);
      throw new MapRLoginException(error, jsone);
    }
  }

  private TicketAndKey extractCredentialsFromResponse(AuthResponse response)
      throws MapRLoginException {
    try {
      return TicketAndKey.parseFrom(
        DatatypeConverter.parseBase64Binary(response.getTicketAndKeyString()));
    } catch (InvalidProtocolBufferException e) {
      String error = "Error parsing base64 encoded ticket and key string into proto object";
      LOG.error(error);
      throw new MapRLoginException(error, e);
    }
  }

  private void writeMapRCredentialsToFile(String cluster,
                                          TicketAndKey creds,
                                          String ticketFilePath)
                                          throws MapRLoginException {
    MutableInt err = new MutableInt();
    byte[] encodedTicket = Security.EncodeDataForWritingToKeyFile(creds.toByteArray(), err);

    if (encodedTicket == null) {
      String error = "Failed to encode ticket and key. Error: " + err.GetValue();
      LOG.error(error);
      throw new MapRLoginException(error);
    }

    File tempFile = getTempFile(ticketFilePath + ".tmp");

    try {
      Map<String, String> clusterToCredentials = Maps.newLinkedHashMap();
      File ticketFile = new File(ticketFilePath);
      if (ticketFile.exists()) {
        clusterToCredentials = getExistingCredsFromTicketFile(ticketFile);
      }

      // replaces any existing ticket with the newer ticket
      clusterToCredentials.put(cluster, new String(encodedTicket, "UTF-8"));

      FileOutputStream fos = new FileOutputStream(tempFile);
      for (String clusterName : clusterToCredentials.keySet()) {
        String clusterTicket = clusterToCredentials.get(clusterName);
        fos.write(clusterName.getBytes("UTF-8"));
        fos.write(" ".getBytes("UTF-8"));
        fos.write(clusterTicket.getBytes("UTF-8"));
        fos.write("\n".getBytes("UTF-8"));
      }
      fos.close();

      renameFile(tempFile, ticketFile);

      if (LOG.isDebugEnabled()) {
        LOG.debug(
          "\n************* \n"
              + "Saved mapr credentials to file: " + ticketFilePath
              + ". \nUser credentials: \n"
              + Security.UserCredsToString(cluster, creds.getUserCreds())
              + "\n************* \n");

      }
      String message = "MapR credentials of user '" + creds.getUserCreds().getUserName() +
          "' for cluster '" + cluster + "' are written to '" + ticketFilePath + "'";
      LOG.info(message);
      if (System.out != null) {
        System.out.println(message);
      }
    } catch (Exception e) {
      String error = "Error writing mapr credentials to file. File path: " + ticketFilePath;
      LOG.error(error, e);
      throw new MapRLoginException(error, e);
    }
  }


  /**
    Rename the file atomically if at all possible. This include atomically removing the
    the old ticket file if possible. This is possible in Java 7 and is possible on linux
    with Java 6. But Windows Java 6 makes this impossible, hence this horribly ugly hack.
    Once we drop support for Java 6, we can just use the java.nio.File.move API and be
    done with it.
  */
  private void renameFile(File tempFile, File ticketFile) throws MapRLoginException {
    boolean result = tempFile.renameTo(ticketFile);
    if (result)
      return;

    //ok, it didn't work. If we are on Windows, try deleting the target first
    if (windows) {
      LOG.debug("The rename failed, since I'm on windows, try deleting the target file first.");
      ticketFile.delete();
      result = tempFile.renameTo(ticketFile);
      if (result)
        return;
     }

     String error = "Unable to rename " + tempFile.getAbsolutePath()
         + " to " + ticketFile.getAbsolutePath();
     LOG.error(error);
     throw new MapRLoginException(error);
  }

  private File getTempFile(String tempFilePath) throws MapRLoginException {
    File tempFile = new File(tempFilePath);
    int maxRetries = 10;
    int retries = 0;
    long retryInterval = 500; // ms
    try {
      while (!tempFile.createNewFile()) {
        LOG.info(tempFilePath + " already exists. will retry again in " + retryInterval + " ms");
        try {
          Thread.sleep(retryInterval);
        } catch (InterruptedException e) {
          LOG.error(e);
          throw new MapRLoginException(e);
        }
        retries++;

        if (retries >= maxRetries) {
          LOG.info(tempFilePath + " is present even after " + maxRetries + " retries. Deleting it.");
          if (!tempFile.delete()) {
            String error = "Unable to delete the temporary ticket file: " + tempFilePath;
            LOG.error(error);
            throw new MapRLoginException(error);
          }
        }
      }
    } catch (IOException e) {
      String message = "Unable to create " + tempFilePath;
      LOG.error(message, e);
      throw new MapRLoginException(message, e);
    }
    // TODO : Put JNI method to create the file with perm 600
    // Following is indirect mechanism in java to set the perm
    // to 600
    tempFile.setExecutable(false, false);
    tempFile.setWritable(false, false);
    tempFile.setReadable(false, false);
    tempFile.setWritable(true, true);
    tempFile.setReadable(true, true);
    return tempFile;
  }

  private Map<String, String> getExistingCredsFromTicketFile(File file)
      throws MapRLoginException {
    Map<String, String> clusterToCredsMap = Maps.newLinkedHashMap(); // preserve the order
    if (file.exists()) {
      try {
        BufferedReader reader = new BufferedReader(
          new InputStreamReader(new FileInputStream(file), "UTF-8"));
        String line;
        while ((line = reader.readLine())  != null) {
          String[] parts = line.split(" ");
          clusterToCredsMap.put(parts[0], parts[1]);
        }
        reader.close();
      } catch (IOException e) {
        String error = "Error reading from " + file.getAbsolutePath();
        LOG.error(error, e);
        throw new MapRLoginException(error, e);
      }
    }
    return clusterToCredsMap;
  }

  private String getPasswordAuthRequestString(String username, String password,
      Long desiredTicketDurInSecs) throws MapRLoginException {
    PasswordAuthRequest par = new PasswordAuthRequest();
    par.setUserName(username);
    par.setPassWord(password);
    par.setTicketDurInSecs(desiredTicketDurInSecs);
    try {
      return PasswordAuthRequest.toJSON(par);
    } catch (JSONException jsone) {
      String error = "JSON parse error";
      LOG.error(error, jsone);
      throw new MapRLoginException(error, jsone);
    }
  }

  /**
   * portions of this are based upon the example from Sun here:
   *   http://docs.oracle.com/javase/7/docs/technotes/guides/security/jgss/lab/part5.html#SPNEGO
   *
   */
  private String getKerberosAuthRequestString(String kerberosToken,
      Long desiredTicketDurInSecs) throws MapRLoginException {

    KerberosAuthRequest kar = new KerberosAuthRequest();
    kar.setToken(kerberosToken);
    kar.setTicketDurInSecs(desiredTicketDurInSecs);
    try {
      return KerberosAuthRequest.toJSON(kar);
    } catch (JSONException jsone) {
      String error = "JSON parse error";
      LOG.error(error, jsone);
      throw new MapRLoginException(error, jsone);
    }
  }

  private String sendRequest(String request, HttpURLConnection cldbConn)
      throws MapRLoginException {
    try {
      OutputStream outputStream = cldbConn.getOutputStream();
      outputStream.write(request.getBytes());
      outputStream.close();

      BufferedReader input = new BufferedReader(new InputStreamReader(cldbConn.getInputStream()));
      StringBuilder sb = new StringBuilder();
      String line;
      while ((line = input.readLine()) != null) {
        sb.append(line);
      }
      input.close();

      return sb.toString();
    } catch (IOException e) {
      String error = "Error sending request data via http output stream";
      LOG.error(error, e);
      throw new MapRLoginException(error, e);
    }
  }

  private HttpURLConnection getCLDBConnection(String cldbAddr, int cldbPort, String authScheme) {
    String authUrl = getAuthURL(cldbAddr, cldbPort, authScheme);
    HttpsURLConnection conn = null;
    try {
      conn = (HttpsURLConnection) (new URL(authUrl)).openConnection();
      conn.setHostnameVerifier(new MyVerifier());
      conn.setRequestMethod("POST");
      conn.setDoInput(true);
      conn.setDoOutput(true);
      conn.connect();
      conn.getOutputStream();
      return conn;
    } catch (IOException e) {
      String error = "Unable to open connection to cldb at " + authUrl;
      if (conn != null && conn.getErrorStream() != null) {
        try {
          StringBuilder sb = new StringBuilder();
          sb.append(error);
          sb.append(". Details: ");

          BufferedReader input = new BufferedReader(new InputStreamReader(conn.getErrorStream()));
          String line;
          while ((line = input.readLine()) != null) {
            sb.append(line);
          }
          input.close();

          LOG.warn(sb.toString());
        } catch (IOException ioe) {
          LOG.warn(error, e);
        }
      } else {
        LOG.warn(error, e);
      }
      return null;
    }
  }

  private String getAuthURL(String cldbAddr, int cldbHttpsPort, String authScheme) {
    StringBuilder builder = new StringBuilder();
    builder.append("https://");
    builder.append(cldbAddr);
    builder.append(":");
    builder.append(cldbHttpsPort);
    builder.append(authScheme);
    return builder.toString();
  }

  class MyVerifier implements javax.net.ssl.HostnameVerifier {

    /**
      This method should only be called by Java when the certificate has been verified
      as having been signed by a trusted signer BUT the hostname verification check
      has failed.

      Our goal is to allow self-signed certs. In the future we may add more complex
      logic to accept other certs based on subject, issuer, and serial.
     */
    @Override
    public boolean verify(String hostname, javax.net.ssl.SSLSession session) {

      try {
        X509Certificate chain[] = (X509Certificate []) session.getPeerCertificates();

	if (LOG.isDebugEnabled()) {
          StringBuffer chainStr = new StringBuffer();
          for (int i =0; i < chain.length;i++) {
            chainStr.append("  Peer cert #" + i + ": " + chain[i].getSubjectX500Principal().getName() + ", signer = " + chain[i].getIssuerX500Principal().getName() + "\n");
          }
	  LOG.debug("Certificate chain: " + chainStr);
	}

        //always accept self signed certs since someone went to the trouble to put it into
        //our trust store. Take note that we assume the chain has a length of zero as one
        //would expect for self signed.
        String subject = chain[0].getSubjectX500Principal().getName();
        String issuer = chain[0].getIssuerX500Principal().getName();
        if ( LOG.isDebugEnabled() ) {
          LOG.debug("Java default verification failed for cert with subject " + subject + ", custom verifier now checking.");
        }

        if ( (chain.length == 1) && issuer.equals(subject)) {
          LOG.debug("Accepting self signed certificate automatically.");
          return true;
        }

        LOG.warn("Peer certificate has failed verification. SubjectDN is: " + subject);
        return false;

      } catch (javax.net.ssl.SSLPeerUnverifiedException e) {
        LOG.error("Unexpected SSL handshake issue", e);
        return false;
      }
    }
  }

}
