package com.mapr.login;

import com.mapr.fs.proto.Security.ServerKeyType;
import com.mapr.fs.proto.Security.TicketAndKey;
import com.mapr.login.client.MapRLoginClient;
import com.mapr.login.client.MapRLoginHttpsClient;
import com.mapr.login.common.GenTicketTypeRequest.TicketType;
import com.mapr.security.JNISecurity;
import com.mapr.security.MutableInt;
import com.mapr.security.Security;
import com.mapr.security.UnixUserGroupHelper;
// import com.mapr.security.UserInformation;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;

import org.apache.log4j.Logger;

import java.io.BufferedReader;
import java.io.Console;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStreamReader;
import java.io.IOException;
import java.util.Arrays;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.HashSet;

/**
 * Author: smarella
 */
public class MapRLogin {
  private static final Logger LOG = Logger.getLogger(MapRLogin.class);

  private static final String PASSWORD = "password";
  private static final String PASSWORD_HELP = "authenticate to a mapr cluster using a valid password";

  private static final String KERBEROS = "kerberos";
  private static final String KERBEROS_HELP = "authenticate to a mapr cluster using kerberos";

  private static final String RENEW = "renew";
  private static final String RENEW_HELP = "renew existing credentials for a mapr cluster";

  private static final String PRINT = "print";
  private static final String PRINT_HELP = "print information on your existing credentials";

  private static final String AUTHTEST = "authtest";
  private static final String AUTHTEST_HELP = "test authentication as a generic client";

  private static final String GENERATE_TICKET = "generateticket";
  private static final String GENERATE_TICKET_HELP = "generate a ticket of particular type";

  private static final String END1 = "end";
  private static final String END2 = "logout";
  private static final String END_HELP = "logout of cluster";

  // 24 hours
  private static final long MAPRLOGIN_TICKET_GENERATION_INTERVAL = 24 * 60 * 60;

  static
  {
     com.mapr.fs.ShimLoader.load();
  }

  private static void printDurationUsage() {
    StringBuilder builder = new StringBuilder();
    builder.append("-duration|renewal [Days:]Hours:Minutes \n");
    builder.append("OR -duration|renewal seconds");
    System.out.println(builder.toString());
  }

  private static void printGenerateTicketHelp() {
    UnixUserGroupHelper helper = new UnixUserGroupHelper();
    String userName = helper.getLoggedinUsername();

    StringBuilder builder = new StringBuilder();

    builder.append(GENERATE_TICKET);

    builder.append("\n");
    builder.append("\t");
    builder.append("-type service/crosscluster/servicewithimpersonation");

    builder.append("\n");
    builder.append("\t");
    builder.append("[ -cluster mapr cluster name ]");

    builder.append("\n");
    builder.append("\t");
    builder.append("[ -user UNIX user name of service identity. default: '");
    builder.append(userName);
    builder.append("' ]");

    builder.append("\n");
    builder.append("\t");
    builder.append("-out ticket location");

    builder.append("\n");
    builder.append("\t");
    builder.append("[ -duration [Days:]Hours:Minutes OR -duration Seconds." +
                   "default: cluster's ticket duration setting ]");

    builder.append("\n");
    builder.append("\t");
    builder.append("[ -renewal [Days:]Hours:Minutes OR -duration Seconds." +
                   "default: cluster's ticket duration setting ]");

    System.out.println(builder.toString());
    LOG.info(builder.toString());
  }

  private static void printPasswordHelp() {
    UnixUserGroupHelper helper = new UnixUserGroupHelper();
    String userName = helper.getLoggedinUsername();

    StringBuilder builder = new StringBuilder();

    builder.append(PASSWORD);

    builder.append("\n");
    builder.append("\t");
    builder.append("[ -cluster mapr cluster name ]");

    builder.append("\n");
    builder.append("\t");
    builder.append("[ -user UNIX user name on MapR cluster. default: '");
    builder.append(userName);
    builder.append("' ]");

    builder.append("\n");
    builder.append("\t");
    builder.append("[ -duration desired ticket duration in seconds. default: cluster's ticket duration setting ]");

    builder.append("\n");
    builder.append("\t");
    builder.append("[ -out ticket location ]");

    System.out.println(builder.toString());
    LOG.info(builder.toString());
  }

  private static void printKerberosHelp() {
    StringBuilder builder = new StringBuilder();

    builder.append(KERBEROS);

    builder.append("\n");
    builder.append("\t");
    builder.append("[ -cluster mapr cluster name ]");

    builder.append("\n");
    builder.append("\t");
    builder.append("[ -duration desired ticket duration in seconds. default: cluster's ticket duration setting ]");

    System.out.println(builder.toString());
    LOG.info(builder.toString());
  }

  private static void printAuthTestHelp() {
    StringBuilder builder = new StringBuilder();
    builder.append(AUTHTEST);

    builder.append("\n");
    builder.append("\t");
    builder.append("[ -cluster mapr cluster name ]");

    System.out.println(builder.toString());
  }

  private static void printEndHelp() {
    StringBuilder builder = new StringBuilder();
    builder.append(END1);
    builder.append(" / " );
    builder.append(END2);

    builder.append("\n");
    builder.append("\t");
    builder.append("[ -cluster mapr cluster name ]");

    System.out.println(builder.toString());
  }

  private static void printPrintHelp() {
    StringBuilder builder = new StringBuilder();
    builder.append(PRINT);

    builder.append("\n");
    builder.append("\t");
    builder.append("[ -ticketfile location of ticket file]");

    System.out.println(builder.toString());
  }

  private static void printRenewHelp() {
    StringBuilder builder = new StringBuilder();
    builder.append(RENEW);

    builder.append("\n");
    builder.append("\t");
    builder.append("[ -cluster mapr cluster name ]");

    builder.append("\n");
    builder.append("\t");
    builder.append("[ -duration desired ticket renew duration in seconds. default: cluster's ticket renew duration setting ]");

    builder.append("\n");
    builder.append("\t");
    builder.append("[ -ticketfile input ticket file ]");

    builder.append("\n");
    builder.append("\t");
    builder.append("[ -out ticket location ]");

    System.out.println(builder.toString());
  }

  private static void printUsage() {
    StringBuilder builder = new StringBuilder();
    builder.append("List of commands: \n");

    builder.append("\n");
    builder.append("\t");
    builder.append(PASSWORD);
    builder.append("\t");
    builder.append(PASSWORD_HELP);

    builder.append("\n");
    builder.append("\t");
    builder.append(KERBEROS);
    builder.append("\t");
    builder.append(KERBEROS_HELP);

    builder.append("\n");
    builder.append("\t");
    builder.append(PRINT + "   ");
    builder.append("\t");
    builder.append(PRINT_HELP);

    builder.append("\n");
    builder.append("\t");
    builder.append(AUTHTEST + "   ");
    builder.append("\t");
    builder.append(AUTHTEST_HELP);

    builder.append("\n");
    builder.append("\t");
    builder.append(END1 + " / " + END2);
    builder.append("\t");
    builder.append(END_HELP);

    builder.append("\n");
    builder.append("\t");
    builder.append(RENEW);
    builder.append("\t");
    builder.append(RENEW_HELP);

    builder.append("\n");
    builder.append("\t");
    builder.append(GENERATE_TICKET);
    builder.append("\t");
    builder.append(GENERATE_TICKET_HELP);

    System.out.println(builder.toString());
    LOG.info(builder.toString());
  }


  private static void printHelp(String command) {
    if (command.equals(PRINT)) {
      printPrintHelp();
    } else if (command.equals(AUTHTEST)) {
      printAuthTestHelp();
    } else if (command.equals(KERBEROS)) {
      printKerberosHelp();
    } else if (command.equals(PASSWORD)) {
      printPasswordHelp();
    } else if (command.equals(END1) || command.equals(END2)) {
      printEndHelp();
    } else if (command.equals(RENEW)) {
      printRenewHelp();
    } else if (command.equals(GENERATE_TICKET)) {
      printGenerateTicketHelp();
    } else {
      printUsage();
    }
  }

  private static void printTicket(String cluster, TicketAndKey tk) throws MapRLoginException {
    long expiration = tk.getExpiryTime();
    int uid = tk.getUserCreds().getUid();
    List<Integer> gids = tk.getUserCreds().getGidsList();
    String user = tk.getUserCreds().getUserName();
    boolean canUserImpersonate = (tk.hasCanUserImpersonate() ?
                                  tk.getCanUserImpersonate() : false);

    System.out.print(cluster + ": ");
    String creationTime = new Date(tk.getCreationTimeSec() * 1000L).toString();
    String renewalMsg;
    if (tk.getMaxRenewalDurationSec() == 0) {
      renewalMsg = "Not renewable";
    } else {
      renewalMsg = "RenewalTill = '" + new Date((tk.getCreationTimeSec() + tk.getMaxRenewalDurationSec()) * 1000L).toString() + "'";
    }

    System.out.print("user = " + user +", created = '" + creationTime +"', expires = '" + new Date(expiration * 1000L) + "', " + renewalMsg + ", uid = " + uid + ", gids = ");
    Iterator<Integer> iter = gids.iterator();
    while (iter.hasNext()) {
      Integer gid = iter.next();
      System.out.print(gid);
      if (iter.hasNext())
        System.out.print(", ");
    }
    System.out.print(", CanImpersonate = " + canUserImpersonate);
    System.out.println();
  }

  /**
   * Open user's ticket file and load it into Security object. Also loop through
   * file's contents directly to find the cluster names so I can ask security object
   * for the key for each cluster.
   *
   * @throws Exception
   */
  private static void handlePrint(String inTicketFile, TicketType type) throws Exception {
    MutableInt err = new MutableInt();
    String loc = (inTicketFile != null ? inTicketFile : JNISecurity.GetUserTicketAndKeyFileLocation());
    System.out.println("Opening keyfile " + loc);
    File file = new File(loc);
    if (! file.exists()) {
      throw new MapRLoginException("keyfile not found");
    }

    int errno = Security.SetTicketAndKeyFile(loc);
    if (errno != 0) {
      throw new MapRLoginException("Problem reading keyfile, error = " + errno);
    }

    HashSet <String> clusterSet = new HashSet <String>();
    FileInputStream inraw = new FileInputStream(file);
    BufferedReader in = new BufferedReader(new InputStreamReader(inraw, "UTF-8"));
    String line = in.readLine();
    if (line == null)
      throw new MapRLoginException("Empty ticket file");

    while (line != null) {
      String cluster = line.split(" ")[0];
      clusterSet.add(cluster);
      line = in.readLine();
    }

    Iterator iterator = clusterSet.iterator();
    while (iterator.hasNext()){
      String cluster = (String)iterator.next();

      for (int i = ServerKeyType.CldbKey.getNumber();
           i < ServerKeyType.ServerKeyTypeMax.getNumber(); ++i) {
        ServerKeyType sKType = ServerKeyType.valueOf(i);
        TicketAndKey tk = Security.GetTicketAndKeyForCluster(sKType, cluster, err);
        if (tk == null)
          continue;

        printTicket(cluster, tk);
      }
    }
  }

  private static long getDesiredDurationInSecs(String desiredDuration) {
    if (desiredDuration == null || desiredDuration.isEmpty()) {
      System.out.println("Invalid input for -duration." +
                         " Please enter a numeric value");
      System.exit(1);
    }

    String[] times = desiredDuration.split(":");
    long durationInSecs = 0;
    long days = 0;
    long hours = 0;
    long minutes = 0;

    try {
      switch (times.length) {
        case 1 :
          durationInSecs = Long.parseLong(times[0]);
          break;
        case 2 :
          /* Hours:Minutes */
          hours = Long.parseLong(times[0]);
          minutes = Long.parseLong(times[1]);

          if (hours < 0) {
            System.out.println("Invalid hours specified");
            printDurationUsage();
            System.exit(1);
          }

          if (minutes < 0 || minutes > 60) {
            System.out.println("Invalid minutes specified");
            printDurationUsage();
            System.exit(1);
          }

          durationInSecs = 60 * 60 * hours +
                           60 * minutes;
          break;
        case 3 :
          /* Days:Hours:Minutes */
          days = Long.parseLong(times[0]);
          hours = Long.parseLong(times[1]);
          minutes = Long.parseLong(times[2]);

          if (days < 0) {
            System.out.println("Invalid days specified");
            printDurationUsage();
            System.exit(1);
          }

          if (hours < 0) {
            System.out.println("Invalid hours specified");
            printDurationUsage();
            System.exit(1);
          }

          if (minutes < 0 || minutes > 60) {
            System.out.println("Invalid minutes specified");
            printDurationUsage();
            System.exit(1);
          }

          durationInSecs = 24 * 60 * 60 * days +
                           60 * 60 * hours +
                           60 * minutes;
          break;
        default :
          System.out.println("Incorrect usage of duration.");
          printDurationUsage();
          System.exit(1);
      }
    } catch (NumberFormatException nfe) {
      System.out.println("Invalid input for duration " +
                         desiredDuration +
                         " Please enter a numeric value");
      printDurationUsage();
      System.exit(1);
    }

    if (durationInSecs <= 0) {
      System.out.println("Invalid time specified");
      printDurationUsage();
      System.exit(1);
    }

    return durationInSecs;
  }

  public static void execute(String[] args) throws Exception{
    String command = args[0];

    //defaults that may be overridden
    String clusterName =
        CLDBRpcCommonUtils.getInstance().getCurrentClusterName();
    boolean clusterNameSpecified = false;
    String user = new UnixUserGroupHelper().getLoggedinUsername();
    Long desiredDurationInSecs = null;
    Long renewalDurationInSecs = null;
    String ticketLocation = null;
    TicketType type = null;
    boolean checkTicketValidity = false;
    String inTicketFile = null;

    for (int i = 1; i < args.length; i++) {
      if (args[i].equals("-cluster") && i != args.length - 1) {
        clusterName = args[i+1];
        clusterNameSpecified = true;
        i++;
      } else if (args[i].equals("-out") && i != args.length - 1) {
        ticketLocation = args[i+1];
        i++;
      } else if (args[i].equals("-user") && i != args.length - 1) {
        user = args[i+1];
        i++;
      } else if (args[i].equals("-type") && i != args.length - 1) {
        type = TicketType.valueOf(args[i+1].toUpperCase());
        i++;
      } else if (args[i].equals("-duration") && i != args.length - 1) {
        try {
          desiredDurationInSecs = getDesiredDurationInSecs(args[i+1]);
          i++;
        } catch (NumberFormatException nfe) {
          System.out.println("Invalid input for -duration. Please enter a numeric value.");
          System.exit(1);
        }
      } else if (args[i].equals("-renewal") && i != args.length - 1) {
        try {
          renewalDurationInSecs = getDesiredDurationInSecs(args[i+1]);
          i++;
        } catch (NumberFormatException nfe) {
          System.out.println("Invalid input for -renewal " +
                             args[i+1] +
                             ". Please enter a numeric value.");
          System.exit(1);
        }
      } else if (args[i].equals("-checkvalidity")) {
        checkTicketValidity = true;
      } else if (args[i].equals("-ticketfile")) {
        if ((i + 1) < args.length) {
          inTicketFile = args[i + 1];
        } else {
          printHelp(command);
          System.exit(1);
        }
        i++;
      } else {
        printHelp(command);
        System.exit(1);
      }
    }

    if (command.equals(PRINT)) {
      handlePrint(inTicketFile, type);
      return;
    }

    if ( !JNISecurity.IsSecurityEnabled(clusterName)) {
      String error = "Security is not enabled on the cluster: " + clusterName
          + ". Running maprlogin will not login. If you think security should be enabled - " +
          " please check your cluster configuration.";
      throw new IOException(error);
    }

    MapRLoginClient client = new MapRLoginHttpsClient();

    if ( command.equals(KERBEROS) ) {
      String kerberosConfigFilePath = System.getProperty("java.security.auth.login.config");
      if (kerberosConfigFilePath == null || kerberosConfigFilePath.isEmpty()) {
        String error = "Kerberos configuration file not specified. " +
            "Please specify it via -Djava.security.auth.login.config.";
        LOG.error(error);
        throw new IOException(error);
      } else {
        File file = new File(kerberosConfigFilePath);
        if (!file.exists()) {
          String error = "Kerberos configuration does not exist: '" + kerberosConfigFilePath +
              "'. Cannot proceed without it.";
          LOG.error(error);
          throw new IOException(error);
        }
      }
      client.setCheckUGI(false);
      client.getMapRCredentialsViaKerberos(clusterName, desiredDurationInSecs);
    } else if (command.equals(PASSWORD)) {
      if (checkTicketValidity) {
        MutableInt err = new MutableInt();
        TicketAndKey tk = Security.GetTicketAndKeyForCluster(
            ServerKeyType.ServerKey, clusterName, err);
        // if err != 0, it means ticket does not exist or any other error
        // we can proceed with the password prompt.
        if (err.GetValue() == 0) {
          /*
           * Conditions that needs to be checked to skip generation
           *
           * 1. TicketExpiryTime > currentTime
           *
           * 2. TicketCreationTime > currentTime -
           * MAPRLOGIN_TICKET_GENERATION_TIME
           */
          long currentTime = System.currentTimeMillis() / 1000;
          long prevTicketGenIntervalTime = currentTime -
              MAPRLOGIN_TICKET_GENERATION_INTERVAL;
          if (tk.getExpiryTime() > currentTime &&
              tk.getCreationTimeSec() > prevTicketGenIntervalTime) {
            LOG.info("A valid ticket is already present.");
            return;
          }
        }
      }
      String password;
      String passwordPrompt = "Password for user '" + user + "' at cluster '" + clusterName + "': ";
      Console cons;
      char[] passwd;
      if ((cons = System.console()) != null && (passwd = cons.readPassword("[%s] ", passwordPrompt)) != null) {
        password = new String(passwd);
        Arrays.fill(passwd, ' ');
        client.getMapRCredentialsViaPassword(clusterName, user, password,
                                             desiredDurationInSecs,
                                             ticketLocation);
      } else if (System.in.available() > 0) {
        int numAvailable = System.in.available();
        byte[] bytes = new byte[numAvailable];
        System.in.read(bytes);
        password = new String(bytes);
        password = password.trim();
        client.getMapRCredentialsViaPassword(clusterName, user, password,
                                             desiredDurationInSecs,
                                             ticketLocation);
      } else {
        String error = "Error obtaining user password from console";
        LOG.error(error);
        throw new IOException(error);
      }
    } else if (command.equals(AUTHTEST)) {
      System.out.println("Attempting to pick up default credentials for cluster " + clusterName);
      client.authenticateIfNeeded(clusterName);
      System.out.println("Successfully obtained credentials.");
    } else if (command.equals(END1) || command.equals(END2)) {
      if (clusterNameSpecified)
        client.logOut(clusterName);
      else client.logOut();
    } else if (command.equals(RENEW)) {
      client.renew(clusterName, desiredDurationInSecs,
                   inTicketFile, ticketLocation);
    } else if (command.equals(GENERATE_TICKET)) {
      if (type == null) {
        System.out.println("Specify the type of the ticket");
        printGenerateTicketHelp();
        System.exit(1);
      }
      if (ticketLocation == null || ticketLocation.isEmpty()) {
        System.out.println("Specify the  of location of the ticket");
        printGenerateTicketHelp();
        System.exit(1);
      }

      if (renewalDurationInSecs == null)
        renewalDurationInSecs = desiredDurationInSecs;

      if (desiredDurationInSecs != null &&
          desiredDurationInSecs > renewalDurationInSecs) {
        String error = "Duration value cannot be more than renewal value";
        throw new IOException(error);
      }

      client.generateTicket(type, user, clusterName,
                            desiredDurationInSecs,
                            renewalDurationInSecs,
                            ticketLocation);
    } else {
      printUsage();
      System.exit(1);
    }
  }

  public static void main(String args[]) {

    if (args == null || args.length < 1) {
      printUsage();
      return;
    }
    try {
      execute(args);
    }  catch (Exception e) {
      LOG.error("Login exception", e);
      System.err.println(e.getMessage());
      System.exit(1);
    }
  }
}
