package com.mapr.cli.table;


import com.mapr.cli.MapRCliUtil;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.mapr.baseutils.Errno;
import com.mapr.cliframework.base.*;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.fs.MapRFileSystem;
import com.mapr.fs.MapRTabletScanner;
import com.mapr.cli.MapRCliUtil;
import com.mapr.fs.proto.Dbserver.TabletDesc;
import com.mapr.fs.proto.Dbserver.TabletStatResponse;
import org.apache.hadoop.fs.Path;
import org.apache.log4j.Logger;

import java.io.IOException;
import java.util.List;
import java.util.Map;

import java.util.concurrent.*;

public class TabletStats {

  private static final Logger LOG = Logger.getLogger(TabletStats.class);


  // Santosh: Making this static so that we restrict the number of simultaneous TabletStat RPCs
  // to utmost the size of this threadpool.
  private static ExecutorService tabletStatRpcService = Executors.newFixedThreadPool(10);

  private Map<TabletDesc, Future<TabletStatResponse>> futureResponses; // Holds all current future responses

  private String userId;
  private String path;


  public TabletStats (String path, String userId) {
    this.userId = userId;
    this.path = path;
  }
  /**
   * Method to get a list of all tablets
   * @param out Output errors
   * @param start The tablet record to start from
   * @param limit The total amount of tablet records to retrieve. If 0, then gets all
   * @return A list of all tablet items
   * @throws CLIProcessingException
   */
  public List<TabletDesc> getTablets(OutputHierarchy out, int start, int limit) throws CLIProcessingException{
    List<TabletDesc> tablets = Lists.newArrayList();

    RecentTablesListManager manager = RecentTablesListManagers.getRecentTablesListManagerForUser(userId);
    final MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
    try {
      MapRTabletScanner scanner = mfs.getTabletScanner(new Path(path));

      if (scanner.hasTotal()) {
        out.setTotal(scanner.getTotal());
      }

      TabletDesc nextTablet;
      // Iterate through each tablet
      while ((nextTablet = scanner.next()) != null) {
        if (start > 0) {
          start--;
        } else if (start == 0 && limit > 0) {
          tablets.add(nextTablet);
          limit--;
        } else if (limit == 0) {
          break;
        }
      }
    } catch (IOException e) {
      out.addError(new OutputError(Errno.EOPFAILED, "Failed to get list of regions for table: "
          + path + ". Reason: " + e.getMessage()));

      manager.deleteIfNotExist(path, mfs);
      return null;
    }
    // Set all future responses
    futureResponses = Maps.newHashMap();
    for (final TabletDesc tablet : tablets) {
      futureResponses.put(tablet, tabletStatRpcService.submit(new Callable<TabletStatResponse>() {
        @Override
        public TabletStatResponse call() {
          try {
            if (LOG.isDebugEnabled()) {
              LOG.debug("Fetching tablet stats for fid: " + MapRCliUtil.getFidAsString(tablet.getFid()));
            }
            return mfs.getTabletStat(new Path(path), tablet.getFid());
          } catch (Exception ioe) {
            LOG.error("Error fetching tablet stats for fid: " + MapRCliUtil.getFidAsString(tablet.getFid()), ioe);
          }
          return null;
        }
      }));
    }

    return tablets;

  }

  // Get future response from method
  public TabletStatResponse getTabletStatResponse(TabletDesc tablet) throws InterruptedException, ExecutionException {
    return futureResponses.get(tablet).get();
  }



}
