package com.mapr.cli.schedulepolicy.commands;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.google.common.collect.ImmutableMap;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.baseutils.Errno;
import com.mapr.cli.MapRCliUtil;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.IntegerInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.cldb.proto.CLDBProto.Policy;
import com.mapr.fs.cldb.proto.CLDBProto.SchedulePolicyProcRequest;
import com.mapr.fs.cldb.proto.CLDBProto.SchedulePolicyProcResponse;
import com.mapr.fs.cldb.proto.CLDBProto.SchedulePolicyProcRequest.SchedulePolicyOP;
import com.mapr.fs.proto.Common;
import com.mapr.security.MaprSecurityException;

/**
 * Command to remove schedule - CLDB should check whether this record can be removed 
 * meaning if no volume is using this schedule
 * @author yufeldman
 *
 */
public class SchedulePolicyRemoveCommand extends CLIBaseClass implements
		CLIInterface {

	private static final String SCHEDULE_MODIFY_PARAM = "remove";
	private static final String SCHEDULE_ID_PARAM = "id";
	
    public static final String CLDB_HOST = "cldbip";
	public static final String CLDB_PORT = "cldbport";

	public static final Log LOG = LogFactory.getLog(SchedulePolicyRemoveCommand.class);

	public static final CLICommand schedulePolicyRemoveCommand = new CLICommand(
			SCHEDULE_MODIFY_PARAM, "modify schedule ", SchedulePolicyRemoveCommand.class, ExecutionTypeEnum.NATIVE, 
			new ImmutableMap.Builder<String,BaseInputParameter>()
			.put(MapRCliUtil.CLUSTER_NAME_PARAM,
									new TextInputParameter(MapRCliUtil.CLUSTER_NAME_PARAM,
											"cluster name",
											CLIBaseClass.NOT_REQUIRED,
											null))
				.put(SCHEDULE_ID_PARAM, new IntegerInputParameter(SCHEDULE_ID_PARAM, "id of schedule policy", CLIBaseClass.REQUIRED, null))
			.build(),null);


	public SchedulePolicyRemoveCommand(ProcessedInput input,
			CLICommand cliCommand) {
		super(input, cliCommand);
	}

	@Override
	public CommandOutput executeRealCommand() throws CLIProcessingException {
		OutputHierarchy ch = new OutputHierarchy();
		CommandOutput co = new CommandOutput(ch);
		
		SchedulePolicyProcRequest.Builder scheduleRequestBuilder = SchedulePolicyProcRequest.newBuilder();
		scheduleRequestBuilder.setPolicyOp(SchedulePolicyOP.REMOVE);
		Policy.Builder policyBuilder = Policy.newBuilder();
		int policyId = getParamIntValue(SCHEDULE_ID_PARAM, 0);
		policyBuilder.setPolicyId(policyId);
		scheduleRequestBuilder.setPolicy(policyBuilder.build());
				
	    byte[] data = null;
	    try {
	      scheduleRequestBuilder.setCreds(getUserCredentials());
	      if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
	    	  data = CLDBRpcCommonUtils.getInstance().sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
	    	  Common.MapRProgramId.CldbProgramId.getNumber(),
                      CLDBProto.CLDBProg.SchedulePolicyProc.getNumber(), 
                      scheduleRequestBuilder.build(), SchedulePolicyProcResponse.class);	    	  
	      } else {
	    	  data = CLDBRpcCommonUtils.getInstance().sendRequest(Common.MapRProgramId
	                                            .CldbProgramId.getNumber(),
	                                            CLDBProto.CLDBProg.SchedulePolicyProc.getNumber(), 
	                                            scheduleRequestBuilder.build(), SchedulePolicyProcResponse.class);
	      }
	      if (data == null) {
				/**
				* <MAPR_ERROR>
				* Message:RPC Request to create Schedule Policy failed. No data returned
				* Function:SchedulePolicyRemoveCommand.executeRealCommand()
				* Meaning:An error occurred.
				* Resolution:Contact technical support.
				* </MAPR_ERROR>
				*/
				LOG.error("RPC Request to create Schedule Policy failed. No data returned");
				ch.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
				return co;			
	      }  
	      SchedulePolicyProcResponse resp = SchedulePolicyProcResponse.parseFrom(data);
	      if ( resp.getStatus() != 0 ) {
	    	   if ( resp.getStatus() == Errno.EOPFORBIDDEN ) {
				/**
				* <MAPR_ERROR>
				* Message:Cannot remove schedule, as some volumes are still using it.
				* Function:SchedulePolicyRemoveCommand.executeRealCommand()
				* Meaning:You can only remove a schedule that no volumes are using.
				* Resolution:Before removing the schedule, select a different schedule for any volume that is using it.
				* </MAPR_ERROR>
				*/
				LOG.error("Cannot remove schedule, as some volumes are still using it.");
				ch.addError(new OutputError(Errno.EOPFAILED,"Can not remove schedule, as some volumes are still using it."));
	    	   } else {
				/**
				* <MAPR_ERROR>
				* Message:RPC Request to remove Schedule Policy failed with error: <error>
				* Function:SchedulePolicyRemoveCommand.executeRealCommand()
				* Meaning:An error occurred.
				* Resolution:Contact technical support.
				* </MAPR_ERROR>
				*/
				LOG.error("RPC Request to remove Schedule Policy failed with error: " + Errno.toString(resp.getStatus()));
				ch.addError(new OutputError(Errno.ERPCFAILED,"Request to remove Schedule Policy with error: " + Errno.toString(resp.getStatus())));
	    	   }
				return co;	    		   
	      }
      } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
            "MaprSecurityException " + "Exception", e);
	    } catch(Exception e) {
			/**
			* <MAPR_ERROR>
			* Message:RPC Request to remove Schedule Policy failed. No data returned
			* Function:SchedulePolicyRemoveCommand.executeRealCommand()
			* Meaning:An error occurred.
			* Resolution:Contact technical support.
			* </MAPR_ERROR>
			*/
			LOG.error("RPC Request to remove Schedule Policy failed. No data returned");
			ch.addError(new OutputError(Errno.ERPCFAILED,"Request to remove Schedule Policy failed. No data returned"));
			return co;				    	
	    }
		return co;
	}

}
