package com.mapr.cli.schedulepolicy.commands;

import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.google.common.collect.ImmutableMap;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.baseutils.Errno;
import com.mapr.cli.MapRCliUtil;
import com.mapr.cli.VolumeCommands;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.cldb.proto.CLDBProto.Policy;
import com.mapr.fs.cldb.proto.CLDBProto.PolicyRule;
import com.mapr.fs.cldb.proto.CLDBProto.SchedulePolicyProcRequest;
import com.mapr.fs.cldb.proto.CLDBProto.SchedulePolicyProcResponse;
import com.mapr.fs.cldb.proto.CLDBProto.SchedulePolicyProcRequest.SchedulePolicyOP;
import com.mapr.fs.proto.Common;
import com.mapr.security.MaprSecurityException;

/**
 * Command to list all the available schedules - no explicit parameters needed
 * @author yufeldman
 *
 */
public class SchedulePolicyListCommand extends CLIBaseClass implements
		CLIInterface {

	private static final String SCHEDULE_LIST_PARAM = "list";
    public static final String CLDB_HOST = "cldbip";
	public static final String CLDB_PORT = "cldbport";
	public static final String OUTPUT_PARAM_NAME = "output";

	public static final Log LOG = LogFactory.getLog(SchedulePolicyListCommand.class);

	public static final CLICommand schedulePolicyListCommand = new CLICommand(
			SCHEDULE_LIST_PARAM, "list available schedules ", SchedulePolicyListCommand.class, ExecutionTypeEnum.NATIVE, 
			new ImmutableMap.Builder<String,BaseInputParameter>()
			.put(MapRCliUtil.CLUSTER_NAME_PARAM,
									new TextInputParameter(MapRCliUtil.CLUSTER_NAME_PARAM,
											"cluster name",
											CLIBaseClass.NOT_REQUIRED,
											null))
				.put(VolumeCommands.OUTPUT_PARAM_NAME, new TextInputParameter(OUTPUT_PARAM_NAME,
                     "output type: verbose or terse",
                     CLIBaseClass.NOT_REQUIRED, "terse"))
			.build(),null);

	public SchedulePolicyListCommand(ProcessedInput input, CLICommand cliCommand) {
		super(input, cliCommand);
	}

	@Override
	public CommandOutput executeRealCommand() throws CLIProcessingException {
		OutputHierarchy ch = new OutputHierarchy();
		CommandOutput co = new CommandOutput(ch);
		
		SchedulePolicyProcRequest.Builder scheduleRequestBuilder = SchedulePolicyProcRequest.newBuilder();
		scheduleRequestBuilder.setPolicyOp(SchedulePolicyOP.LIST);
		Policy.Builder policyBuilder = Policy.newBuilder();
		scheduleRequestBuilder.setPolicy(policyBuilder.build());
			
	    byte[] data = null;
	    try {
	      scheduleRequestBuilder.setCreds(getUserCredentials());
	      if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
		      data = CLDBRpcCommonUtils.getInstance().sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
		    		  Common.MapRProgramId.CldbProgramId.getNumber(),
                      CLDBProto.CLDBProg.SchedulePolicyProc.getNumber(), 
                      scheduleRequestBuilder.build(), SchedulePolicyProcResponse.class);	    	  
	      } else {
	      	  data = CLDBRpcCommonUtils.getInstance().sendRequest(Common.MapRProgramId
	                                            .CldbProgramId.getNumber(),
	                                            CLDBProto.CLDBProg.SchedulePolicyProc.getNumber(), 
	                                            scheduleRequestBuilder.build(), SchedulePolicyProcResponse.class);
	      }
	      if (data == null) {
				/**
				* <MAPR_ERROR>
				* Message:RPC Request to list Schedule Policies failed. No data returned
				* Function:SchedulePolicyListCommand.executeRealCommand()
				* Meaning:An error occurred.
				* Resolution:Contact technical support.
				* </MAPR_ERROR>
				*/
				LOG.error("RPC Request to list Schedule Policies failed. No data returned");
				ch.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
				return co;			
	      }  
	      SchedulePolicyProcResponse resp = SchedulePolicyProcResponse.parseFrom(data);
	      if (resp.getStatus() == Errno.E_NOT_ENOUGH_PRIVILEGES) {
	        /**
	        * <MAPR_ERROR>
	        * Message:RPC to list Schedule Policies failed
	        * Function:SchedulePolicyListCommand.executeRealCommand()
	        * Meaning:The schedule policies could not be listed because the caller had insufficient privileges.
	        * Resolution:Make sure you have sufficient permissions and try again.
	        * </MAPR_ERROR>
	        */
	        LOG.error("RPC to list Schedule Policies failed");
	        /**
	        * <MAPR_ERROR>
	        * Message:Not enough privileges to view the list of schedules.
	        * Function:SchedulePolicyListCommand.executeRealCommand()
	        * Meaning:The schedule policies could not be listed because the caller had insufficient privileges.
	        * Resolution:Make sure you have sufficient permissions and try again.
	        * </MAPR_ERROR>
	        */
	        ch.addError(new OutputError(Errno.E_NOT_ENOUGH_PRIVILEGES,"Not enough privileges to view the list of schedules."));
	        return co;
	      } else if ( resp.getStatus() != 0 ) {
				/**
				* <MAPR_ERROR>
				* Message:RPC to list Schedule Policies faile
				* Function:SchedulePolicyListCommand.executeRealCommand()
				* Meaning:An error occurred.
				* Resolution:Contact technical support.
				* </MAPR_ERROR>
				*/
				LOG.error("RPC to list Schedule Policies failed");
				/**
				* <MAPR_ERROR>
				* Message:Request to list Schedule Policies failed
				* Function:SchedulePolicyListCommand.executeRealCommand()
				* Meaning:An error occurred.
				* Resolution:Contact technical support.
				* </MAPR_ERROR>				
				*/
				ch.addError(new OutputError(Errno.ERPCFAILED,"Request to list Schedule Policies failed"));
				return co;	    		   
	      }
	      List<Policy> policies = resp.getPoliciesList();
	      
	      for ( Policy policy : policies ) {
	    	  OutputNode mainNode = new OutputNode();
	    	  OutputNode policyNodeId = new OutputNode("id", policy.getPolicyId());
	    	  OutputNode policyNodeName = new OutputNode("name", policy.getPolicyName());
	    	  OutputNode policyNodeInUse = new OutputNode("inuse", (policy.getVolumeIdCount() > 0) ? 1 : 0); 
	    	  mainNode.addChild(policyNodeId);
	    	  mainNode.addChild(policyNodeName);
	    	  mainNode.addChild(policyNodeInUse);
	    	  List<PolicyRule> rules = policy.getPolicyRulesList();
	    	  mainNode.addChild(new OutputNode("rules"));
	    	  for ( PolicyRule rule : rules) {
		    	  OutputNode policyNodeRules = new OutputNode("rules");
	    		  policyNodeRules.addChild(new OutputNode("frequency", rule.getFrequency().name()));
	    		  if ( rule.hasDate() ) {
	    			  policyNodeRules.addChild(new OutputNode("date", rule.getDate()));
	    		  }
	    		  if ( rule.hasTime() ) {
	    			  policyNodeRules.addChild(new OutputNode("time", rule.getTime()));
	    		  }
	    		  if ( rule.hasMinutes()) {
	    			  policyNodeRules.addChild(new OutputNode("minute", rule.getMinutes()));
	    		  }
	    		  
	    		  policyNodeRules.addChild(new OutputNode("retain", rule.getRetainTime().getNumberOfUnits() + rule.getRetainTime().getTimeUnitsEnum().name()));
		    	  mainNode.addChild(policyNodeRules);
	    	  }
		      ch.addNode(mainNode);
	      }
      } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
            "MaprSecurityException " + "Exception", e);
	    } catch(Exception e) {
			/**
			* <MAPR_ERROR>
			* Message:RPC Request to list Schedule Policies failed. No data returned
			* Function:SchedulePolicyListCommand.executeRealCommand()
			* Meaning:An error occurred.
			* Resolution:Contact technical support.
			* </MAPR_ERROR>
			*/
			LOG.error("RPC Request to list Schedule Policies failed. No data returned");
			/**
			* <MAPR_ERROR>
			* Message:Request to to list Schedule Policies. No data returned
			* Function:SchedulePolicyListCommand.executeRealCommand()
			* Meaning:An error occurred.
			* Resolution:Contact technical support.
			* </MAPR_ERROR>
			*/
			ch.addError(new OutputError(Errno.ERPCFAILED,"Request to to list Schedule Policies. No data returned"));
			return co;				    	
	    }
		return co;
	}

}
