package com.mapr.cli.schedulepolicy.commands;

import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import com.google.common.collect.ImmutableMap;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.baseutils.Errno;
import com.mapr.cli.MapRCliUtil;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.JSONObjectInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.cldb.proto.CLDBProto.Policy;
import com.mapr.fs.cldb.proto.CLDBProto.PolicyFrequencyEnum;
import com.mapr.fs.cldb.proto.CLDBProto.PolicyRule;
import com.mapr.fs.cldb.proto.CLDBProto.RetainTimeUnitsEnum;
import com.mapr.fs.cldb.proto.CLDBProto.SchedulePolicyProcRequest;
import com.mapr.fs.cldb.proto.CLDBProto.SchedulePolicyProcResponse;
import com.mapr.fs.cldb.proto.CLDBProto.PolicyRule.RetainPeriod;
import com.mapr.fs.cldb.proto.CLDBProto.SchedulePolicyProcRequest.SchedulePolicyOP;
import com.mapr.fs.proto.Common;
import com.mapr.security.MaprSecurityException;

/**
 * Command to create a new schedule 
 * params:
 *   schedule - JSONObject that incorporates all schedule details
 * @author yufeldman
 *
 */
public class SchedulePolicyCreateCommand extends CLIBaseClass implements
		CLIInterface {

	private static final String SCHEDULE_CREATE_PARAM = "create";
	private static final String SCHEDULE_PARAM = "schedule";
    public static final String CLDB_HOST = "cldbip";
	public static final String CLDB_PORT = "cldbport";

	public static final Log LOG = LogFactory.getLog(SchedulePolicyCreateCommand.class);

	public static Pattern p = Pattern.compile("(\\d+)|(\\w+)");
	
	public static final CLICommand schedulePolicyCreateCommand = new CLICommand(
			SCHEDULE_CREATE_PARAM, "create schedule ", SchedulePolicyCreateCommand.class, ExecutionTypeEnum.NATIVE, 
							new ImmutableMap.Builder<String,BaseInputParameter>()
							.put(MapRCliUtil.CLUSTER_NAME_PARAM,
									new TextInputParameter(MapRCliUtil.CLUSTER_NAME_PARAM,
											"cluster name",
											CLIBaseClass.NOT_REQUIRED,
											null))
								.put(SCHEDULE_PARAM, new JSONObjectInputParameter(SCHEDULE_PARAM, "JSON Object to denote all the schedule properties", CLIBaseClass.REQUIRED))
							.build(),null);


	public SchedulePolicyCreateCommand(ProcessedInput input,
			CLICommand cliCommand) {
		super(input, cliCommand);
	}

	@Override
	public CommandOutput executeRealCommand() throws CLIProcessingException {
		OutputHierarchy ch = new OutputHierarchy();
		CommandOutput co = new CommandOutput(ch);
		
		SchedulePolicyProcRequest.Builder scheduleRequestBuilder = SchedulePolicyProcRequest.newBuilder();
		scheduleRequestBuilder.setPolicyOp(SchedulePolicyOP.CREATE);
		Policy.Builder policyBuilder = Policy.newBuilder();

		// validation is already done by this point - no point to revalidate here again
		String jSonString = input.getParameterByName(SCHEDULE_PARAM).getParamValues().get(0);
		String scheduleName;
		
		try {
			JSONObject jO = new JSONObject(jSonString);
			scheduleName = jO.getString("name");
			policyBuilder.setPolicyName(scheduleName);
			JSONArray jRules = jO.optJSONArray("rules");
			List<PolicyRule> rules = new ArrayList<PolicyRule>();
			for (int i = 0; i < jRules.length(); i++ ) {
				PolicyRule.Builder ruleBuilder = PolicyRule.newBuilder();
				JSONObject jRO = jRules.getJSONObject(i);
				String jFreq = jRO.getString("frequency");
				ruleBuilder.setFrequency(PolicyFrequencyEnum.valueOf(jFreq));
				String date = jRO.optString("date");
				if ( date !=  null && date.length() > 0) {
					ruleBuilder.setDate(date);
				}
				if ( !jRO.isNull("time") ) {
				  int time = jRO.optInt("time");
				  ruleBuilder.setTime(time);
				}
				if ( !jRO.isNull("minute")) {
					int minutes = jRO.optInt("minute");
					ruleBuilder.setMinutes(minutes);
				}
				String retain = jRO.getString("retain");
		        List<String> groups = new ArrayList<String>();
		        Matcher m = p.matcher(retain);
		        while (m.find()) {
		            groups.add(m.group().trim());
		        }
		        if ( groups.size() != 2 ) {
					/**
					* <MAPR_ERROR>
					* Message:retain portion of the input is invalid: <retain>
					* Function:SchedulePolicyCreateCommand.executeRealCommand()
					* Meaning:An invalid {{retain}} value was specified.
					* Resolution:Check the command syntax and try again.
					* </MAPR_ERROR>
					*/
					LOG.error("retain portion of the input is invalid: " + retain);
					ch.addError(new OutputError(Errno.EINVAL,"retain portion of the input is invalid: " + retain).setField("retain").setFieldValue(retain));
					continue;
		        }
		        RetainPeriod.Builder retainBuilder = RetainPeriod.newBuilder();
		        retainBuilder.setNumberOfUnits(Integer.valueOf(groups.get(0)));
		        retainBuilder.setTimeUnitsEnum(RetainTimeUnitsEnum.valueOf(groups.get(1)));
		        ruleBuilder.setRetainTime(retainBuilder);
		        rules.add(ruleBuilder.build());
			}
			policyBuilder.addAllPolicyRules(rules);
			scheduleRequestBuilder.setPolicy(policyBuilder.build());
		} catch (JSONException e) {
			/**
			* <MAPR_ERROR>
			* Message:Provided Input cannot be converted to JSONObject: <JSON string>
			* Function:SchedulePolicyCreateCommand.executeRealCommand()
			* Meaning:The specified JSON was invalid.
			* Resolution:Check the command syntax and try again.
			* </MAPR_ERROR>
			*/
			LOG.error("Provided Input cannot be converted to JSONObject: " + jSonString);
			ch.addError(new OutputError(Errno.EINVALMISC,"Provided Input can not be converted to JSONObject: " + jSonString));
			return co;
		} catch (IllegalArgumentException e) {
			/**
			* <MAPR_ERROR>
			* Message:Provided Input is not compliant with enums: <JSON string>
			* Function:SchedulePolicyCreateCommand.executeRealCommand()
			* Meaning:The input to the command was specified incorrectly.
			* Resolution:Check the command syntax and try again.
			* </MAPR_ERROR>
			*/
			LOG.error("Provided Input is not compliant with enums: " + jSonString);
			ch.addError(new OutputError(Errno.EINVALMISC,"Provided Input is not compliant with enums: " + jSonString));
			return co;			
		}

	    byte[] data = null;
	    try {
	      scheduleRequestBuilder.setCreds(getUserCredentials());
	      if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
		      data = CLDBRpcCommonUtils.getInstance().sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
		    		  Common.MapRProgramId.CldbProgramId.getNumber(),
                      CLDBProto.CLDBProg.SchedulePolicyProc.getNumber(), 
                      scheduleRequestBuilder.build(), SchedulePolicyProcResponse.class); 	  
	      } else {
	      data = CLDBRpcCommonUtils.getInstance().sendRequest(Common.MapRProgramId
	                                            .CldbProgramId.getNumber(),
	                                            CLDBProto.CLDBProg.SchedulePolicyProc.getNumber(), 
	                                            scheduleRequestBuilder.build(), SchedulePolicyProcResponse.class);
	      }
	      if (data == null) {
				/**
				* <MAPR_ERROR>
				* Message:RPC Request to create Schedule Policy failed. No data returned
				* Function:SchedulePolicyCreateCommand.executeRealCommand()
				* Meaning:An error occurred.
				* Resolution:Contact technical support.
				* </MAPR_ERROR>
				*/
				LOG.error("RPC Request to create Schedule Policy failed. No data returned");
				ch.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
				return co;			
	      }  
	      SchedulePolicyProcResponse resp = SchedulePolicyProcResponse.parseFrom(data);
	      if ( resp.getStatus() != 0 ) {
				/**
				* <MAPR_ERROR>
				* Message:Request to create Schedule Policy failed with error: <error>
				* Function:SchedulePolicyCreateCommand.executeRealCommand()
				* Meaning:An error occurred.
				* Resolution:Contact technical support.
				* </MAPR_ERROR>
				*/
                if (resp.getStatus() == Errno.EEXIST && scheduleName != null) {
                  String error = "A schedule policy with name '" + scheduleName + "' already exists.";
                  LOG.error(error);
                  ch.addError(new OutputError(resp.getStatus(), error));
                } else {
                  LOG.error("Request to create Schedule Policy failed with error: " + Errno.toString(resp.getStatus()));
                  ch.addError(new OutputError(resp.getStatus(),"Request to create Schedule Policy failed with error: " + Errno.toString(resp.getStatus())));
                }
				return co;				    	  
	      }
      } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
            "MaprSecurityException " + "Exception", e);
	    } catch(Exception e) {
			/**
			* <MAPR_ERROR>
			* Message:RPC Request to create Schedule Policy failed. No data returned
			* Function:SchedulePolicyCreateCommand.executeRealCommand()
			* Meaning:An error occurred.
			* Resolution:Contact technical support.
			* </MAPR_ERROR>
			*/
			LOG.error("RPC Request to create Schedule Policy failed. No data returned");
			ch.addError(new OutputError(Errno.ERPCFAILED,"Request to create Schedule Policy failed. No data returned"));
			return co;				    	
	    }
		
		return co;
	}
}
