/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.cli.marlin;

import com.google.common.collect.ImmutableMap;
import com.mapr.baseutils.Errno;
import com.mapr.cli.DbCommands;
import com.mapr.cli.MapRCliUtil;
import com.mapr.cli.common.FileclientRun;
import com.mapr.cli.table.TabletStats;
import com.mapr.cliframework.base.*;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.BooleanInputParameter;
import com.mapr.cliframework.base.inputparams.IntegerInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.streams.Admin;
import com.mapr.streams.Streams;
import com.mapr.streams.impl.admin.CursorInfo;
import com.mapr.streams.impl.admin.TopicFeedInfo;
import com.mapr.streams.impl.admin.MarlinAdminImpl;
import com.mapr.fs.proto.Marlinserver.MarlinTopicMetaEntry;
import com.mapr.fs.proto.Marlinserver.TopicFeedStatInfo;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.log4j.Logger;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

public class TopicCommands extends CLIBaseClass implements CLIInterface {

  private static final Logger LOG = Logger.getLogger(TopicCommands.class);
  private static final String PATH_PARAM_NAME = "path";
  private static final String TOPIC_PARAM_NAME = "topic";
  private static final String NUM_PARTITIONS_PARAM_NAME = "partitions";

  private static final CLICommand createCommand =
      new CLICommand(
          "create",
          "usage: stream topic create -path <path> -topic <topic> -partitions <partitions>",
          TopicCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "Stream Path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(TOPIC_PARAM_NAME,
                  new TextInputParameter(TOPIC_PARAM_NAME,
                      "Topic Name",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(NUM_PARTITIONS_PARAM_NAME,
                  new IntegerInputParameter(NUM_PARTITIONS_PARAM_NAME,
                      "Number of partitions. default: attribute defaultpartitions on the stream",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .build(),
          null)
          .setShortUsage("stream topic create -path <path> -topic <topic>");

  private static final CLICommand editCommand =
      new CLICommand(
          "edit",
          "usage: stream topic edit -path <path> -topic <topic> -partitions <partitions>",
          TopicCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "Stream Path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(TOPIC_PARAM_NAME,
                  new TextInputParameter(TOPIC_PARAM_NAME,
                      "Topic Name",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(NUM_PARTITIONS_PARAM_NAME,
                  new IntegerInputParameter(NUM_PARTITIONS_PARAM_NAME,
                      "Number of partitions",
                      CLIBaseClass.REQUIRED,
                      null))
              .build(),
          null)
          .setShortUsage("stream topic edit -path <path> -topic <topic> -partitions <partitions>");

  private static final CLICommand infoCommand =
      new CLICommand(
          "info",
          "usage: stream topic info -path <path> -topic <topic>",
          TopicCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "Stream Path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(TOPIC_PARAM_NAME,
                  new TextInputParameter(TOPIC_PARAM_NAME,
                      "Topic Name",
                      CLIBaseClass.REQUIRED,
                      null))
              .build(),
          null)
          .setShortUsage("stream topic info -path <path> -topic <topic>");

  private static final CLICommand listCommand =
      new CLICommand(
          "list",
          "usage: stream topic list -path <path>",
          TopicCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "Stream Path",
                      CLIBaseClass.REQUIRED,
                      null))
              .build(),
          null)
          .setShortUsage("stream topic list -path <path>");

  private static final CLICommand deleteCommand =
      new CLICommand(
          "delete",
          "usage: stream topic delete -path <path> -topic <topic>",
          TopicCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "Stream Path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(TOPIC_PARAM_NAME,
                  new TextInputParameter(TOPIC_PARAM_NAME,
                      "Topic Name",
                      CLIBaseClass.REQUIRED,
                      null))
              .build(),
          null)
          .setShortUsage("stream topic delete -path <path> -topic <topic>");

  public static final CLICommand topicCommands =
    new CLICommand(
          "topic", "topic [create|edit|delete|info]",
          CLIUsageOnlyCommand.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new CLICommand[]{
              createCommand,
              editCommand,
              deleteCommand,
              infoCommand,
              listCommand
          }
      ).setShortUsage("stream topic [create|edit|delete|info|list]");

  public TopicCommands(ProcessedInput input, CLICommand cliCommand) {
    super(input, cliCommand);
  }

  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    if (!super.validateInput()) {
      return output;
    }

    String cmdName = cliCommand.getCommandName();
    if (cmdName.equalsIgnoreCase(createCommand.getCommandName())) {
      createTopic(out);
    } else if (cmdName.equalsIgnoreCase(editCommand.getCommandName())) {
      editTopic(out);
    } else if (cmdName.equalsIgnoreCase(infoCommand.getCommandName())) {
      infoTopic(out);
    } else if (cmdName.equalsIgnoreCase(listCommand.getCommandName())) {
      listTopics(out);
    } else if (cmdName.equalsIgnoreCase(deleteCommand.getCommandName())) {
      deleteTopic(out);
    }
    return output;
  }

  private void createTopic(OutputHierarchy out) throws CLIProcessingException {
    try {
      final String user = getUserLoginId();
      final String streamName = getParamTextValue(PATH_PARAM_NAME, 0);
      final String path = DbCommands.getTransformedPath(streamName,
                                                        getUserLoginId());
      final String topicName = getParamTextValue(TOPIC_PARAM_NAME, 0);
      final Configuration conf = new Configuration();

      if (LOG.isDebugEnabled())
        LOG.debug("begin creating topic=" + topicName + " in stream=" +
                  path + " user=" + user);

      int nf = 0;
      if (isParamPresent(NUM_PARTITIONS_PARAM_NAME)) {
        nf = getParamIntValue(NUM_PARTITIONS_PARAM_NAME, 0);
        if (nf < 1)
          throw new CLIProcessingException("Invalid value " + nf +
                                           " for numpartitions, but be >=1");
      }
      final int numFeeds = nf;

      new FileclientRun(user) {
        @Override
        public void runAsProxyUser() throws CLIProcessingException,IOException {
          try {
            Admin madmin = Streams.newAdmin(conf);

            if (numFeeds == 0)
              madmin.createTopic(path, topicName);
            else
              madmin.createTopic(path, topicName, numFeeds);
          } catch (Exception e) {
            LOG.error("createTopic failed : " + e);
            throw e;
          }
        }
      };
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    }
  }

  private void editTopic(OutputHierarchy out) throws CLIProcessingException {
    try {
      final String user = getUserLoginId();
      final String streamName = getParamTextValue(PATH_PARAM_NAME, 0);
      final String path = DbCommands.getTransformedPath(streamName,
                                                        getUserLoginId());
      final String topicName = getParamTextValue(TOPIC_PARAM_NAME, 0);
      final Configuration conf = new Configuration();

      if (LOG.isDebugEnabled())
        LOG.debug("begin editing topic=" + topicName + " in stream=" +
                  path + " user=" + user);

      final int numFeeds = getParamIntValue(NUM_PARTITIONS_PARAM_NAME, 0);
      if (numFeeds < 1)
        throw new CLIProcessingException("Invalid value " + numFeeds +
                                         " for numpartitions, but be >=1");

      new FileclientRun(user) {
        @Override
        public void runAsProxyUser() throws CLIProcessingException,IOException {
          try {
            Admin madmin = Streams.newAdmin(conf);

            madmin.editTopic(path, topicName, numFeeds);
          } catch (Exception e) {
            LOG.error("editTopic failed : " + e);
            throw e;
          }
        }
      };
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    }
  }

  private void infoTopic(OutputHierarchy out) throws CLIProcessingException {
    try {
      final String user = getUserLoginId();
      final String streamName = getParamTextValue(PATH_PARAM_NAME, 0);
      final String path = DbCommands.getTransformedPath(streamName,
                                                        getUserLoginId());
      final String topic = getParamTextValue(TOPIC_PARAM_NAME, 0);
      final List<TopicFeedInfo> feedsList =
          new ArrayList<TopicFeedInfo>();
      final Configuration conf = new Configuration();

      if (LOG.isDebugEnabled())
        LOG.debug("begin info topic=" + topic + " stream=" + path +
                  " user=" + user);

      new FileclientRun(user) {
        @Override
        public void runAsProxyUser() throws CLIProcessingException,IOException {
          try {
            Admin admin = Streams.newAdmin(conf);
            MarlinAdminImpl madmin = (MarlinAdminImpl)admin;

            String topicFullName = path + ":" + topic;
            List<TopicFeedInfo> tlist = madmin.infoTopic(topicFullName);
            feedsList.addAll(tlist);
          } catch (Exception e) {
            LOG.error("infoTopic failed : " + e);
            throw e;
          }
        }
      };

      for (TopicFeedInfo tinfo : feedsList)
        out.addNode(formatFeedInfo(tinfo));
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    }
  }

  private String getMaster(TopicFeedStatInfo tinfo) {
    return tinfo.getMaster() + ':' + tinfo.getMasterPort();
  }

  private String getServers(TopicFeedStatInfo tinfo) {
    StringBuilder builder = new StringBuilder();
    for (int i = 0; i < tinfo.getServersList().size(); ++i) {
      if (i != 0)
         builder.append(", ");

      builder.append(tinfo.getServersList().get(i));
      builder.append(":");
      builder.append(tinfo.getServerPortsList().get(i));
    }
    return builder.toString();
  }

  private OutputNode formatFeedInfo(TopicFeedInfo topicFeedinfo) {
    OutputNode feedNode = new OutputNode();
    long minSeq = Long.MAX_VALUE;
    long minListenerTs = Long.MAX_VALUE;
    List<CursorInfo> cursorList = topicFeedinfo.cursorList();
    TopicFeedStatInfo tinfo = topicFeedinfo.stat();
    for (CursorInfo c : cursorList) {
      if (c.cursor() < minSeq)
        minSeq = c.cursor();
      if (c.timestamp() < minListenerTs)
        minListenerTs = c.timestamp();
    }

    feedNode.addChild(new OutputNode("partitionid", tinfo.getFeedId()));
    feedNode.addChild(new OutputNode("physicalsize",
                                     tinfo.getPhysicalSize()));
    feedNode.addChild(new OutputNode("logicalsize",
                                     tinfo.getLogicalSize()));
    //feedNode.addChild(new OutputNode("minoffset", tinfo.getMinSeq()));
    feedNode.addChild(new OutputNode("maxoffset", tinfo.getMaxSeq()));

    feedNode.addChild(new OutputNode("minoffsetacrossconsumers",
                                     minSeq == Long.MAX_VALUE ? 0 : minSeq));

    long minTS = tinfo.getTimeRange().getMinTS();
    if (minTS == Long.MAX_VALUE)
      minTS = 0;
    feedNode.addChild(new OutputNode("mintimestamp",
                      StreamsCommands.millisToDate(minTS)));

    long maxTS = tinfo.getTimeRange().getMaxTS();
    if (maxTS == Long.MIN_VALUE)
      maxTS = 0;
    feedNode.addChild(new OutputNode("maxtimestamp",
                      StreamsCommands.millisToDate(maxTS)));

    if (minListenerTs == Long.MAX_VALUE)
      minListenerTs = 0;
    feedNode.addChild(new OutputNode("mintimestampacrossconsumers",
                      StreamsCommands.millisToDate(minListenerTs)));

    if (tinfo.hasFid()) {
      String fid = MapRCliUtil.getFidAsString(tinfo.getFid());
      feedNode.addChild(new OutputNode("fid", fid));
    }
    if (tinfo.hasMaster()) {
      String master = getMaster(tinfo);
      feedNode.addChild(new OutputNode("master", master));
    }
    if (tinfo.getServersList().size() > 0) {
      String servers = getServers(tinfo);
      feedNode.addChild(new OutputNode("servers", servers));
    }
    return feedNode;
  }

  private void listTopics(OutputHierarchy out) throws CLIProcessingException {
    try {
      final String user = getUserLoginId();
      final String streamName = getParamTextValue(PATH_PARAM_NAME, 0);
      final String path = DbCommands.getTransformedPath(streamName,
                                                        getUserLoginId());
      final Map<String, List<TopicFeedInfo>> topicsMap;
      final Configuration conf = new Configuration();

      if (LOG.isDebugEnabled())
        LOG.debug("begin topic list for stream=" + path +
                  " user=" + user);

      topicsMap = new HashMap<String, List<TopicFeedInfo>>();
      new FileclientRun(user) {
        @Override
        public void runAsProxyUser() throws CLIProcessingException,IOException {
          try {
            Admin admin = Streams.newAdmin(conf);
            MarlinAdminImpl madmin = (MarlinAdminImpl)admin;
            Map<String, List<TopicFeedInfo>> tmap;

            tmap = madmin.listTopics(streamName);
            topicsMap.putAll(tmap);
          } catch (Exception e) {
            LOG.error("listTopics failed : " + e);
            throw e;
          }
        }
      };

      Iterator entries = topicsMap.entrySet().iterator();
      while (entries.hasNext()) {
        Map.Entry entry = (Map.Entry) entries.next();
        String topicName = (String) entry.getKey();
        List<TopicFeedInfo> feedList;
        feedList = (List<TopicFeedInfo>) entry.getValue();

        OutputNode topicNode = new OutputNode();
        topicNode.addChild(new OutputNode("topic", topicName));
        topicNode.addChild(new OutputNode("partitions", feedList.size()));

        long physicalSize = 0;
        long logicalSize = 0;
        long maxLag = 0;
        long numConsumers = 0;
        for (TopicFeedInfo feed : feedList) {
          TopicFeedStatInfo statInfo = feed.stat();
          List<CursorInfo> cursorList = feed.cursorList();

          physicalSize += statInfo.getPhysicalSize();
          logicalSize += statInfo.getLogicalSize();

          long maxProducerTS = statInfo.getTimeRange().getMaxTS();
          if (maxProducerTS != Long.MIN_VALUE) {
            for (CursorInfo c : cursorList) {
              long lag = maxProducerTS - c.timestamp();
              if (lag > maxLag)
                maxLag = lag;
            }
          }

          if (cursorList.size() > numConsumers)
            numConsumers = cursorList.size();
        }
        topicNode.addChild(new OutputNode("consumers", numConsumers));
        topicNode.addChild(new OutputNode("physicalsize", physicalSize));
        topicNode.addChild(new OutputNode("logicalsize", logicalSize));
        topicNode.addChild(new OutputNode("maxlag", maxLag));
        out.addNode(topicNode);
      }
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    }
  }

  private void deleteTopic(OutputHierarchy out) throws CLIProcessingException {
    try {
      final String user = getUserLoginId();
      final String streamName = getParamTextValue(PATH_PARAM_NAME, 0);
      final String path = DbCommands.getTransformedPath(streamName,
                                                        getUserLoginId());
      final String topicName = getParamTextValue(TOPIC_PARAM_NAME, 0);
      final Configuration conf = new Configuration();

      if (LOG.isDebugEnabled())
        LOG.debug("begin deleting topic=" + topicName + " in stream=" +
                  path + " user=" + user);

      new FileclientRun(user) {
        @Override
        public void runAsProxyUser() throws CLIProcessingException,IOException {
          try {
            Admin madmin = Streams.newAdmin(conf);
            madmin.deleteTopic(path, topicName);
          } catch (Exception e) {
            LOG.error("deleteTopic failed : " + e);
            throw e;
          }
        }
      };
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    }
  }
}
