/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.cli.marlin;

import com.google.common.collect.ImmutableMap;
import com.mapr.baseutils.Errno;
import com.mapr.cli.DbCommands;
import com.mapr.cli.DbRegionCommands;
import com.mapr.cli.MapRCliUtil;
import com.mapr.cli.common.FileclientRun;
import com.mapr.cli.table.TabletStats;
import com.mapr.cliframework.base.*;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.BooleanInputParameter;
import com.mapr.cliframework.base.inputparams.IntegerInputParameter;
import com.mapr.cliframework.base.inputparams.LongInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.fs.MapRFileSystem;
import com.mapr.streams.Admin;
import com.mapr.streams.impl.admin.MStreamDescriptor;
import com.mapr.streams.StreamDescriptor;
import com.mapr.streams.Streams;
import com.mapr.fs.proto.Dbserver.ForcedCompactionType;
import com.mapr.fs.proto.Dbserver.TabletDesc;
import com.mapr.fs.proto.Dbserver.TabletStatResponse;
import com.mapr.fs.proto.Dbserver.SpaceUsage;
import com.mapr.fs.tables.TableProperties;

import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.log4j.Logger;

import java.io.IOException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.Map;
import java.util.List;

public class StreamsCommands extends CLIBaseClass implements CLIInterface {

  private static final Logger LOG = Logger.getLogger(StreamsCommands.class);
  private static final String PATH_PARAM_NAME = "path";
  private static final String TTL_PARAM_NAME = "ttl";
  private static final String AUTO_CREATE_TOPICS_PARAM_NAME = "autocreate";
  private static final String DEFAULT_PARTITIONS = "defaultpartitions";
  private static final String COMPRESSION_PARAM_NAME = "compression";
  private static final String CLIENT_COMPRESSION_PARAM_NAME = "clientcompression";
  private static final String PRODUCE_PERM_PARAM_NAME = "produceperm";
  private static final String LISTEN_PERM_PARAM_NAME = "consumeperm";
  private static final String TOPIC_PERM_PARAM_NAME = "topicperm";
  private static final String COPY_PERM_PARAM_NAME = "copyperm";
  private static final String ADMIN_PERM_PARAM_NAME = "adminperm";
  private static final String COPY_META_FROM_PARAM_NAME = "copymetafrom";
  private static final String NTHREADS_PARAM_NAME = "nthreads";

  private static final CLICommand createCommand =
      new CLICommand(
          "create",
          "usage: stream create -path <path>",
          StreamsCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "Stream Path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(TTL_PARAM_NAME,
                  new LongInputParameter(TTL_PARAM_NAME,
                      "Time to live in seconds. default:604800",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(AUTO_CREATE_TOPICS_PARAM_NAME,
                  new BooleanInputParameter(AUTO_CREATE_TOPICS_PARAM_NAME,
                      "Auto create topics. default:true",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(DEFAULT_PARTITIONS,
                  new IntegerInputParameter(DEFAULT_PARTITIONS,
                      "Default partitions per topic. default:1",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(COMPRESSION_PARAM_NAME,
                  new TextInputParameter(COMPRESSION_PARAM_NAME,
                      "off|lz4|lzf|zlib. default:inherit from parent directory",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(CLIENT_COMPRESSION_PARAM_NAME,
                  new BooleanInputParameter(COMPRESSION_PARAM_NAME,
                      "Compress on client. default:false",
                      CLIBaseClass.NOT_REQUIRED,
                      null).setInvisible(true))
              .put(PRODUCE_PERM_PARAM_NAME,
                  new TextInputParameter(PRODUCE_PERM_PARAM_NAME,
                      "Producer access control expression. default u:creator",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(LISTEN_PERM_PARAM_NAME,
                  new TextInputParameter(LISTEN_PERM_PARAM_NAME,
                      "Consumer access control expression. default u:creator",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(TOPIC_PERM_PARAM_NAME,
                  new TextInputParameter(TOPIC_PERM_PARAM_NAME,
                      "Topic CRUD access control expression. default u:creator",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(COPY_PERM_PARAM_NAME,
                  new TextInputParameter(COPY_PERM_PARAM_NAME,
                      "Stream copy access control expression. default u:creator",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(ADMIN_PERM_PARAM_NAME,
                  new TextInputParameter(ADMIN_PERM_PARAM_NAME,
                      "Stream administration access control expression. default u:creator",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(COPY_META_FROM_PARAM_NAME,
                   new TextInputParameter(COPY_META_FROM_PARAM_NAME,
                       "Stream to copy attributes from. default:none",
                       CLIBaseClass.NOT_REQUIRED,
                       null))
              .build(),
          null)
          .setShortUsage("stream create -path <path>");

  private static final CLICommand editCommand =
      new CLICommand(
          "edit",
          "usage: stream edit -path <path>",
          StreamsCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "Stream Path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(TTL_PARAM_NAME,
                  new LongInputParameter(TTL_PARAM_NAME,
                      "Time to live in seconds",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(AUTO_CREATE_TOPICS_PARAM_NAME,
                  new BooleanInputParameter(AUTO_CREATE_TOPICS_PARAM_NAME,
                      "Auto create topics",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(DEFAULT_PARTITIONS,
                  new IntegerInputParameter(DEFAULT_PARTITIONS,
                      "Default partitions per topic",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(COMPRESSION_PARAM_NAME,
                  new TextInputParameter(COMPRESSION_PARAM_NAME,
                      "off|lz4|lzf|zlib",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(CLIENT_COMPRESSION_PARAM_NAME,
                  new BooleanInputParameter(COMPRESSION_PARAM_NAME,
                      "Compress on client. default:false",
                      CLIBaseClass.NOT_REQUIRED,
                      null).setInvisible(true))
              .put(PRODUCE_PERM_PARAM_NAME,
                  new TextInputParameter(PRODUCE_PERM_PARAM_NAME,
                      "Producer access control expression. default u:creator",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(LISTEN_PERM_PARAM_NAME,
                  new TextInputParameter(LISTEN_PERM_PARAM_NAME,
                      "Consumer access control expression. default u:creator",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(TOPIC_PERM_PARAM_NAME,
                  new TextInputParameter(TOPIC_PERM_PARAM_NAME,
                      "Topic CRUD access control expression. default u:creator",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(COPY_PERM_PARAM_NAME,
                  new TextInputParameter(COPY_PERM_PARAM_NAME,
                      "Stream copy access control expression. default u:creator",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(ADMIN_PERM_PARAM_NAME,
                  new TextInputParameter(ADMIN_PERM_PARAM_NAME,
                      "Stream administration access control expression. default u:creator",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .build(),
          null)
          .setShortUsage("stream edit -path <path>");

  private static final CLICommand infoCommand =
      new CLICommand(
          "info",
          "usage: stream info -path <path>",
          StreamsCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "Stream Path",
                      CLIBaseClass.REQUIRED,
                      null))
              .build(),
          null)
          .setShortUsage("stream info -path <path>");

  private static final CLICommand deleteCommand =
      new CLICommand(
          "delete",
          "usage: stream delete -path <path>",
          StreamsCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "Stream Path",
                      CLIBaseClass.REQUIRED,
                      null))
              .build(),
          null)
          .setShortUsage("stream delete -path <path>");

  private static final CLICommand purgeCommand =
      new CLICommand(
          "purge",
          "usage: stream purge -path <path>",
          StreamsCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "Stream Path",
                      CLIBaseClass.REQUIRED,
                      null))
             .put(NTHREADS_PARAM_NAME,
                  new IntegerInputParameter(NTHREADS_PARAM_NAME,
                      "number of parallel threads",
                      CLIBaseClass.NOT_REQUIRED,
                      16).setInvisible(true))
              .build(),
          null)
          .setShortUsage("stream purge -path <path>");

  public static final CLICommand streamCommands =
      new CLICommand("stream",
          "stream",
          CLIUsageOnlyCommand.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new CLICommand[]{
              createCommand,
              editCommand,
              infoCommand,
              deleteCommand,
              purgeCommand,
              TopicCommands.topicCommands,
              CursorCommands.cursorCommands,
              AssignCommands.assignCommands,
              StreamReplicaCommands.replicaCommands,
              StreamUpstreamCommands.upstreamCommands,
          })
          .setShortUsage("stream [create|edit|info|delete|topic|cursor|replica|upstream]");

  public StreamsCommands(ProcessedInput input, CLICommand cliCommand) {
    super(input, cliCommand);
  }

  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    if (!super.validateInput()) {
      return output;
    }

    String cmdName = cliCommand.getCommandName();
    if (cmdName.equalsIgnoreCase(createCommand.getCommandName())) {
      return createStream();
    } else if (cmdName.equalsIgnoreCase(editCommand.getCommandName())) {
      editStream(out);
    } else if (cmdName.equalsIgnoreCase(infoCommand.getCommandName())) {
      infoStream(out);
    } else if (cmdName.equalsIgnoreCase(deleteCommand.getCommandName())) {
      deleteStream(out);
    } else if (cmdName.equalsIgnoreCase(purgeCommand.getCommandName())) {
      purgeStream(out);
    }
    return output;
  }

  private CommandOutput createStream() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    try {
      final String user = getUserLoginId();
      final String streamName = getParamTextValue(PATH_PARAM_NAME, 0);
      final String path = DbCommands.getTransformedPath(streamName,
                                                        getUserLoginId());
      final Configuration conf = new Configuration();
      StreamDescriptor sdesc = null;

      if (LOG.isDebugEnabled())
        LOG.debug("begin creating stream=" + path + " user=" + user);

      if (isParamPresent(COPY_META_FROM_PARAM_NAME)) {
        String copyFrom = getParamTextValue(COPY_META_FROM_PARAM_NAME, 0);
        sdesc = getStreamDescriptor(copyFrom);
      } else {
        sdesc = Streams.newStreamDescriptor();
      }

      if (isParamPresent(DEFAULT_PARTITIONS)) {
        int val = getParamIntValue(DEFAULT_PARTITIONS, 0);
        sdesc.setDefaultPartitions(val); 
      }
      if (isParamPresent(TTL_PARAM_NAME)) {
        long ttl = getParamLongValue(TTL_PARAM_NAME, 0);
        sdesc.setTimeToLiveSec(ttl);
      }
      if (isParamPresent(COMPRESSION_PARAM_NAME)) {
        String ctype = getParamTextValue(COMPRESSION_PARAM_NAME, 0);
        sdesc.setCompressionAlgo(ctype);
      }
      if (isParamPresent(CLIENT_COMPRESSION_PARAM_NAME)) {
        boolean val = getParamBooleanValue(CLIENT_COMPRESSION_PARAM_NAME, 0);

        MStreamDescriptor mdesc = (MStreamDescriptor)sdesc;
        mdesc.setClientCompression(val);
      }
      if (isParamPresent(AUTO_CREATE_TOPICS_PARAM_NAME)) {
        boolean val = getParamBooleanValue(AUTO_CREATE_TOPICS_PARAM_NAME, 0);
        sdesc.setAutoCreateTopics(val);
      }

      boolean permsSpecified = false;
      if (isParamPresent(PRODUCE_PERM_PARAM_NAME)) {
        String perms = getParamTextValue(PRODUCE_PERM_PARAM_NAME, 0);
        sdesc.setProducePerms(perms);
        permsSpecified = true;
      }
      if (isParamPresent(LISTEN_PERM_PARAM_NAME)) {
        String perms = getParamTextValue(LISTEN_PERM_PARAM_NAME, 0);
        sdesc.setConsumePerms(perms);
        permsSpecified = true;
      }
      if (isParamPresent(TOPIC_PERM_PARAM_NAME)) {
        String perms = getParamTextValue(TOPIC_PERM_PARAM_NAME, 0);
        sdesc.setTopicPerms(perms);
        permsSpecified = true;
      }
      if (isParamPresent(COPY_PERM_PARAM_NAME)) {
        String perms = getParamTextValue(COPY_PERM_PARAM_NAME, 0);
        sdesc.setCopyPerms(perms);
        permsSpecified = true;
      }
      if (isParamPresent(ADMIN_PERM_PARAM_NAME)) {
        String perms = getParamTextValue(ADMIN_PERM_PARAM_NAME, 0);
        sdesc.setAdminPerms(perms);
        permsSpecified = true;
      }

      final StreamDescriptor desc = sdesc;
      new FileclientRun(user) {
        @Override
        public void runAsProxyUser() throws CLIProcessingException,IOException {
          try {
            Admin madmin = Streams.newAdmin(conf);
            madmin.createStream(path, desc);
          } catch (Exception e) {
            LOG.error("createStream failed : " + e);
            throw e;
          }
        }
      };

      if (!permsSpecified) {
        return new TextCommandOutput(
          ("Warning: produce/consume/topic permissions defaulting to creator." +
           " To change, execute" +
           " 'maprcli stream edit -path " + path +
           " -produceperm p -consumeperm p -topicperm p'").getBytes());
      }
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    }
    return output;
  }

  private void editStream(OutputHierarchy out) throws CLIProcessingException {
    try {
      final String user = getUserLoginId();
      final String streamName = getParamTextValue(PATH_PARAM_NAME, 0);
      final String path = DbCommands.getTransformedPath(streamName,
                                                        getUserLoginId());
      final Configuration conf = new Configuration();
      final StreamDescriptor sdesc = Streams.newStreamDescriptor();

      if (LOG.isDebugEnabled())
        LOG.debug("begin editing stream=" + path + " user=" + user);

      if (isParamPresent(DEFAULT_PARTITIONS)) {
        int val = getParamIntValue(DEFAULT_PARTITIONS, 0);
        sdesc.setDefaultPartitions(val);
      }
      if (isParamPresent(TTL_PARAM_NAME)) {
        long ttl = getParamLongValue(TTL_PARAM_NAME, 0);
        sdesc.setTimeToLiveSec(ttl);
      }
      if (isParamPresent(COMPRESSION_PARAM_NAME)) {
        String ctype = getParamTextValue(COMPRESSION_PARAM_NAME, 0);
        sdesc.setCompressionAlgo(ctype);
      }
      if (isParamPresent(CLIENT_COMPRESSION_PARAM_NAME)) {
        boolean val = getParamBooleanValue(CLIENT_COMPRESSION_PARAM_NAME, 0);

        MStreamDescriptor mdesc = (MStreamDescriptor)sdesc;
        mdesc.setClientCompression(val);
      }
      if (isParamPresent(AUTO_CREATE_TOPICS_PARAM_NAME)) {
        boolean val = getParamBooleanValue(AUTO_CREATE_TOPICS_PARAM_NAME, 0);
        sdesc.setAutoCreateTopics(val);
      }
      if (isParamPresent(PRODUCE_PERM_PARAM_NAME)) {
        String perms = getParamTextValue(PRODUCE_PERM_PARAM_NAME, 0);
        sdesc.setProducePerms(perms);
      }
      if (isParamPresent(LISTEN_PERM_PARAM_NAME)) {
        String perms = getParamTextValue(LISTEN_PERM_PARAM_NAME, 0);
        sdesc.setConsumePerms(perms);
      }
      if (isParamPresent(TOPIC_PERM_PARAM_NAME)) {
        String perms = getParamTextValue(TOPIC_PERM_PARAM_NAME, 0);
        sdesc.setTopicPerms(perms);
      }
      if (isParamPresent(COPY_PERM_PARAM_NAME)) {
        String perms = getParamTextValue(COPY_PERM_PARAM_NAME, 0);
        sdesc.setCopyPerms(perms);
      }
      if (isParamPresent(ADMIN_PERM_PARAM_NAME)) {
        String perms = getParamTextValue(ADMIN_PERM_PARAM_NAME, 0);
        sdesc.setAdminPerms(perms);
      }

      new FileclientRun(user) {
        @Override
        public void runAsProxyUser() throws CLIProcessingException,IOException {
          try {
            Admin madmin = Streams.newAdmin(conf);
            madmin.editStream(path, sdesc);
          } catch (Exception e) {
            LOG.error("editStream failed : " + e);
            throw e;
          }
        }
      };
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    }
  }

  private StreamDescriptor getStreamDescriptor(final String path)
    throws CLIProcessingException,IOException {

    final List<StreamDescriptor> sdList = new ArrayList<StreamDescriptor>();
    final String user = getUserLoginId();
    final Configuration conf = new Configuration();

    new FileclientRun(user) {
      @Override
      public void runAsProxyUser() throws CLIProcessingException,IOException {
        try {
          Admin madmin = Streams.newAdmin(conf);

          StreamDescriptor sdesc = madmin.getStreamDescriptor(path);
          sdList.add(sdesc);
        } catch (Exception e) {
          LOG.error("getting stream info for stream " + path + " failed: " + e);
          throw new IOException(e.getMessage());
        }
      }
    };

    return sdList.get(0);
  }

  private void infoStream(OutputHierarchy out) throws CLIProcessingException {
    try {
      final String user = getUserLoginId();
      final String streamName = getParamTextValue(PATH_PARAM_NAME, 0);
      final String path = DbCommands.getTransformedPath(streamName,
                                                        getUserLoginId());
      final Configuration conf = new Configuration();
      final List<Integer> ntopicsList = new ArrayList<Integer>();

      if (LOG.isDebugEnabled())
        LOG.debug("begin info stream=" + path + " user=" + user);


      // Get the stream descriptor
      StreamDescriptor mdesc = getStreamDescriptor(streamName);
      MStreamDescriptor sdesc = (MStreamDescriptor)mdesc;

      // Get the topic counts
      new FileclientRun(user) {
        @Override
        public void runAsProxyUser() throws CLIProcessingException,IOException {
          try {
            Admin madmin = Streams.newAdmin(conf);
            int ntopics = madmin.countTopics(path);
            ntopicsList.add(ntopics);
          } catch (Exception e) {
            LOG.error("getting count of topics failed : " + e);
            throw e;
          }
        }
      };
      int ntopics = ntopicsList.get(0);

      // Get space usage
      long physicalSize = 0;
      long logicalSize = 0;
      TabletStats tabletStats = new TabletStats(path, getUserLoginId());
      OutputHierarchy tmpOut = new OutputHierarchy();
      List<TabletDesc> tablets = tabletStats.getTablets(tmpOut, 0,
                                                        Integer.MAX_VALUE);
      if (tablets == null) {
        if (out.getOutputErrors() != null && out.getOutputErrors().size() > 1)
          LOG.error("Error fetching regions for stream " +
                    streamName + " : " +
                    out.getOutputErrors().get(0).getErrorDescription());
        else
          LOG.error("Error fetching regions for stream " + streamName);
      }

      for (TabletDesc tablet : tablets) {
        try {
          // blocks until a response is available
          TabletStatResponse tsr = tabletStats.getTabletStatResponse(tablet);
          if (tsr != null && tsr.hasUsage()) {
            SpaceUsage su = tsr.getUsage();
            long blockSize = 8 * 1024L;
            physicalSize += su.getNumPhysicalBlocks() * blockSize;
            logicalSize += su.getNumLogicalBlocks() * blockSize;
          }
        } catch (Exception e) {
          LOG.error("Error fetching region stats for stream " +
                    streamName, e);
        }
      }

      OutputNode streamNode = new OutputNode();
      streamNode.addChild(new OutputNode("path", path));
      streamNode.addChild(new OutputNode("physicalsize", physicalSize));
      streamNode.addChild(new OutputNode("logicalsize", logicalSize));
      streamNode.addChild(new OutputNode("numtopics", ntopics));
      streamNode.addChild(new OutputNode(DEFAULT_PARTITIONS,
                                         sdesc.getDefaultPartitions()));
      streamNode.addChild(new OutputNode(TTL_PARAM_NAME,
                                         sdesc.getTimeToLiveSec()));
      streamNode.addChild(new OutputNode(COMPRESSION_PARAM_NAME,
                                         sdesc.getCompressionAlgo()));
      streamNode.addChild(new OutputNode(AUTO_CREATE_TOPICS_PARAM_NAME,
                                         sdesc.getAutoCreateTopics()));
      if (sdesc.hasProducePerms())
        streamNode.addChild(new OutputNode(PRODUCE_PERM_PARAM_NAME,
                                           sdesc.getProducePerms()));

      if (sdesc.hasConsumePerms())
        streamNode.addChild(new OutputNode(LISTEN_PERM_PARAM_NAME,
                                           sdesc.getConsumePerms()));

      if (sdesc.hasTopicPerms())
        streamNode.addChild(new OutputNode(TOPIC_PERM_PARAM_NAME,
                                           sdesc.getTopicPerms()));

      if (sdesc.hasCopyPerms())
        streamNode.addChild(new OutputNode(COPY_PERM_PARAM_NAME,
                                           sdesc.getCopyPerms()));

      if (sdesc.hasAdminPerms())
        streamNode.addChild(new OutputNode(ADMIN_PERM_PARAM_NAME,
                                           sdesc.getAdminPerms()));

      out.addNode(streamNode);
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    }

  }

  private void deleteStream(OutputHierarchy out) throws CLIProcessingException {
    try {
      final String user = getUserLoginId();
      final String streamName = getParamTextValue(PATH_PARAM_NAME, 0);
      final String path = DbCommands.getTransformedPath(streamName,
                                                        getUserLoginId());
      final Configuration conf = new Configuration();

      if (LOG.isDebugEnabled())
        LOG.debug("begin deleting stream=" + path + " user=" + user);

      new FileclientRun(user) {
        @Override
        public void runAsProxyUser() throws CLIProcessingException,IOException {
          try {
            Admin madmin = Streams.newAdmin(conf);
            madmin.deleteStream(path);
          } catch (Exception e) {
            LOG.error("deleteStream failed : " + e);
            throw e;
          }
        }
      };
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    }
  }

  private void purgeStream(OutputHierarchy out) throws CLIProcessingException {
    try {
      String user = getUserLoginId();
      String streamName = getParamTextValue(PATH_PARAM_NAME, 0);
      String path = DbCommands.getTransformedPath(streamName,
                                                  getUserLoginId());
      if (LOG.isDebugEnabled())
        LOG.debug("begin purging stream=" + path + " user=" + user);

      int nthreads = getParamIntValue(NTHREADS_PARAM_NAME, 0);
      int ctype = ForcedCompactionType.ForcedCompactionTTL.getNumber();

      OutputError err = DbRegionCommands.packAllRegionsOfTable(streamName,
                                                               ctype,
                                                               nthreads,
                                                               getUserLoginId());
      if (err != null && (err.getErrorCode() != Errno.SUCCESS)) {
        out.addError(err);
        return;
      }
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
      throw e;
    }
  }

  public static boolean isStream(String pathName)
      throws IOException,CLIProcessingException {

     MapRFileSystem mfs = MapRCliUtil.getMapRFileSystem();
     Path path = new Path(pathName);

     if (!mfs.isTable(path))
       return false;

     TableProperties tableProp = mfs.getTableProperties(path);
     if (tableProp.getAttr().getIsMarlinTable())
       return true;
     else
       return false;
  }

  public static String millisToDate(long millisSinceEpoch) {
    DateFormat format = new SimpleDateFormat("yyyy-MM-dd'T'hh:mm:ss.SSSZ");
    return format.format(new Date(millisSinceEpoch));
  }
}
