/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.cli.marlin;

import org.apache.log4j.Logger;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.security.UserGroupInformation;

import com.google.common.collect.ImmutableMap;
import com.mapr.baseutils.Errno;
import com.mapr.cli.DbUpstreamCommands;
import com.mapr.cliframework.base.*;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.inputparams.*;
import com.mapr.fs.AceHelper;

import java.io.IOException;

public class StreamUpstreamCommands extends DbUpstreamCommands {

  private static final Logger LOG = Logger.getLogger(StreamUpstreamCommands.class);

  public static final String COLUMNS_PARAM_NAME = "columns";
  public static final String OUTPUT_PARAM_NAME = "output";
  public static final String START_PARAM_NAME = "start";
  public static final String LIMIT_PARAM_NAME = "limit";

  private static final String PATH_PARAM_NAME = "path";
  private static final String UPSTREAM_PARAM_NAME = "upstream";
  private static final String PAUSED_PARAM_NAME = "paused";

  private static final CLICommand listCommand =
      new CLICommand(
          "list",
          "usage: stream upstream list -path <streamPath>",
          StreamUpstreamCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "stream path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(COLUMNS_PARAM_NAME,
                  new TextInputParameter(COLUMNS_PARAM_NAME,
                      "columns",
                      CLIBaseClass.NOT_REQUIRED,
                      "all").setInvisible(true))
              .put(OUTPUT_PARAM_NAME,
                  new TextInputParameter(OUTPUT_PARAM_NAME,
                      "verbose|terse",
                      CLIBaseClass.NOT_REQUIRED,
                      "verbose").setInvisible(true))
              .put(START_PARAM_NAME,
                  new IntegerInputParameter(START_PARAM_NAME,
                      "start",
                      CLIBaseClass.NOT_REQUIRED,
                      0).setInvisible(true))
              .put(LIMIT_PARAM_NAME,
                  new IntegerInputParameter(LIMIT_PARAM_NAME,
                      "limit",
                      CLIBaseClass.NOT_REQUIRED,
                      Integer.MAX_VALUE).setInvisible(true))
              .build(), null)
          .setShortUsage("stream upstream list -path <streamPath>");

  private static final CLICommand addCommand =
      new CLICommand(
          "add",
          "usage: stream upstream add -path <streamPath> -upstream <upstreamPath>",
          StreamUpstreamCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "stream path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(UPSTREAM_PARAM_NAME,
                  new TextInputParameter(UPSTREAM_PARAM_NAME,
                      "upstream stream path",
                      CLIBaseClass.REQUIRED,
                      null))
             .build(), null)
          .setShortUsage("stream upstream add -path <streamPath> -upstream <upstreamPath>");

  private static final CLICommand removeCommand =
      new CLICommand(
          "remove",
          "usage: stream upstream remove -path <streamPath> -upstream <upstreamPath>",
          StreamUpstreamCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "stream path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(UPSTREAM_PARAM_NAME,
                  new TextInputParameter(UPSTREAM_PARAM_NAME,
                      "upstream stream path",
                      CLIBaseClass.REQUIRED,
                      null))
             .build(), null)
          .setShortUsage("stream upstream remove -path <streamPath> -upstream <upstreamPath>");

  // main command
  public static final CLICommand upstreamCommands =
      new CLICommand(
          "upstream", "upstream [add|list|remove]",
          CLIUsageOnlyCommand.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new CLICommand[]{
              addCommand,
              listCommand,
              removeCommand,
          }
      ).setShortUsage("stream upstream [add|list|remove]");

  public StreamUpstreamCommands(ProcessedInput input, CLICommand cliCommand) {
    super(input, cliCommand);
  }

  @Override
  public String entityName() {
    return "stream";
  }

  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    if (!super.validateInput()) {
      return output;
    }

    // Common validation for all cmds : stream cmds can act on only streams
    String streamPath = getParamTextValue(PATH_PARAM_NAME, 0);
    try {
      if (!StreamsCommands.isStream(streamPath)) {
        out.addError(new OutputError(Errno.EINVAL,
                                     "Path " + streamPath +
                                     " is not a valid stream"));
        return output;
      }
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
      return output;
    } catch (IOException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    }

    String cname = cliCommand.getCommandName();
    if (cname.equalsIgnoreCase(addCommand.getCommandName())) {
      addUpstream(out);
    } else if (cname.equalsIgnoreCase(listCommand.getCommandName())) {
      listUpstream(out);
    } else if (cname.equalsIgnoreCase(removeCommand.getCommandName())) {
      removeUpstream(out);
    }

    return output;
  }

  protected void addUpstream(OutputHierarchy out) throws CLIProcessingException {
    super.addUpstream(out);
  }

  protected void listUpstream(OutputHierarchy out) throws CLIProcessingException {
    super.listUpstream(out);
  }

  protected void removeUpstream(OutputHierarchy out) throws CLIProcessingException {
    super.removeUpstream(out);
  }
}
