/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.cli.marlin;

import org.apache.log4j.Logger;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.security.UserGroupInformation;

import com.google.common.collect.ImmutableMap;
import com.mapr.baseutils.Errno;
import com.mapr.cli.DbReplicaCommands;
import com.mapr.cliframework.base.*;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.inputparams.*;
import com.mapr.fs.AceHelper;

import java.io.IOException;

public class StreamReplicaCommands extends DbReplicaCommands {

  private static final Logger LOG = Logger.getLogger(StreamReplicaCommands.class);

  public static final String COLUMNS_PARAM_NAME = "columns";
  public static final String OUTPUT_PARAM_NAME = "output";
  public static final String START_PARAM_NAME = "start";
  public static final String LIMIT_PARAM_NAME = "limit";

  private static final String PATH_PARAM_NAME = "path";
  private static final String REPLICA_PARAM_NAME = "replica";
  private static final String NEWREPLICA_PARAM_NAME = "newreplica";
  private static final String PAUSED_PARAM_NAME = "paused";
  private static final String THROTTLE_PARAM_NAME = "throttle";
  private static final String ENCRYPTION_PARAM_NAME = "networkencryption";
  private static final String SYNCHRONOUS_PARAM_NAME = "synchronous";
  private static final String COMPRESSION_PARAM_NAME = "networkcompression";
  private static final String MULTI_ARG_SEP = ",";
  private static final String COLUMN_SEP = ":";
  private static final String REFRESH_PARAM_NAME = "refreshnow";
  private static final String WAIT_FOR_COMPLETION_PARAM_NAME="waitforcompletion";
  private static final String MULTI_MASTER_PARAM_NAME="multimaster";
  private static final String TICKET_PATH_PARAM_NAME="ticketpath";

  private static final CLICommand listCommand =
      new CLICommand(
          "list",
          "usage: stream replica list -path <streampath>",
          StreamReplicaCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "stream path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(REFRESH_PARAM_NAME,
                  new BooleanInputParameter(REFRESH_PARAM_NAME,
                      REFRESH_PARAM_NAME,
                      CLIBaseClass.NOT_REQUIRED,
                      false))
              .put(COLUMNS_PARAM_NAME,
                  new TextInputParameter(COLUMNS_PARAM_NAME,
                      "columns",
                      CLIBaseClass.NOT_REQUIRED,
                      "all").setInvisible(true))
              .put(OUTPUT_PARAM_NAME,
                  new TextInputParameter(OUTPUT_PARAM_NAME,
                      "verbose|terse",
                      CLIBaseClass.NOT_REQUIRED,
                      "verbose").setInvisible(true))
              .put(START_PARAM_NAME,
                  new IntegerInputParameter(START_PARAM_NAME,
                      "start",
                      CLIBaseClass.NOT_REQUIRED,
                      0).setInvisible(true))
              .put(LIMIT_PARAM_NAME,
                  new IntegerInputParameter(LIMIT_PARAM_NAME,
                      "limit",
                      CLIBaseClass.NOT_REQUIRED,
                      Integer.MAX_VALUE).setInvisible(true))
              .build(), null)
          .setShortUsage("stream replica list -path <tablepath>");

  private static final CLICommand addCommand =
      new CLICommand(
          "add",
          "usage: stream replica add -path <streamPath> -replica <replicaPath> " +
          "-paused <true|false> " +
          "-throttle <true|false> " +
          "-synchronous <true|false> " +
          "-networkencryption <true|false> " +
          "-networkcompression <off|on|lz4|lzf|zlib> ",
          StreamReplicaCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "stream path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(REPLICA_PARAM_NAME,
                  new TextInputParameter(REPLICA_PARAM_NAME,
                      "remote stream path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(PAUSED_PARAM_NAME,
                  new BooleanInputParameter(PAUSED_PARAM_NAME,
                      "start replication in paused state",
                      CLIBaseClass.NOT_REQUIRED,
                      false))
              .put(THROTTLE_PARAM_NAME,
                  new BooleanInputParameter(THROTTLE_PARAM_NAME,
                      "throttle replication operations under load",
                      CLIBaseClass.NOT_REQUIRED,
                      false))
              .put(ENCRYPTION_PARAM_NAME,
                  new BooleanInputParameter(ENCRYPTION_PARAM_NAME,
                      "enable on-wire encryption",
                      CLIBaseClass.NOT_REQUIRED,
                      false))
              .put(SYNCHRONOUS_PARAM_NAME,
                  new BooleanInputParameter(SYNCHRONOUS_PARAM_NAME,
                      "replicate to remote stream before acknowledging producers",
                      CLIBaseClass.NOT_REQUIRED,
                      false))
              .put(COMPRESSION_PARAM_NAME,
                  new TextInputParameter(COMPRESSION_PARAM_NAME,
                      "on-wire compression type: off|lz4|lzf|zlib default: compression setting on stream",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
             .build(), null)
          .setShortUsage("stream replica add -path <streamPath> -replica <replicaPath>");

  private static final CLICommand autoSetupCommand =
      new CLICommand(
          "autosetup",
          "usage: stream replica autosetup -path <streamPath> -replica <replicaPath> " +
          "-multimaster <true/false>" +
          "-waitforcompletion <true/false>" +
          "-throttle <true|false> " +
          "-synchronous <true|false> " +
          "-networkencryption <true|false> " +
          "-networkcompression <off|lz4|lzf|zlib> " +
          "-ticketpath <path>" ,
          StreamReplicaCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "stream path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(REPLICA_PARAM_NAME,
                  new TextInputParameter(REPLICA_PARAM_NAME,
                      "remote stream path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(SYNCHRONOUS_PARAM_NAME,
                  new BooleanInputParameter(SYNCHRONOUS_PARAM_NAME,
                      "replicate to remote stream before acknowledging producers",
                      CLIBaseClass.NOT_REQUIRED,
                      false))
              .put(MULTI_MASTER_PARAM_NAME,
                  new BooleanInputParameter(MULTI_MASTER_PARAM_NAME,
                      "set up bi-directional replication",
                      CLIBaseClass.NOT_REQUIRED,
                      false))
              .put(THROTTLE_PARAM_NAME,
                  new BooleanInputParameter(THROTTLE_PARAM_NAME,
                      "throttle replication operations under load",
                      CLIBaseClass.NOT_REQUIRED,
                      false))
              .put(ENCRYPTION_PARAM_NAME,
                  new BooleanInputParameter(ENCRYPTION_PARAM_NAME,
                      "enable on-wire encryption",
                      CLIBaseClass.NOT_REQUIRED,
                      false))
              .put(COMPRESSION_PARAM_NAME,
                  new TextInputParameter(COMPRESSION_PARAM_NAME,
                      "on-wire compression type: off|lz4|lzf|zlib default: compression setting on stream",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(WAIT_FOR_COMPLETION_PARAM_NAME,
                  new BooleanInputParameter(WAIT_FOR_COMPLETION_PARAM_NAME,
                      "wait for completion",
                      CLIBaseClass.NOT_REQUIRED,
                      true).setInvisible(true))
              .put(TICKET_PATH_PARAM_NAME,
                  new TextInputParameter(TICKET_PATH_PARAM_NAME,
                      "ticket path",
                      CLIBaseClass.NOT_REQUIRED,
                      null).setInvisible(true))
             .build(), null)
          .setShortUsage("stream replica autosetup -path <streamPath> -replica <replicaPath>");

  private static final CLICommand editCommand =
      new CLICommand(
          "edit",
          "usage: stream replica edit -path <streamPath> -replica <replicaPath> " +
          "-newreplica <newReplicaPath> " +
          "-paused <true|false> -allowallcfs <true> " +
          "-throttle <true|false> " +
          "-synchronous <true|false> " +
          "-networkencryption <true|false> " +
          "-networkcompression <off|lz4|lzf|zlib> ",
          StreamReplicaCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "stream path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(REPLICA_PARAM_NAME,
                  new TextInputParameter(REPLICA_PARAM_NAME,
                      "remote stream path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(NEWREPLICA_PARAM_NAME,
                  new TextInputParameter(NEWREPLICA_PARAM_NAME,
                      "renamed stream path",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(THROTTLE_PARAM_NAME,
                  new BooleanInputParameter(THROTTLE_PARAM_NAME,
                      "throttle replication operations under load",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(ENCRYPTION_PARAM_NAME,
                  new BooleanInputParameter(ENCRYPTION_PARAM_NAME,
                      "enable on-wire encryption",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(SYNCHRONOUS_PARAM_NAME,
                  new BooleanInputParameter(SYNCHRONOUS_PARAM_NAME,
                      "replicate to remote stream before acknowledging producers",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(COMPRESSION_PARAM_NAME,
                  new TextInputParameter(COMPRESSION_PARAM_NAME,
                      "on-wire compression type: off|lz4|lzf|zlib",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
             .build(), null)
          .setShortUsage("stream replica edit -path <streamPath> " +
                         "-replica <replicaPath> -newreplica <newReplicaPath>" +
                         "-paused <true|false> " +
                         "-allowallcfs <true> -throttle <true|false>  " +
                         "-synchronous <true|false> -networkencryption <true|false " +
                         "-networkcompression <off|lz4|lzf|zlib>");

  private static final CLICommand pauseCommand =
      new CLICommand(
          "pause",
          "usage: stream replica pause -path <streamPath> -replica <replicaPath>",
          StreamReplicaCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "stream path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(REPLICA_PARAM_NAME,
                  new TextInputParameter(REPLICA_PARAM_NAME,
                      "remote stream path",
                      CLIBaseClass.REQUIRED,
                      null))
             .build(), null)
           .setShortUsage("stream replica pause -path <streamPath> -replica <replicaPath>");

  private static final CLICommand resumeCommand =
      new CLICommand(
          "resume",
          "usage: stream replica resume -path <streamPath> -replica <replicaPath>",
          StreamReplicaCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "stream path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(REPLICA_PARAM_NAME,
                  new TextInputParameter(REPLICA_PARAM_NAME,
                      "remote stream path",
                      CLIBaseClass.REQUIRED,
                      null))
             .build(), null)
           .setShortUsage("stream replica resume -path <streamPath> -replica <replicaPath>");


  private static final CLICommand removeCommand =
      new CLICommand(
          "remove",
          "usage: stream replica remove -path <streamPath> -replica <replicaPath>",
          StreamReplicaCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "stream path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(REPLICA_PARAM_NAME,
                  new TextInputParameter(REPLICA_PARAM_NAME,
                      "remote stream path",
                      CLIBaseClass.REQUIRED,
                      null))
             .build(), null)
          .setShortUsage("stream replica remove -path <streamPath> -replica <replicaPath>");

  // main command
  public static final CLICommand replicaCommands =
      new CLICommand(
          "replica", "replica [add|edit|list|remove|pause|resume]",
          CLIUsageOnlyCommand.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new CLICommand[]{
              addCommand,
              editCommand,
              listCommand,
              removeCommand,
              pauseCommand,
              resumeCommand,
              autoSetupCommand
          }
      ).setShortUsage("stream replica [add|edit|list|remove|pause|resume]");

  public StreamReplicaCommands(ProcessedInput input, CLICommand cliCommand) {
    super(input, cliCommand);
  }

  @Override
  public String entityName() {
    return "stream";
  }

  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    if (!super.validateInput()) {
      return output;
    }

    // Common validation for all cmds : stream cmds can act on only streams
    String streamPath = getParamTextValue(PATH_PARAM_NAME, 0);
    try {
      if (!StreamsCommands.isStream(streamPath)) {
        out.addError(new OutputError(Errno.EINVAL,
                                     "Path " + streamPath +
                                     " is not a valid stream"));
        return output;
      }
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
      return output;
    } catch (IOException e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    }

    String cname = cliCommand.getCommandName();
    if (cname.equalsIgnoreCase(addCommand.getCommandName())) {
      addReplica(out);
    } else if (cname.equalsIgnoreCase(editCommand.getCommandName())) {
      editReplica(out);
    } else if (cname.equalsIgnoreCase(listCommand.getCommandName())) {
      listReplica(out);
    } else if (cname.equalsIgnoreCase(removeCommand.getCommandName())) {
      removeReplica(out);
    } else if (cname.equalsIgnoreCase(pauseCommand.getCommandName())) {
      pauseOrResumeReplica(out, true);
    } else if (cname.equalsIgnoreCase(resumeCommand.getCommandName())) {
      pauseOrResumeReplica(out, false);
    } else if (cname.equalsIgnoreCase(autoSetupCommand.getCommandName())) {
      autoSetup(out);
    }
    return output;
  }

  protected void autoSetup(OutputHierarchy out)
      throws CLIProcessingException {
    super.autoSetup(out);
  }

  protected void addReplica(OutputHierarchy out)
      throws CLIProcessingException {
    super.addReplica(out);
  }

  protected void listReplica(OutputHierarchy out)
      throws CLIProcessingException {
    super.listReplica(out);
  }

  protected void editReplica(OutputHierarchy out)
      throws CLIProcessingException {
    super.editReplica(out);
  }

  protected void removeReplica(OutputHierarchy out)
      throws CLIProcessingException {
    super.removeReplica(out);
  }

  protected void pauseOrResumeReplica(OutputHierarchy out, boolean pause)
      throws CLIProcessingException {
    super.pauseOrResumeReplica(out, pause);
  }
}
