/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.cli.marlin;

import com.google.common.collect.ImmutableMap;

import com.mapr.baseutils.Errno;
import com.mapr.cli.common.FileclientRun;
import com.mapr.cli.table.TabletStats;
import com.mapr.cliframework.base.*;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.BooleanInputParameter;
import com.mapr.cliframework.base.inputparams.IntegerInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.log4j.Logger;
import com.mapr.streams.Admin;
import com.mapr.streams.Streams;
import com.mapr.streams.impl.admin.MarlinAdminImpl;
import com.mapr.streams.impl.admin.CursorInfo;
import com.mapr.fs.proto.Marlinserver.TopicFeedStatInfo;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

public class CursorCommands extends CLIBaseClass implements CLIInterface {

  private static final Logger LOG = Logger.getLogger(CursorCommands.class);
  private static final String LISTENER_GROUP_PARAM_NAME = "consumergroup";
  private static final String PATH_PARAM_NAME = "path";
  private static final String TOPIC_PARAM_NAME = "topic";
  private static final String PARTITION_PARAM_NAME = "partition";

  private static final CLICommand listCommand =
      new CLICommand(
          "list",
          "usage: stream cursor list -path <path> -consumergroup <consumer group Id> -topic <topic> -partition <partitionId>",
          CursorCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "Stream Path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(LISTENER_GROUP_PARAM_NAME,
                  new TextInputParameter(LISTENER_GROUP_PARAM_NAME,
                      "Consumer Group ID",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(TOPIC_PARAM_NAME,
                  new TextInputParameter(TOPIC_PARAM_NAME,
                      "Topic Name",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PARTITION_PARAM_NAME,
                  new IntegerInputParameter(PARTITION_PARAM_NAME,
                      "Partition ID",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .build(),
          null)
          .setShortUsage("stream cursor list -path <path>");

  private static final CLICommand deleteCommand =
      new CLICommand(
          "delete",
          "usage: stream cursor delete -path <path> -consumergroup <consumer group Id> -topic <topic> -partition <partitionId>",
          CursorCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "Stream Path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(LISTENER_GROUP_PARAM_NAME,
                  new TextInputParameter(LISTENER_GROUP_PARAM_NAME,
                      "Consumer Group ID",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(TOPIC_PARAM_NAME,
                  new TextInputParameter(TOPIC_PARAM_NAME,
                      "Topic Name",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PARTITION_PARAM_NAME,
                  new IntegerInputParameter(PARTITION_PARAM_NAME,
                      "Partition ID",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .build(),
          null)
          .setShortUsage("stream cursor delete -path <path>");

  public static final CLICommand cursorCommands =
    new CLICommand(
          "cursor", "cursor [delete|list]",
          CLIUsageOnlyCommand.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new CLICommand[]{
              deleteCommand,
              listCommand
          }
      ).setShortUsage("stream cursor [delete|list]");

  public CursorCommands(ProcessedInput input, CLICommand cliCommand) {
    super(input, cliCommand);
  }

  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    if (!super.validateInput()) {
      return output;
    }

    String cmdName = cliCommand.getCommandName();
    if (cmdName.equalsIgnoreCase(listCommand.getCommandName())) {
      listCursors(out);
    } else if (cmdName.equalsIgnoreCase(deleteCommand.getCommandName())) {
      deleteCursor(out);
    }
    return output;
  }


  private void listCursors(OutputHierarchy out) throws CLIProcessingException {
    try {
      final String user = getUserLoginId();
      final String streamName = getParamTextValue(PATH_PARAM_NAME, 0);
      final Configuration conf = new Configuration();
      String listenerGIDParam = null;
      String topicNameParam = null;
      int feedIdParam = -1;

      if (isParamPresent(TOPIC_PARAM_NAME)) {
        topicNameParam = getParamTextValue(TOPIC_PARAM_NAME, 0);
      }

      if (isParamPresent(LISTENER_GROUP_PARAM_NAME)) {
        listenerGIDParam = getParamTextValue(LISTENER_GROUP_PARAM_NAME, 0);
      }

      if (isParamPresent(PARTITION_PARAM_NAME)) {
        feedIdParam = getParamIntValue(PARTITION_PARAM_NAME, 0);
      }
      final String listenerGID = listenerGIDParam;
      final String topicName = topicNameParam;
      final int feedId = feedIdParam;

      final Map<String, List<TopicFeedStatInfo>> topicMap =
          new HashMap<String, List<TopicFeedStatInfo>>();
      final List<CursorInfo> cursorList = new ArrayList<CursorInfo>();

      new FileclientRun(user) {
        @Override
        public void runAsProxyUser() throws CLIProcessingException,IOException {
          try {
            Admin admin = Streams.newAdmin(conf);
            MarlinAdminImpl madmin = (MarlinAdminImpl)admin;
            List<CursorInfo> cursors = madmin.listCursors(streamName, listenerGID,
                                                          topicName, feedId);
            cursorList.addAll(cursors);
          } catch (Exception e) {
            LOG.error("listCursors failed : " + e);
            throw e;
          }
        }
      };

      for (CursorInfo ci : cursorList) {
        TopicFeedStatInfo sinfo = ci.topicFeedInfo().stat();
        long producerMaxSeq = sinfo.getMaxSeq();
        long producerMaxTs = sinfo.getTimeRange().getMaxTS();

        OutputNode cursorNode = new OutputNode();
        cursorNode.addChild(new OutputNode("consumergroup", ci.listenerID()));
        cursorNode.addChild(new OutputNode("topic", ci.topic()));
        cursorNode.addChild(new OutputNode("partitionid",
                                           Integer.toString(ci.feedId())));
        cursorNode.addChild(new OutputNode("produceroffset",
                                           Long.toString(producerMaxSeq)));
        cursorNode.addChild(new OutputNode("committedoffset",
                                           Long.toString(ci.cursor())));
        cursorNode.addChild(new OutputNode("producertimestamp",
                            StreamsCommands.millisToDate(producerMaxTs)));
        cursorNode.addChild(new OutputNode("consumertimestamp",
                            StreamsCommands.millisToDate(ci.timestamp())));
        cursorNode.addChild(new OutputNode("consumerlagmillis",
            Long.toString(producerMaxTs - ci.timestamp())));
        out.addNode(cursorNode);
      }
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    }
  }

  private void deleteCursor(OutputHierarchy out) throws CLIProcessingException {
    try {
      final String user = getUserLoginId();
      final Configuration conf = new Configuration();
      final String streamName = getParamTextValue(PATH_PARAM_NAME, 0);
      String listenerGIDParam = null;
      String topicNameParam = null;
      int feedIdParam = -1;

      if (isParamPresent(TOPIC_PARAM_NAME)) {
        topicNameParam = getParamTextValue(TOPIC_PARAM_NAME, 0);
      }

      if (isParamPresent(LISTENER_GROUP_PARAM_NAME)) {
        listenerGIDParam = getParamTextValue(LISTENER_GROUP_PARAM_NAME, 0);
      }

      if (isParamPresent(PARTITION_PARAM_NAME)) {
        feedIdParam = getParamIntValue(PARTITION_PARAM_NAME, 0);
      }
      final String listenerGID = listenerGIDParam;
      final String topicName = topicNameParam;
      final int feedId = feedIdParam;

      new FileclientRun(user) {
        @Override
        public void runAsProxyUser() throws CLIProcessingException,IOException {
          try {
            Admin admin = Streams.newAdmin(conf);
            MarlinAdminImpl madmin = (MarlinAdminImpl)admin;
            madmin.deleteCursors(streamName, listenerGID, topicName, feedId);
          } catch (Exception e) {
            LOG.error("deleteCursor failed : " + e);
            throw e;
          }
        }
      };

    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    }
  }
}
