/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */
package com.mapr.cli.marlin;

import com.google.common.collect.ImmutableMap;

import com.mapr.baseutils.Errno;
import com.mapr.cli.common.FileclientRun;
import com.mapr.cli.table.TabletStats;
import com.mapr.cliframework.base.*;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.BooleanInputParameter;
import com.mapr.cliframework.base.inputparams.IntegerInputParameter;
import com.mapr.cliframework.base.inputparams.NoValueInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.log4j.Logger;
import com.mapr.streams.Admin;
import com.mapr.streams.Streams;
import com.mapr.streams.impl.admin.MarlinAdminImpl;
import com.mapr.streams.impl.admin.AssignInfo;
import java.net.InetAddress;
import java.math.BigInteger;
import java.io.IOException;
import java.util.List;

public class AssignCommands extends CLIBaseClass implements CLIInterface {

  private static final Logger LOG = Logger.getLogger(AssignCommands.class);
  private static final String LISTENER_GROUP_PARAM_NAME = "consumergroup";
  private static final String PATH_PARAM_NAME = "path";
  private static final String TOPIC_PARAM_NAME = "topic";
  private static final String PARTITION_PARAM_NAME = "partition";
  private static final String DETAIL_PARAM_NAME = "detail";

  private static final CLICommand listCommand =
      new CLICommand(
          "list",
          "usage: stream assign list -path <path> -consumergroup <consumer group Id> -topic <topic> -partition <partitionId>",
          AssignCommands.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
              .put(PATH_PARAM_NAME,
                  new TextInputParameter(PATH_PARAM_NAME,
                      "Stream Path",
                      CLIBaseClass.REQUIRED,
                      null))
              .put(LISTENER_GROUP_PARAM_NAME,
                  new TextInputParameter(LISTENER_GROUP_PARAM_NAME,
                      "Consumer Group ID",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(TOPIC_PARAM_NAME,
                  new TextInputParameter(TOPIC_PARAM_NAME,
                      "Topic Name",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(PARTITION_PARAM_NAME,
                  new IntegerInputParameter(PARTITION_PARAM_NAME,
                      "Partition ID",
                      CLIBaseClass.NOT_REQUIRED,
                      null))
              .put(DETAIL_PARAM_NAME,
                  new NoValueInputParameter(DETAIL_PARAM_NAME,
                      "Detail",
                      CLIBaseClass.NOT_REQUIRED,
                      false))
              .build(),
          null)
          .setShortUsage("stream assign list -path <path>");

  public static final CLICommand assignCommands =
    new CLICommand(
          "assign", "assign [list]",
          CLIUsageOnlyCommand.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          new CLICommand[]{
              listCommand
          }
      ).setShortUsage("stream assign [list]");

  public AssignCommands(ProcessedInput input, CLICommand cliCommand) {
    super(input, cliCommand);
  }

  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    if (!super.validateInput()) {
      return output;
    }

    String cmdName = cliCommand.getCommandName();
    if (cmdName.equalsIgnoreCase(listCommand.getCommandName())) {
      listTopics(out);
    }
    return output;
  }


  private void listTopics(OutputHierarchy out) throws CLIProcessingException {
    try {
      final String streamName = getParamTextValue(PATH_PARAM_NAME, 0);
      final Configuration conf = new Configuration();
      String listenerGID = null;
      String topicName = null;
      int feedId = -1;

      if (isParamPresent(TOPIC_PARAM_NAME)) {
        topicName = getParamTextValue(TOPIC_PARAM_NAME, 0);
      }

      if (isParamPresent(LISTENER_GROUP_PARAM_NAME)) {
        listenerGID = getParamTextValue(LISTENER_GROUP_PARAM_NAME, 0);
      }

      boolean detail = isParamPresent(DETAIL_PARAM_NAME);
      Admin admin = Streams.newAdmin(conf);
      MarlinAdminImpl madmin = (MarlinAdminImpl)admin;
      List<AssignInfo> assignList = madmin.listAssigns(streamName,
          listenerGID, topicName);

      for (AssignInfo ci : assignList) {
        String [] listenerList = ci.listeners();
        for (int i = 0; i < ci.numListeners(); i++) {
          List<Integer> assignments = ci.listenerAssignment(i);

          OutputNode assignNode = new OutputNode();
          assignNode.addChild(new OutputNode("consumergroup", ci.listenerID()));
          assignNode.addChild(new OutputNode("topic", ci.topic()));
          if (detail)
            assignNode.addChild(new OutputNode("assignseqnum",
                                               ci.assignSeqNum()));

          String[] consumer = listenerList[i].split(":", 4);
          int ipAdd = Integer.parseInt(consumer[1]);
          byte[] bytes = BigInteger.valueOf(ipAdd).toByteArray();
          InetAddress addr = InetAddress.getByAddress(bytes);

          if (detail)
            assignNode.addChild(new OutputNode("consumerguid", consumer[0]));

          assignNode.addChild(new OutputNode("consumer", consumer[3]));
          assignNode.addChild(new OutputNode("consumerip",
                                             addr.getHostAddress()));
          assignNode.addChild(new OutputNode("consumerpid", consumer[2]));

          String assignStr = "";
          boolean isFirst = true;
          for (Integer p : assignments) {
            if (isFirst) {
              assignStr = "" + p;
              isFirst = false;
            } else {
              assignStr = assignStr + "," + p;
            }
          }
          assignNode.addChild(new OutputNode("assignment", assignStr));
          out.addNode(assignNode);
        }
      }
    } catch (CLIProcessingException e) {
      out.addError(new OutputError(Errno.EINVAL, e.getMessage()));
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED, e.getMessage()));
    }
  }
}
