package com.mapr.cli.common;

import com.google.protobuf.InvalidProtocolBufferException;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.fs.Rpc;
import com.mapr.fs.cldb.util.Util;
import com.mapr.fs.cldb.proto.CLDBProto.FileServerInfo;
import com.mapr.fs.proto.Common;
import com.mapr.fs.proto.Common.IPAddress;
import com.mapr.fs.proto.Security.ServerKeyType;
import com.mapr.fs.proto.Security.TicketAndKey;
import com.mapr.fs.proto.clustermetrics.ClusterMetricsProto.CommandId;
import com.mapr.fs.proto.clustermetrics.ClusterMetricsProto.ExecuteCommandRequest;
import com.mapr.fs.proto.clustermetrics.ClusterMetricsProto.ExecuteCommandResponse;
import com.mapr.fs.proto.clustermetrics.ClusterMetricsProto.MetricProcId;
import com.mapr.login.client.MapRLoginHttpsClient;
import com.mapr.security.JNISecurity;
import com.mapr.security.MutableInt;
import com.mapr.security.Security;
import org.apache.log4j.Logger;

import java.io.IOException;
import java.util.concurrent.ConcurrentHashMap;
import java.util.List;
import java.util.Map;

/**
 * Executes a remote command by sending an RPC to remote hoststats process.
 * Returns the stdout output of the command.
 * <p/>
 * Author: smarella
 */
public class RemoteCommandExecutor {
  private static final Logger LOG = Logger.getLogger(RemoteCommandExecutor.class);
  private static final int HOSTSTATS_RPC_PORT = 1111;
  private static final String CLUSTER = CLDBRpcCommonUtils.getInstance().getCurrentClusterName();
  private static Map<String, String> hostNameToMapRIpCache = new ConcurrentHashMap<String, String>();

  public static List<String> execute(String remoteHost, ExecuteCommandRequest request) throws CLIProcessingException {
    try {

      // attempt to authenticate
      MapRLoginHttpsClient loginClient = new MapRLoginHttpsClient();
      loginClient.authenticateIfNeeded(CLUSTER);
      String hostName = remoteHost;

      if (hostNameToMapRIpCache.containsKey(remoteHost)) {
        if (LOG.isDebugEnabled()) {
          LOG.debug("Will use " + hostNameToMapRIpCache.get(remoteHost) + " instead of " + remoteHost + " for RPC");
        }
        remoteHost = hostNameToMapRIpCache.get(remoteHost);
      }

      CommandId commandId = request.getCommandId();
      long hsBinding = createBinding(remoteHost);
      byte[] retBytes = Rpc.sendRequest(hsBinding,
          Common.MapRProgramId.HoststatsProgramId.getNumber(),
          MetricProcId.ExecuteCommandProc.getNumber(),
          request);
  
      if (retBytes == null) { 
        IPAddress.Builder ipAddressBuilder = IPAddress.newBuilder();
        ipAddressBuilder.setHostname(remoteHost);
        IPAddress fileServer = ipAddressBuilder.build();
        
        FileServerInfo remoteFsInfo = NodesCommonUtils.getFileServerInfo(fileServer, request.getCreds(), null);
        if (remoteFsInfo == null || remoteFsInfo.getAddressCount() == 0) {
          LOG.error("FileServerLookup for " + remoteHost + " did not return any valid ip addresses." + 
            " The node possibly does not belong to cluster '" + CLUSTER + "'");
          throw new CLIProcessingException("RPC to execute '" + commandId.name() + "' on node: "
              + remoteHost + " returned no data. The node may no longer be part of cluster '" + CLUSTER + "'");
        } else {
         // can pickup the first entry because MapR RPC listens on all the interfaces included in MAPR_SUBNETS
          for (IPAddress server : remoteFsInfo.getAddressList()) {
            if (LOG.isDebugEnabled()) {
              LOG.debug("Trying RPC to hoststats on " + Util.intToIp(server.getHost()));
            }
            hsBinding = createBinding(Util.intToIp(server.getHost()));
            retBytes = Rpc.sendRequest(hsBinding,
                Common.MapRProgramId.HoststatsProgramId.getNumber(),
                MetricProcId.ExecuteCommandProc.getNumber(),
                request);
            if (retBytes != null) {
              if (LOG.isDebugEnabled()) {
                LOG.debug("Caching entry " + hostName + " -> " + Util.intToIp(server.getHost()));
              }
              hostNameToMapRIpCache.put(hostName, Util.intToIp(server.getHost()));
              break;
            }
          }
        }
      }

      if (retBytes != null) {
        ExecuteCommandResponse response = ExecuteCommandResponse.parseFrom(retBytes);
        int status = response.getStatus();
        if (status != 0) {
          String msg = "Operation '" + commandId.name() + "' on node: "
              + remoteHost + " failed with status: " + status;
          if (response.hasErrorMsg()) {
            msg += ". Reason: '" + response.getErrorMsg() + "'";
          }
          throw new CLIProcessingException(msg);
        } else {
          return response.getStdoutLinesList();
        }
      } else {
        throw new CLIProcessingException("RPC to execute '" + commandId.name() + "' on node: "
            + remoteHost + " returned no data.");
      }
    } catch (InvalidProtocolBufferException e) {
      throw new CLIProcessingException("Error parsing the RPC response");
    } catch (Exception e) {
      if (e instanceof CLIProcessingException) {
        throw (CLIProcessingException) e;
      } else {
        throw new CLIProcessingException(e);
      }
    }
  }

  private static long createBinding(String remoteHost) throws Exception {
    int host = CLDBRpcCommonUtils.ipToInt(CLDBRpcCommonUtils.convertHostToIp(remoteHost));

    try {
      int result = Rpc.initialize(0, 0, CLUSTER); // Client
      if (result < 0) {
        throw new IOException("Error in RPC init");
      }
    } catch (Exception e) {
      throw new IOException("Exception in Rpc.initialize " + e);
    }
    if (JNISecurity.IsSecurityEnabled(CLUSTER)) {
      MutableInt mErr = new MutableInt();
      TicketAndKey ticketAndKey = Security.GetTicketAndKeyForCluster(ServerKeyType.ServerKey, CLUSTER, mErr);
      if (mErr.GetValue() != 0 || !Security.IsTicketAndKeyUsable(ticketAndKey)) {
        LOG.warn("Error while trying to get correct Ticket with errorCode: " + mErr.GetValue());
        return -1;
      }
    }
    return Rpc.createBindingFor(host, HOSTSTATS_RPC_PORT, CLUSTER, ServerKeyType.ServerKey.getNumber());
  }

}
