/**
 * Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved
 * User: agliga
 */
package com.mapr.cli.common;

import com.google.common.collect.ImmutableMap;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.google.protobuf.InvalidProtocolBufferException;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.util.FieldInfo;
import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.cldb.proto.CLDBProto.AlarmViewRequest;
import com.mapr.fs.cldb.proto.CLDBProto.AlarmViewResponse;
import com.mapr.fs.cldb.topology.Node;
import com.mapr.fs.cli.proto.CLIProto.Limiter;
import com.mapr.fs.proto.Common;
import com.mapr.fs.proto.Security.CredentialsMsg;
import com.mapr.fs.proto.Common.AlarmId;
import com.mapr.fs.proto.Common.PluggableAlarm;
import com.mapr.cli.common.NodeField;
import com.mapr.cli.common.VolumeField;
import com.mapr.fs.cldb.alarms.PluggableAlarms;

import com.mapr.security.MaprSecurityException;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.HashSet;
import java.util.Map;

import org.apache.log4j.Logger;

/**
 * Pluggable alarms util. Used to get a list of pluggable alarms from CLDB
 */
public class PluggableAlarmUtil {

  // Fixed max amount of records per RPC
  private static final int NUM_ALARMS_PER_RPC = 100;

  private static final Logger LOG = Logger.getLogger(PluggableAlarmUtil.class);
  private static Set<VolumeField> volumeAlarmsAdded  = new HashSet<VolumeField>();
  private static Set<NodeField> nodeAlarmsAdded  = new HashSet<NodeField>();
  private static List<PluggableAlarm> pluggableAlarms = null;
  /**
   * Method to get all the pluggable alarms from the CLDB
   * @param creds The user credentials to use
   * @param alarmName The alarmName to get specifically. If null, then gets them all
   * @param start The first record to start at. Default should be 0
   * @param limit The last record. Default should be 50. Uses NUM_ALARMS_PER_RPC if greater than it's value
   * @return The List of all PluggableAlarms retrieved
   * @throws CLIProcessingException Throws exception on any error, such as connecting to CLDB.
   */
  public static List<PluggableAlarm> getAlarms(CredentialsMsg creds, String alarmName, int start, int limit) throws CLIProcessingException {

    if (pluggableAlarms == null) {

      AlarmViewRequest req =
          AlarmViewRequest.newBuilder()
              .setAlarmname(alarmName)
              .setCreds(creds)
              .setLimiter(Limiter.newBuilder()
                  .setStart(start)
                  .setLimit(limit > NUM_ALARMS_PER_RPC ? NUM_ALARMS_PER_RPC : limit)
                  .build())
              .build();

      AlarmViewResponse resp = null;
      byte[] replyData;

      try {

        replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProto.CLDBProg.AlarmViewProc.getNumber(),
            req, AlarmViewResponse.class);
        if (replyData == null) {
          /**
           * <MAPR_ERROR>
           * Message:Failed <error>
           * Function:AlarmCommands.executeRealCommand()
           * Meaning:An error occurred.
           * Resolution:Contact technical support.
           * </MAPR_ERROR>
           */
          throw new CLIProcessingException("Failed to connect to CLDB");
        } else {
          resp = AlarmViewResponse.parseFrom(replyData);

          if (resp != null) {
            pluggableAlarms = resp.getAlarmsList();
          } else {
            throw new CLIProcessingException("Reply data did not get retrieved properly.");
          }
        }


      } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
            "MaprSecurityException " + "Exception", e);

      } catch (InvalidProtocolBufferException ipbe) {
        throw new CLIProcessingException("Exception while parsing the RPC "
            + "response data into AlarmViewResponse proto object.", ipbe);
      } catch (Exception e) {
        /**
         * <MAPR_ERROR>
         * Message:Failed due to exception
         * Function:AlarmCommands.executeRealCommand()
         * Meaning:An error occurred.
         * Resolution:Contact technical support.
         * </MAPR_ERROR>
         */
        LOG.error("Exception: " + e.getLocalizedMessage());
        throw new CLIProcessingException("Failed due to exception", e);
      }

    }

    return pluggableAlarms;
  }

  /**
   * Method that calls getAlarms with the given values. Will call getAlarms with null alarmName
   * @param creds User credentials to use
   * @param start The first record to start at. Default should be 0
   * @param end The last record. Default should be 50. Uses NUM_ALARMS_PER_RPC if greater than it's value
   * @return The List of all PluggableAlarms retrieved
   * @throws CLIProcessingException Throws exception on any error, such as connecting to CLDB.
   */
  public static List<PluggableAlarm> getAlarms(CredentialsMsg creds, int start, int end) throws CLIProcessingException{
    return getAlarms(creds, "", start, end);
  }
  public static List<PluggableAlarm> getAlarmsByType(CredentialsMsg creds, int start, int end, String type) throws CLIProcessingException{
    List<PluggableAlarm> returnAlarms = new ArrayList<PluggableAlarm>();
    for (PluggableAlarm pluggableAlarm : getAlarms(creds, start, end)) {
      if (PluggableAlarms.checkType(pluggableAlarm, type)) {
        returnAlarms.add(pluggableAlarm);
      }

    }
    return returnAlarms;
  }

  public static List<PluggableAlarm> getVolumeAlarms(CredentialsMsg creds, int start, int end) throws CLIProcessingException{
      return getAlarmsByType(creds, start, end, "VOLUME");
  }
  public static List<PluggableAlarm> getNodeAlarms(CredentialsMsg creds, int start, int end) throws CLIProcessingException{
      return getAlarmsByType(creds, start, end, "NODE");
  }



  /**
   * Method that calls getAlarms with the given values. Will call getAlarms with 0 start and 1 limit
   * @param creds User credentials to use
   * @param alarmName The alarmName to get specifically. If null, then gets them all
   * @return The List of all PluggableAlarms retrieved
   * @throws CLIProcessingException Throws exception on any error, such as connecting to CLDB.
   */
  public static List<PluggableAlarm> getAlarms(CredentialsMsg creds, String alarmName) throws CLIProcessingException {
    return getAlarms(creds, alarmName, 0, 1);

  }

  private static List<PluggableAlarm> getAlarmsInternally(CredentialsMsg creds) {
    List<PluggableAlarm> pluggableAlarmsList;
    try {
      pluggableAlarmsList = getAlarms(creds, "", 0, 50);
    } catch (CLIProcessingException e) {
      LOG.error(e.getMessage());
      pluggableAlarmsList = new ArrayList<PluggableAlarm>();
    }
    return pluggableAlarmsList;

  }

  private static void logMaps(Map<VolumeField, FieldInfo> volMap,  Map<NodeField, FieldInfo> nodeMap) {
    String constMsg = "logMaps: [" + Thread.currentThread().getId() + "] ";
    LOG.error(constMsg + "entries in pluggableAlarms " + pluggableAlarms.size());
    StringBuilder msgStr = new StringBuilder();
    for (PluggableAlarm pa : pluggableAlarms) {
      msgStr.append(constMsg + "PluggableAlarms dump: id:" + pa.getId() +
          ", name(key):" + pa.getName() + ", display:" + pa.getDisplayName()
          );
      if (volMap != null) {
        VolumeField vf = new VolumeField(pa);
        msgStr.append(", VolumeField hash:[" + vf.hashCode() + "]");
      }
      if (nodeMap != null) {
        NodeField nf = new NodeField(pa);
        msgStr.append(", NodeField hash:[" + nf.hashCode() + "]");
      }
      LOG.error(constMsg + msgStr.toString());
      msgStr.setLength(0);
    }

    if (volMap != null) {
      LOG.error(constMsg + "entries in volumeMap " + volMap.size());
      for (VolumeField vf : volMap.keySet()) {
        LOG.error(constMsg + "volumeMap dump: id:" + vf.getId() +
            ", lable:" + vf.getLabel() + ", hash:[" + vf.hashCode() + "]");
      }
    }

    if (nodeMap != null) {
      LOG.error(constMsg + "entries in nodeMap " + nodeMap.size());
      for (NodeField nf : nodeMap.keySet()) {
        LOG.error(constMsg + "nodeMap dump: id:" + nf.getId() +
            ", lable:" + nf.getLabel() + ", hash:[" + nf.hashCode() + "]");
      }
    }
  }

  /**
   * Map volume type alarms to the volume immutable map
   * @param creds
   * @param table
   * @return
   */
  public static Map<VolumeField, FieldInfo> appendVolumeMap(CredentialsMsg creds, ImmutableMap.Builder<VolumeField, FieldInfo> table) {
    Map<VolumeField, FieldInfo> volMap = table.build();
    List<PluggableAlarm> pluggableAlarmList = getAlarmsInternally(creds);
    String type = "VOLUME";
    synchronized (volumeAlarmsAdded) {
      for (PluggableAlarm pluggableAlarm : pluggableAlarmList) {
        if (PluggableAlarms.checkType(pluggableAlarm, type)) {
          VolumeField volumeField = new VolumeField(pluggableAlarm);
          if (volumeAlarmsAdded.add(volumeField)) {
            table.put(volumeField,
                new FieldInfo(pluggableAlarm.getId(), pluggableAlarm.getTerse(), pluggableAlarm.getDisplayName(),
                  Integer.class));

          }
        }
      }
    }

    try {
      return table.build();
    } catch (java.lang.IllegalArgumentException e) {
      LOG.error("Exception: [bug 21946: contact mapr support] appendVolumeMap: " + e.getLocalizedMessage());
      logMaps(volMap, null);
    }
    return table.build();
  }

  public static Map<NodeField, FieldInfo> appendNodeMap(CredentialsMsg creds, ImmutableMap.Builder<NodeField, FieldInfo> table) {
    Map<NodeField, FieldInfo> nodeMap = table.build();
    List<PluggableAlarm> pluggableAlarmList = getAlarmsInternally(creds);
    String type = "NODE";
    synchronized (nodeAlarmsAdded) {
      for (PluggableAlarm pluggableAlarm : pluggableAlarmList) {
        if (PluggableAlarms.checkType(pluggableAlarm, type)) {
          NodeField nodeField = new NodeField(pluggableAlarm);
          if (nodeAlarmsAdded.add(nodeField)) {
            table.put(nodeField,
                new FieldInfo(pluggableAlarm.getId(), pluggableAlarm.getTerse(), pluggableAlarm.getDisplayName(),
                  Integer.class));

          }
        }
      }
    }

    try {
      return table.build();
    } catch (java.lang.IllegalArgumentException e) {
      LOG.error("Exception: [bug 21946: contact mapr support] appendNodeMap: " + e.getLocalizedMessage());
      logMaps(null, nodeMap);
    }
    return table.build();
  }

  /**
   * NOTE: This method is in com.mapr.fs.cldb.alarms.PluggableAlarms as well
   * Static method to get the AlarmId object from a given alarm string
   * @param alarmName The alarm name to look up in AlarmId enum object
   * @return The AlarmId associated with the given string. Null if not found (for pluggable alarms)
   */
  public static AlarmId getAlarmId(String alarmName) {
    for (AlarmId aid : AlarmId.values()) {
      if (aid.name().equals(alarmName)) {
        return AlarmId.valueOf(alarmName);
      }
    }
    return null;
  }

  public static PluggableAlarm getAlarmByName(CredentialsMsg creds, String alarmName) {
    PluggableAlarm ret = null;
    for(PluggableAlarm pluggableAlarm : getAlarmsInternally(creds)) {
      if (alarmName.equalsIgnoreCase(pluggableAlarm.getName())) {
        ret = pluggableAlarm;
        break;
      }
    }
    return ret;
  }

  public static int getMaxNumNodes(CredentialsMsg creds, int currentMax) {
    List<PluggableAlarm> pluggableAlarmList = getAlarmsInternally(creds);


    for (PluggableAlarm pluggableAlarm : pluggableAlarmList) {
      if (pluggableAlarm.getId() > currentMax) {
        currentMax = pluggableAlarm.getId();
      }

    }

    return currentMax;

  }

  /**
   * Method used to reset pluggableAlarms cache.
   * It will be called by maprcli alarm view (used to restart webserver cache, since maprcli is held in memory by webserver process)
   */
  public static void resetCache() {
    pluggableAlarms = null;
  }

}
