package com.mapr.cli.common;

import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.yarn.conf.HAUtil;
import org.apache.hadoop.yarn.conf.YarnConfiguration;
import org.apache.hadoop.yarn.proto.YarnServerResourceManagerServiceProtos;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.ZooKeeper;
import org.apache.zookeeper.data.Stat;

import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.baseutils.zookeeper.ZKUtils;
import com.mapr.cli.MapRCliUtil;
import com.mapr.cli.ServerCommands;
import com.mapr.cliframework.base.CLICommandFactory;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;

import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.cldb.proto.CLDBProto.FileServerInfo;
import com.mapr.fs.cldb.proto.CLDBProto.FileServerLookupRequest;
import com.mapr.fs.cldb.proto.CLDBProto.FileServerLookupResponse;
import com.mapr.fs.cldb.util.Util;
import com.mapr.fs.proto.Common;
import com.mapr.fs.proto.Common.IPAddress;
import com.mapr.fs.proto.Security.CredentialsMsg;
import com.mapr.security.MaprSecurityException;
import com.mapr.util.zookeeper.ZKDataRetrieval;
import com.google.protobuf.InvalidProtocolBufferException;

import com.mapr.fs.proto.Common.ServiceData;
import com.google.protobuf.InvalidProtocolBufferException;

public class NodesCommonUtils {

	public static final String CLDB_SERVICE_MASTER_PATH = "/services/cldb/" + ZKUtils.SERVICE_MASTER_NODE;
	public static final String SERVER_PATH = "/servers";
	public static final String SERVICES_PATH = "/services";
	public static final String CONF_SERVICES_PATH = "/services_config";
	public static final String ZK_CONNECTSTRING = "zkconnect";

	public static final int TIMEOUT_SERVER = 30 * 1000;
	public static final String[] localAddresses = new String [] {"::1", "0:0:0:0:0:0:0:1", "127."};

	private static NodesCommonUtils s_instance = new NodesCommonUtils();
	private static Map<String, ZKDataRetrieval> zkMap = new ConcurrentHashMap<String, ZKDataRetrieval>(); 
	

	private NodesCommonUtils() {
	}
	
	public static NodesCommonUtils getInstance() {
		return s_instance;
	}

	public static final Log LOG = LogFactory.getLog(NodesCommonUtils.class);
	
	public static CommandOutput executeNodeListCommand(String args[]) throws CLIProcessingException {
		ProcessedInput input = new ProcessedInput(args);
		
		CLIInterface commandIFace = null;
		try {
			commandIFace = CLICommandFactory.getInstance().getCLI(input);
		} catch (CLIProcessingException e) {
			/**
			* <MAPR_ERROR>
			* Message:Exception during search for command Interface
			* Function:NodesCommonUtils.executeNodeListCommand()
			* Meaning:An error occurred.
			* Resolution:Contact technical support.
			* </MAPR_ERROR>
			*/
			LOG.error("Exception during search for command Interface", e);
			throw e;
		}

		if ( commandIFace == null ) {
			/**
			* <MAPR_ERROR>
			* Message:Could not find Interface for a command: <command>
			* Function:NodesCommonUtils.executeNodeListCommand()
			* Meaning:The interface for the command does not appear to exist.
 			* Resolution:Check the command syntax and try again.
			* </MAPR_ERROR>
			*/
			LOG.error("Could not find Interface for a command: " + input.getCommandName());
			throw new CLIProcessingException("Could not find Interface for a command: " + input.getCommandName());
		}
					
		try {
			CommandOutput output = commandIFace.executeCommand();
			return output;
		} catch (CLIProcessingException e) {
			/**
			* <MAPR_ERROR>
			* Message:Exception during search for command Interface
			* Function:NodesCommonUtils.executeNodeListCommand()
			* Meaning:An error occurred.
			* Resolution:Contact technical support.
			* </MAPR_ERROR>
			*/
			LOG.error("Exception during command execution", e);
			throw e;
		}	

	}

	public static String [] prepareParamsForNodeListCommand(String filter, String zkConnectString) {
		List<String> sb = new ArrayList<String>();
		sb.add("node");
		sb.add("list");
		if ( zkConnectString != null ) {
			sb.add("-"+ZK_CONNECTSTRING);
			sb.add(zkConnectString);
		}
		
		sb.add("-"+ServerCommands.COLUMNS_PARAM_NAME);
		// names of the columns
		sb.add("hn,ip");
		sb.add("-"+ServerCommands.FILTER_PARAM_NAME); 
		sb.add(filter);
		
		return sb.toArray(new String[]{});
	}
	
	public static List<String> findNodeIps(String zkConnectString) {
		ZKDataRetrieval currentZK = zkMap.get(zkConnectString);
		if ( currentZK == null ) {
			connect(zkConnectString);
		}
		currentZK = zkMap.get(zkConnectString);
		// if still null - return empty map
		if ( currentZK ==  null ) {
			LOG.warn("Unable to get ZK based on: " + zkConnectString);
			return new ArrayList<String>();
		} 
		return currentZK.getServersInfo();
	}
	
	@Deprecated
	public List<String> findNodeIps(ZooKeeper s_zk) {

		List<String> nodes = new ArrayList<String>();
		try {
			List<String> znodes = s_zk.getChildren(SERVER_PATH, false);
			for ( String znode : znodes ) {
				byte[] znodeData = s_zk.getData(SERVER_PATH + "/" + znode, false, null);
				String znodeName = new String(znodeData);
				nodes.add(znodeName);
			}
			return nodes;
		} catch (KeeperException e) {
			/**
			* <MAPR_ERROR>
			* Message:Cannot get children of <server path> with error: <error>
			* Function:NodesCommonUtils.findNodeIps()
			* Meaning:An error occurred.
			* Resolution:Contact technical support.
			* </MAPR_ERROR>
			*/
			LOG.error("Cannot get children of " + SERVER_PATH + " with error: "+ e.getLocalizedMessage());
			return null;
		} catch (InterruptedException e) {
			/**
			* <MAPR_ERROR>
			* Message:Cannot get children of <server path> with error: <error>
			* Function:NodesCommonUtils.findNodeIps()
			* Meaning:An error occurred.
			* Resolution:Contact technical support.
			* </MAPR_ERROR>
			*/
			LOG.error("Cannot get children of " + SERVER_PATH + " with error: "+ e.getLocalizedMessage());
			return null;
		}

	}
	
	public static List<String> findFilteredNodeIps(String filter, String zkConnectString) throws CLIProcessingException {
		List<String> nodeHostNames = new ArrayList<String>();
		filter = filter.replace("*", ".*");
		// prepare params for node list command
		String [] args = NodesCommonUtils.prepareParamsForNodeListCommand(filter, zkConnectString);
		CommandOutput coNodeList = NodesCommonUtils.executeNodeListCommand(args);
		List<OutputNode> outNodes = coNodeList.getOutput().getOutputNodes();
		for ( OutputNode outNode : outNodes ) {
			List<OutputNode> children = outNode.getChildren();
			// should be two children as was asking for nodeName and IP
			for (OutputNode childNode : children ) {
				if ( childNode.getValue() != null ) {
					nodeHostNames.add((String)childNode.getValue());
					break;
				}
			}
		}	
		return nodeHostNames;
	}
	
	public static List<String> convertHostToIp(List<String> nodeHostNames) {
		// convert if needed hostNames to IPs
		List<String> nodeNames = new ArrayList<String>();
		for ( String host : nodeHostNames ) {
			try {
				InetAddress[] ias = InetAddress.getAllByName(host);
				if ( ias != null ) {
					for (InetAddress ia : ias ) {
						String ip = ia.getHostAddress();
						boolean isLocalAddress  = false;
						for ( String localIp : localAddresses) {
							if ( ip.startsWith(localIp)) {
								isLocalAddress = true;
								break;
							}
						}
						if ( !isLocalAddress ) {
							nodeNames.add(ip);								
						}
					}
				}
			} catch (UnknownHostException e) {
				/**
				* <MAPR_ERROR>
				* Message:Can not find IP for host: <host>
				* Function:NodesCommonUtils.convertHostToIp()
				* Meaning:An error occurred.
				* Resolution:Contact technical support.
				* </MAPR_ERROR>
				*/
				LOG.error("Can not find IP for host: " + host, e);
				
			}
		}
		return nodeNames;
	}
	
	public static List<String> convertHostToIpIncludingLocal(List<String> nodeHostNames) {
		// convert if needed hostNames to IPs
		List<String> nodeNames = new ArrayList<String>();
		for ( String host : nodeHostNames ) {
			try {
				InetAddress[] ias = InetAddress.getAllByName(host);
				if ( ias != null ) {
					for (InetAddress ia : ias ) {
						String ip = ia.getHostAddress();
							nodeNames.add(ip);								
					}
				}
			} catch (UnknownHostException e) {
				/**
				* <MAPR_ERROR>
				* Message:Can not find IP for host: <host>
				* Function:NodesCommonUtils.convertHostToIp()
				* Meaning:An error occurred.
				* Resolution:Contact technical support.
				* </MAPR_ERROR>
				*/
				LOG.error("Can not find IP for host: " + host, e);
				
			}
		}
		return nodeNames;
	}

	public static List<String> convertIpToHost(List<String> nodeIpNames) {
		List<String> nodeNames = new ArrayList<String>();
		for ( String host : nodeIpNames ) {
			try {
				InetAddress[] ias = InetAddress.getAllByName(host);
				if ( ias != null ) {
					for (InetAddress ia : ias ) {
						String ip = ia.getCanonicalHostName();
						nodeNames.add(ip);
					}
				}
			} catch (UnknownHostException e) {
				/**
				* <MAPR_ERROR>
				* Message:Can not find host for IP: <host>
				* Function:NodesCommonUtils.convertIpToHost()
				* Meaning:An error occurred.
				* Resolution:Contact technical support.
				* </MAPR_ERROR>
				*/
				LOG.error("Can not find host for IP: " + host, e);
				
			}
		}
		return nodeNames;	
	}
	
	public static ServiceData getServiceMasterData(String zkConnectString, String serviceName) {
		ZKDataRetrieval currentZK = zkMap.get(zkConnectString);
		if ( currentZK == null ) {
			connect(zkConnectString);
		}
		currentZK = zkMap.get(zkConnectString);
		// if still null - return empty map
		if ( currentZK ==  null ) {
			LOG.warn("Unable to get ZK based on: " + zkConnectString);
			return null;
		} 
		ServiceData data = currentZK.getServiceToMasterMap().get(serviceName);
		if ( data == null && "fileserver".equalsIgnoreCase(serviceName)) {
			return currentZK.getServiceToMasterMap().get("kvstore");
		} else {
			return data;
		}
	}
	
	public static boolean isServiceAvailable(String zkConnectString, String serviceName) {
	  ServiceData hostInfo =
        NodesCommonUtils.getServiceMasterData(zkConnectString, serviceName);
        
        if ( hostInfo != null ) {
          if (hostInfo.hasIsRunning() && hostInfo.getIsRunning()) {
            // Service is running
            return true;
          }
        }
        return false;
	}

  public static Map<String, ServiceData> getServiceNodeData(String zkConnectString, String serviceName) {
		ZKDataRetrieval currentZK = zkMap.get(zkConnectString);
		if ( currentZK == null ) {
			connect(zkConnectString);
		}
		currentZK = zkMap.get(zkConnectString);
		// if still null - return empty map
		if ( currentZK ==  null ) {
			LOG.warn("Unable to get ZK based on: " + zkConnectString);
			return null;
		}

		Map<String, ServiceData> nodesData = currentZK.getServiceToNodeMap().get(serviceName);
		if ("fileserver".equalsIgnoreCase(serviceName)) {
		  //Get union of kvstore(from non-cldb nodes) and filserver(from cldb nodes)
		  if(nodesData!= null)
		    nodesData.putAll(currentZK.getServiceToNodeMap().get("kvstore"));
		  else
		    return currentZK.getServiceToNodeMap().get("kvstore");
		}
		  return nodesData;
		}
	
	public static Map<String, Properties> getServiceNodesProperties(String zkConnectString, String serviceName) {
		ZKDataRetrieval currentZK = zkMap.get(zkConnectString);
		if ( currentZK == null ) {
			connect(zkConnectString);
		}
		currentZK = zkMap.get(zkConnectString);
		// if still null - return empty map
		if ( currentZK ==  null ) {
			LOG.warn("Unable to get ZK based on: " + zkConnectString);
			return null;
		} 
		Map<String, Properties> servicePropsMap = currentZK.getServiceToPropertiesMap().get(serviceName);
		if ( "fileserver".equalsIgnoreCase(serviceName)) {
			// get kvstore data as well
			Map<String, Properties> servicePropsMapKv = currentZK.getServiceToPropertiesMap().get("kvstore");
			if ( servicePropsMap == null ) {
				return servicePropsMapKv;
			} else if ( servicePropsMapKv != null ){
				servicePropsMap.putAll(servicePropsMapKv);
				return servicePropsMap;
			}
		}
		return servicePropsMap;
	}
	
	public static Map<String,List<String>> serviceToNodesMap(String zkConnectString) throws CLIProcessingException {
	  if ( LOG.isDebugEnabled()) {
	    LOG.debug("serviceToNodesMap start");
	  }
	  ZKDataRetrieval currentZK = zkMap.get(zkConnectString);
	  if ( currentZK == null ) {
	    connect(zkConnectString);
	  }
	  currentZK = zkMap.get(zkConnectString);
	  // if still null - return empty map
	  if ( currentZK ==  null ) {
	    LOG.warn("Unable to get ZK based on: " + zkConnectString);
	    return new HashMap<String,List<String>>();
	  } 
	  // BEWARE that fileserver and kvstore treated as different services
	  // so you may need to combine data for both of them if needed
	  Map<String,List<String>> servicesMap = currentZK.getServicesMap();
	  return servicesMap;
	}
	
	public static Map<String,List<String>> findServicesRunningHierarchy(String zkConnectString) throws CLIProcessingException {
		if ( LOG.isDebugEnabled()) {
			LOG.debug("findServicesRunningHierarchy start");
		}
	ZKDataRetrieval currentZK = zkMap.get(zkConnectString);
	if ( currentZK == null ) {
		connect(zkConnectString);
	}
	currentZK = zkMap.get(zkConnectString);
	// if still null - return empty map
	if ( currentZK ==  null ) {
		LOG.warn("Unable to get ZK based on: " + zkConnectString);
		return new HashMap<String,List<String>>();
	} 
	Map<String,List<String>> servicesMap = currentZK.getServicesMap();
	Map<String,List<String>> nodeToServiceMap = new HashMap<String,List<String>>();
	for ( Map.Entry<String,List<String>> entry : servicesMap.entrySet()) {
		String service = entry.getKey();
		List<String> nodes = entry.getValue();
		for ( String node : nodes ) {
			if ( !nodeToServiceMap.containsKey(node)) {
				nodeToServiceMap.put(node, new ArrayList<String>());
			}
			// kind of an exceptional case as kvstore is a special instance of fileserver where cldb info is stored
			if ( "kvstore".equalsIgnoreCase(service) ) { 
				 service = "fileserver";
			}
			nodeToServiceMap.get(node).add(service);
		}
	}
	if ( LOG.isDebugEnabled() ) {
	LOG.debug("findServicesRunningHierarchy end");
	}
	return nodeToServiceMap;
	}
	
  public static Map<String, List<String>> findServicesConfiguredHierarchy(String zkConnectString)
      throws CLIProcessingException {
    if (LOG.isDebugEnabled()) {
      LOG.debug("findServicesConfiguredHierarchy start");
    }
    ZKDataRetrieval currentZK = zkMap.get(zkConnectString);
    if (currentZK == null) {
      connect(zkConnectString);
    }
    currentZK = zkMap.get(zkConnectString);
    // if still null - return empty map
    if (currentZK == null) {
      LOG.warn("Unable to get ZK based on: " + zkConnectString);
      return new HashMap<String, List<String>>();
    }
    Map<String, List<String>> configServicesMap = currentZK.getConfigServicesMap();
    Map<String, List<String>> nodeToConfigServicesMap = new HashMap<String, List<String>>();
    for (Map.Entry<String, List<String>> entry : configServicesMap.entrySet()) {
      String service = entry.getKey();
      List<String> nodes = entry.getValue();
      for (String node : nodes) {
        if (!nodeToConfigServicesMap.containsKey(node)) {
          nodeToConfigServicesMap.put(node, new ArrayList<String>());
        }
        // kind of an exceptional case as kvstore is a special instance of
        // fileserver where cldb info is stored
        if ("kvstore".equalsIgnoreCase(service)) {
          service = "fileserver";
        }
        nodeToConfigServicesMap.get(node).add(service);
      }
    }
    if (LOG.isDebugEnabled()) {
      LOG.debug("findServicesConfiguredHierarchy end");
    }
    return nodeToConfigServicesMap;
  }
  
	/**
	 * Return full blown hierarchy
	 * @param s_zk
	 * @return
	 * @throws CLIProcessingException
	 * 
	 */
	@Deprecated
	public static Map<String,List<String>> findServicesRunningHierarchy(ZooKeeper s_zk) throws CLIProcessingException {
		Map<String,List<String>> servicesMap = new HashMap<String,List<String>>();
		try {
			if ( s_zk == null ) {
				return servicesMap;
			}
			Stat stats = s_zk.exists(SERVICES_PATH, false);
			if ( stats == null ) {
				return servicesMap;
			}
			if ( LOG.isDebugEnabled()) {
			 LOG.debug("Start s_zk.getChildren: " + SERVICES_PATH);
			}
			List<String> services = s_zk.getChildren(SERVICES_PATH, false);
			for ( String service : services ) {
				List<String> nodes = s_zk.getChildren(SERVICES_PATH + "/" + service, false);
				for (String node : nodes ) {
					if ( ZKUtils.SERVICE_MASTER_NODE.equalsIgnoreCase(node)) {
						continue;
					}
					if ( !servicesMap.containsKey(node)) {
						servicesMap.put(node, new ArrayList<String>());
					}
					// kind of an exceptional case as kvstore is a special instance of fileserver where cldb info is stored
					if ( "kvstore".equalsIgnoreCase(service) ) { 
						 service = "fileserver";
					}
					servicesMap.get(node).add(service);
				}
			}
			if ( LOG.isDebugEnabled()) {
			  LOG.debug("End s_zk.getChildren: " + SERVICES_PATH);
			}
		} catch (KeeperException e) {
			/**
			* <MAPR_ERROR>
			* Message:ZooKeeper Exception while trying to get running services
			* Function:NodesCommonUtils.findServicesRunningHierarchy()
			* Meaning:An error occurred.
			* Resolution:Contact technical support.
			* </MAPR_ERROR>
			*/
			LOG.error("ZooKeeper Exception while trying to get running services", e);
			throw new CLIProcessingException("ZooKeeper Exception while trying to get running services", e);
		} catch (InterruptedException e) {
			/**
			* <MAPR_ERROR>
			* Message:ZooKeeper Exception while trying to get running services on node
			* Function:NodesCommonUtils.findServicesRunningHierarchy()
			* Meaning:An error occurred.
			* Resolution:Contact technical support.
			* </MAPR_ERROR>
			*/
			LOG.error("Interrupted Exception while trying to get running services on node", e);
			throw new CLIProcessingException("Interrupted Exception while trying to get running services on node", e);
		}
		return servicesMap;
	}
	
	@Deprecated
	public static Map<String,List<String>> findConfiguredServicesHierarchy(ZooKeeper s_zk) throws CLIProcessingException {
		Map<String,List<String>> servicesMap = new HashMap<String,List<String>>();
		try {
			if ( s_zk == null ) {
				return servicesMap;
			}
			Stat stats = s_zk.exists(CONF_SERVICES_PATH, false);
			if ( stats == null ) {
				return servicesMap;
			}
			List<String> services = s_zk.getChildren(CONF_SERVICES_PATH, false);
			for ( String service : services ) {
				List<String> nodes = s_zk.getChildren(CONF_SERVICES_PATH + "/" + service, false);
				for (String node : nodes ) {
					if ( !servicesMap.containsKey(node)) {
						servicesMap.put(node, new ArrayList<String>());
					}
					// kind of an exceptional case as kvstore is a special instance of fileserver where cldb info is stored
					if ( "kvstore".equalsIgnoreCase(service) ) { 
						 service = "fileserver";
					}
					servicesMap.get(node).add(service);
				}
			}
		} catch (KeeperException e) {
			/**
			* <MAPR_ERROR>
			* Message:ZooKeeper Exception while trying to get running services
			* Function:NodesCommonUtils.findConfiguredServicesHierarchy()
			* Meaning:An error occurred.
			* Resolution:Contact technical support.
			* </MAPR_ERROR>
			*/
			LOG.error("ZooKeeper Exception while trying to get running services", e);
			throw new CLIProcessingException("ZooKeeper Exception while trying to get configured services", e);
		} catch (InterruptedException e) {
			/**
			* <MAPR_ERROR>
			* Message:"Interrupted Exception while trying to get running services on node
			* Function:NodesCommonUtils.findConfiguredServicesHierarchy()
			* Meaning:An error occurred.
			* Resolution:Contact technical support.
			* </MAPR_ERROR>
			*/
			LOG.error("Interrupted Exception while trying to get running services on node", e);
			throw new CLIProcessingException("Interrupted Exception while trying to get configured services on node", e);
		}
		return servicesMap;
	}

	public static Map<String,List<String>> findConfiguredServicesByServiceHierarchy(String zkConnectString) throws CLIProcessingException {
		if ( LOG.isDebugEnabled() ) {
		LOG.debug("findConfiguredServicesByServiceHierarchy start");
		}
		ZKDataRetrieval currentZK = zkMap.get(zkConnectString);
		if ( currentZK == null ) {
			connect(zkConnectString);
		}
		currentZK = zkMap.get(zkConnectString);
		// if still null - return empty map
		if ( currentZK ==  null ) {
			LOG.warn("Unable to get ZK based on: " + zkConnectString);
			return new HashMap<String,List<String>>();
		}
		Map<String, List<String>> serviceToNode = new HashMap<String, List<String>>();
		Map<String, List<String>> configMap = currentZK.getConfigServicesMap();
		for ( Map.Entry<String, List<String>> entry : configMap.entrySet()) {
			String service = entry.getKey();
			List<String> nodes = new ArrayList<String>(entry.getValue());
			if ( service.equalsIgnoreCase("kvstore")) {
				service = "fileserver";
			}
			if ( serviceToNode.containsKey(service)) {
				serviceToNode.get(service).addAll(nodes);
			} else {
				serviceToNode.put(service, nodes);
			}	
		}
		if ( LOG.isDebugEnabled() ) {
		LOG.debug("findConfiguredServicesByServiceHierarchy end");
		}
		return serviceToNode;
	}
	
	@Deprecated
	public static Map<String,List<String>> findConfiguredServicesByServiceHierarchy(ZooKeeper s_zk) throws CLIProcessingException {
		Map<String,List<String>> servicesMap = new HashMap<String,List<String>>();
		try {
			if ( s_zk == null ) {
				return servicesMap;
			}
			Stat stats = s_zk.exists(CONF_SERVICES_PATH, false);
			if ( stats == null ) {
				return servicesMap;
			}
			List<String> services = s_zk.getChildren(CONF_SERVICES_PATH, false);
			for ( String service : services ) {
				List<String> nodes = s_zk.getChildren(CONF_SERVICES_PATH + "/" + service, false);
				if ( "kvstore".equalsIgnoreCase(service) ) { 
					 service = "fileserver";
				}
				if ( servicesMap.containsKey(service)) {
					servicesMap.get(service).addAll(nodes);
				} else {
					servicesMap.put(service, nodes);
				}
			}
		} catch (KeeperException e) {
			/**
			* <MAPR_ERROR>
			* Message:"Interrupted Exception while trying to get running services
			* Function:NodesCommonUtils.findConfiguredServicesHierarchy()
			* Meaning:An error occurred.
			* Resolution:Contact technical support.
			* </MAPR_ERROR>
			*/
			LOG.error("ZooKeeper Exception while trying to get running services", e);
			throw new CLIProcessingException("ZooKeeper Exception while trying to get configured services", e);
		} catch (InterruptedException e) {
			/**
			* <MAPR_ERROR>
			* Message:"Interrupted Exception while trying to get running services on node
			* Function:NodesCommonUtils.findConfiguredServicesHierarchy()
			* Meaning:An error occurred.
			* Resolution:Contact technical support.
			* </MAPR_ERROR>
			*/
			LOG.error("Interrupted Exception while trying to get running services on node", e);
			throw new CLIProcessingException("Interrupted Exception while trying to get configured services on node", e);
		}
		return servicesMap;
	}
	
	/**
	 * Primary method to connect to ZK - will return cached connection if exist
	 * otherwise will try to connect 
	 * @param zkConnectString
	 * @return
	 * @throws IOException
	 */
	public static synchronized ZooKeeper connect(String zkConnectString) {
		ZKDataRetrieval currentZK = zkMap.get(zkConnectString);
		if ( currentZK != null ) {
			if ( LOG.isDebugEnabled()) {
			  LOG.debug("Got cached ZK for: " + zkConnectString);
			}
			return currentZK.getZKObject();
		}

    ZKDataRetrieval zkHolder;
    YarnConfiguration yarnconf = new YarnConfiguration();
    if(HAUtil.isHAEnabled(yarnconf)) {
      String zkBasePath = yarnconf.get(YarnConfiguration.AUTO_FAILOVER_ZK_BASE_PATH,
              YarnConfiguration.DEFAULT_AUTO_FAILOVER_ZK_BASE_PATH);
      String rmHAZNode = zkBasePath + "/" + YarnConfiguration.getClusterId(yarnconf) + "/ActiveStandbyElectorLock";
      zkHolder = new ZKDataRetrieval(zkConnectString, rmHAZNode);
    }
    else {
      zkHolder = new ZKDataRetrieval(zkConnectString);
    }

    zkMap.put(zkConnectString, zkHolder);
    return zkHolder.getZKObject();
	}

  public static String getCurrentRMMasterID(String zkConnectString) {

    if (LOG.isDebugEnabled()) {
      LOG.debug("getCurrentRMMasterID start");
    }

    ZKDataRetrieval currentZK = zkMap.get(zkConnectString);
    if (currentZK == null) {
      connect(zkConnectString);
    }
    currentZK = zkMap.get(zkConnectString);
    // if still null - return empty map
    if (currentZK == null) {
      LOG.warn("Unable to get ZK based on: " + zkConnectString);
      return null;
    }

    YarnServerResourceManagerServiceProtos.ActiveRMInfoProto proto;
    byte[] data;
    if((data = currentZK.getActiveRMZNodeData()) != null) {
      try {
        proto = YarnServerResourceManagerServiceProtos.ActiveRMInfoProto
                .parseFrom(data);
      } catch (InvalidProtocolBufferException e) {
        LOG.error("Invalid data in ZK: " + data.toString());
        return null;
      }

      return proto.getRmId();
    }

    if (LOG.isDebugEnabled()) {
      LOG.debug("getCurrentRMMasterID end");
    }

    LOG.error("Unable to get Current RM Master ID from ZK");
    return null;

  }

  public static FileServerInfo getFileServerInfo(IPAddress server, CredentialsMsg creds,
      String cluster) throws CLIProcessingException {
    FileServerLookupRequest req = FileServerLookupRequest.newBuilder()
                                          .setFileServerIP(server)
                                          .setCreds(creds)
                                          .build();
    FileServerLookupResponse resp = null;
    byte[] data = null;
    try {
      if ( cluster != null && !cluster.isEmpty() ) {
        data = CLDBRpcCommonUtils.getInstance().sendRequest(
            cluster,
            Common.MapRProgramId.CldbProgramId.getNumber(),
          CLDBProto.CLDBProg.FileServerLookupProc.getNumber(), 
              req, FileServerLookupResponse.class);
      } else {
        data = CLDBRpcCommonUtils.getInstance().sendRequest(
            Common.MapRProgramId.CldbProgramId.getNumber(),
          CLDBProto.CLDBProg.FileServerLookupProc.getNumber(), 
              req, FileServerLookupResponse.class);
      }    

      if (data == null) {
        LOG.error("FileServerLookup RPC to CLDB failed for IP " + 
            Util.printIPAddressForCLI(server));
        return null;
      }    
      resp = FileServerLookupResponse.parseFrom(data);
      if (resp.getStatus() == 0) { 
        return resp.getFileServerInfo(); 
      }    
    } catch (MaprSecurityException e) { 
      throw new CLIProcessingException(
        "MaprSecurityException " + "Exception", e);
    } catch (Exception e) { 
      LOG.error("Exception during FileServerLookup RPC to CLDB " + 
          e.getLocalizedMessage());
      return null;
    }
    return null;
  }

}
