package com.mapr.cli.common;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;

import com.google.protobuf.MessageLite;
import com.mapr.baseutils.Errno;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.util.FieldInfo;
import com.mapr.cliframework.util.FilterProcessingException;
import com.mapr.cliframework.util.FilterUtil;
import com.mapr.fs.cli.proto.CLIProto.Filter;
import com.mapr.fs.cli.proto.CLIProto.Limiter;

public abstract class ListCommand extends CLIBaseClass implements ListIterator {
  
  private static final Logger LOG = Logger.getLogger(ListCommand.class);

  public ListCommand(ProcessedInput input, CLICommand cliCommand) {
    super(input, cliCommand);
  }

  /**
   * Helper method to build limiter for next request.
   * @param prevStart - start index of the limiter in prev request
   * @param prevCount - number of records returned in prev response
   * @param origStart - starting index of the records originally requested (using the cli -start param)
   * @param origLimit - number of records originally requested (using the cli -limit param)
   * @param numRecsPerRpc - number of records an rpc response can carry
   * @return - Limiter object for the next request
   */
  protected Limiter getNextLimiter(int prevStart, int prevCount, 
      int origStart, int origLimit, int numRecsPerRpc) {
    int nextStart = prevStart + prevCount;
    int nextLimit = origLimit - (nextStart - origStart); // origLimit - cumulative count till last iteration

    if (nextLimit > numRecsPerRpc) {
      nextLimit = numRecsPerRpc;
    }
    
    if (LOG.isDebugEnabled()) {
      LOG.debug("getNextLimiter - origStart = " + origStart + ", origLimit = " + origLimit 
          + ", prevCount = " + prevCount + ", numRecsPerRpc = " + numRecsPerRpc
          + ", nextStart = " + nextStart + ", nextLimit = "  + nextLimit);
    }
    return Limiter.newBuilder()
                  .setStart(nextStart)
                  .setLimit(nextLimit)
                  .build();
  }

  /**
   * Helper method for the sub classes to build a list of Filters from the input filter string.
   * @param <T>
   * @param table 
   * @param filterParam
   * @return
   * @throws CLIProcessingException
   */
  protected <T> List<Filter> getFilters(Map <T, FieldInfo> table, String filterParam) throws CLIProcessingException {
    List<Filter> filters;
    String filterString = getParamTextValue(filterParam, 0);
    if (filterString != null && (!filterString.equals("none")) && (!filterString.equals("[id=all]"))) {
      try {
        List<String> filterStrings = new ArrayList<String>();
        filterStrings.add(filterString);
        filters = FilterUtil.compileFilter(table, filterStrings);
      } catch (FilterProcessingException fpe) {
        throw new CLIProcessingException(fpe.getMessage(), fpe);
      }
    } else {
      filters = new ArrayList<Filter>();
    }
    return filters;
  }

  /**
   * A method that helps the sub classes to determine if there are more records to be processed.
   * @param origStart - The original value for the -start parameter
   * @param origLimit - The original value for the -limit parameter
   * @param prevStart - The value for the start parameter supplied during previous iteration.
   * @param prevCount - The number of records returned in the previous iteration.
   * @return - true, if there are more records. false otherwise.
   */
  protected boolean hasMore(int origStart, int origLimit, int prevStart, int prevCount) {
    if (prevCount == 0) { // no more records to fetch
      LOG.info("in hasMore:: Last fetched record count is 0. no more records to fetch");
      return false;
    } 

    int endIndex = 0;
    if (origLimit == Integer.MAX_VALUE) {
      endIndex = origLimit;
    } else {
      endIndex = origStart + origLimit;
    }
    
    return endIndex > prevStart + prevCount;
  }

  @Override
  public void list(OutputHierarchy out) throws CLIProcessingException {
    long time = System.currentTimeMillis(), start = System.currentTimeMillis();
    MessageLite req = null;
    MessageLite resp = null;

    while ((req == null && resp == null) || hasMore(req, resp)) {
      req = buildNextRequest(req, resp);
      if (LOG.isDebugEnabled()) {
        LOG.debug("Perf::buildNextRequest() = " + (System.currentTimeMillis() - time));
        time = System.currentTimeMillis();
      }
      resp = sendRequest(req);
      if (LOG.isDebugEnabled()) {
        LOG.debug("Perf::sendRequest() = " + (System.currentTimeMillis() - time));
        time = System.currentTimeMillis();
      }
      if ( resp == null ) {
    	  // CLDB is down
    	  out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service. Check if at least one CLDB is running."));
    	  return;
      } else {
    	  processResponse(out, resp);
        if (LOG.isDebugEnabled()) {
          LOG.debug("Perf::processResponse() = " + (System.currentTimeMillis() - time));
          time = System.currentTimeMillis();
        }
      }
    }
    if (LOG.isDebugEnabled()) {
      LOG.debug("Perf::list() = " + (System.currentTimeMillis() - start));
    }
  }
  
  public abstract boolean hasMore(MessageLite prevReq, MessageLite prevResp) throws CLIProcessingException;
  
  public abstract MessageLite buildNextRequest(MessageLite prevReq, MessageLite prevResp) throws CLIProcessingException;

  public abstract MessageLite sendRequest(MessageLite req) throws CLIProcessingException;
  
  public abstract void processResponse(OutputHierarchy out, MessageLite resp) throws CLIProcessingException;
}
