package com.mapr.cli.common;

import java.io.BufferedReader;
import java.io.File;
import java.io.StringWriter;
import java.io.PrintWriter;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.Callable;

import org.apache.log4j.Logger;

import com.mapr.cli.DbReplicaCommands;
import com.mapr.fs.AceHelper;
import com.mapr.fs.proto.Dbserver.TableReplicaDesc;


public class CopyTableCallable implements Callable<Integer> {
  private String srcTable;
  private String dstTable;
  private String user;
  private int maxVersions;
  private String columnSpec;
  private String ticketPath;
  private boolean isMultiMaster;
  private AceHelper.DBPermission dbPerm;
  private TableReplicaDesc.Builder replBuilder;
  private static final Logger LOG = Logger.getLogger(CopyTableCallable.class);

  public CopyTableCallable(String srcTable,String destTable, String user, int maxVersions, String columnSpec, String ticketPath,
      boolean isMultiMaster, AceHelper.DBPermission dbPerm, TableReplicaDesc.Builder replBuilder) {
    this.srcTable = srcTable;
    this.dstTable = destTable;
    this.user = user;
    this.maxVersions = maxVersions;
    this.columnSpec = columnSpec;
    this.ticketPath = ticketPath;
    this.isMultiMaster = isMultiMaster;
    this.dbPerm = dbPerm;
    this.replBuilder = replBuilder;
  }

  @Override
  public Integer call() throws Exception {
    try
    {
      LOG.info("Calling copy table script for tables: "+srcTable+","+dstTable);
      if(srcTable == null || dstTable == null || user == null) throw new Exception("Invalid input.Please check your input");
      //Set status as prep
      JobExecutor.setStatus(srcTable+"_"+dstTable, Double.valueOf(2.0));
      //Build the command
      List<String> command = new ArrayList<String>();
      command.add("/opt/mapr/bin/runcopytable");
      command.add(srcTable);
      command.add(dstTable);
      command.add(user);
      command.add(String.valueOf(maxVersions));
      if (columnSpec != null && columnSpec.length() > 0) {
        command.add(columnSpec);
      }
      LOG.info("Calling command: "+command.toString());
      //Build the process builder
      ProcessBuilder builder = new ProcessBuilder(command);
      builder.directory(new File("/opt/mapr/bin"));
      //Set the environment variable MAPR_TICKETFILE_LOCATION
      if(ticketPath!=null)
      builder.environment().put("MAPR_TICKETFILE_LOCATION", ticketPath);
      //Start the process
      Process p = builder.start();
      //Read from the scripts output
      BufferedReader stdInput = new BufferedReader(new InputStreamReader(p.getInputStream()));
      String s=null;
      //Get the job completion status
      while ((s = stdInput.readLine()) != null) {
        if (s.trim().isEmpty())
          continue;
        JobExecutor.setStatus(srcTable+"_"+dstTable, Double.valueOf(s.trim()));
      }
      //If there is any error then update the job status to -1
      BufferedReader stdErr = new BufferedReader(new InputStreamReader(p.getErrorStream()));
      while((s=stdErr.readLine()) != null) {
        JobExecutor.setStatus(srcTable+"_"+dstTable, Double.valueOf(-1.0));
      }

      //Return the status of running command
      int status = p.waitFor();
      if(status !=0) {
        JobExecutor.setStatus(srcTable+"_"+dstTable, Double.valueOf(-1.0));
        throw new Exception("Copy Table script failed with status " + status);
      }

      LOG.info("Finished copy table script for tables: "+srcTable+","+dstTable);
      try {
        if (isMultiMaster) {
          DbReplicaCommands.setupMultiMaster(dbPerm,replBuilder,srcTable,dstTable,columnSpec,user);
        }
      }catch(Exception e) {
        LOG.info("Failed to setup multi-master replication from table "+srcTable+" to "+dstTable+" with exception "+e.getMessage());
        return null;
      }
      //Resume the replication
      LOG.info("Resuming the replication from table: "+srcTable+" to "+dstTable);
      DbReplicaCommands.pauseOrResumeReplication(srcTable, dstTable, false, user);
      return status;
    }catch(Exception e)
    {
      StringWriter sw = new StringWriter();
      PrintWriter pw = new PrintWriter(sw);
      e.printStackTrace(pw);
      LOG.info("Failed with: " + sw.toString());
      LOG.info("Failed to run the copytable script for tables: "+srcTable+","+dstTable+" with error: "+e.getMessage());
      JobExecutor.setStatus(srcTable+"_"+dstTable, Double.valueOf(-1.0));
      return null;
    }catch(Throwable t)
    {
      LOG.info("Throwable",t);
      JobExecutor.setStatus(srcTable+"_"+dstTable, Double.valueOf(-1.0));
      return null;
    }
  }
}

