package com.mapr.cli.common;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.google.protobuf.InvalidProtocolBufferException;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.fs.cldb.proto.Accesscontrol.ClientAuthorizationRequest;
import com.mapr.fs.cldb.proto.Accesscontrol.ClientAuthorizationResponse;
import com.mapr.fs.cldb.proto.CLDBProto.CLDBProg;
import com.mapr.fs.cldb.proto.CLDBProto.SecureObjectType;
import com.mapr.fs.cldb.proto.CLDBProto.SecurityGetAclRequest;
import com.mapr.fs.cldb.proto.CLDBProto.SecurityGetAclResponse;
import com.mapr.fs.cldb.proto.CLDBProto.VolumeProperties;
import com.mapr.fs.cldb.security.ACL;
import com.mapr.fs.proto.Common;
import com.mapr.fs.proto.Security.AccessControlList;
import com.mapr.fs.proto.Security.CredentialsMsg;

public class AuthManager {

  private static AuthManager s_instance;
  private static final Log LOG = LogFactory.getLog(AuthManager.class);

  public synchronized static AuthManager getInstance()
  {
    if (s_instance == null) {
      s_instance = new AuthManager();
    }
    return s_instance;
  }

  /*
   * First, send the Authorization request to CLDB. If the CLDB does not
   * support client authorization, an EPRCFAILED status is returned. We
   * then revert to using the old way of Authorizing the client by fetching
   * the ACL and checking for permissions.
   *
   * Case 1: CLDB supports Client Authorization. Return the Authorization
   *         flag, populating the error output if necessary.
   * Case 2: The CLDB supports Client Authorization, but returns an error.
   *         Construct the error node and return false.
   * Case 3: CLDB returns ERPCFAILED. It is interpreted as unsupported RPC,
   *         although an RPC could be rejected for other reasons.
   */

  /**
   * Checks if a client is authorized to perform actions on a certain Volume.
   *
   * @param creds       Credentials of the user performing the Actions
   * @param volumeActions List of volume actions
   * @param volProps    VolumeProperties structure of the Volume
   * @param volumeName  Name of the Volume
   * @param out         OutputHierarchy object to record error messages
   * @return            true, if the user is authorized to perform any of the
   *                    actions ; false otherwise and in the case of errors
   * @throws CLIProcessingException
   */
  public boolean canPerformVolumeAction(String clusterName, String volumeName,
      VolumeProperties volProps, int volumeActions, CredentialsMsg creds,
      OutputHierarchy out) throws CLIProcessingException
  {
    ClientAuthorizationResponse resp = sendAuthRequestToCldb(clusterName,
        creds, volumeActions, volumeName, 0 /* clusterActions */);
    if (resp.getStatus() == 0) {
      /* Case 1 */
      if (resp.getIsAuthorized() == false) {
        if (out != null) {
          out.addError(new OutputError(Errno.E_NOT_ENOUGH_PRIVILEGES,
              Errno.toString(Errno.E_NOT_ENOUGH_PRIVILEGES)));
        }
      }
      return resp.getIsAuthorized();
    }
    if (resp.getStatus() != Errno.ERPCFAILED) {
      /* Case 2 */
      if (out != null) {
        out.addError(new OutputError(resp.getStatus(), resp.getErrMsg()));
      }
      return false;
    }

    /* Case 3 */
    if (LOG.isInfoEnabled()) {
      LOG.info("Authorization Not Supported at CLDB: Reverting to Obtaining ACLs");
    }

    /*
     * If VolumeProperties is passed to this function, the ACLs are
     * expected to be present in that. Hence, we look no further.
     */
    AccessControlList accessControlList = null;
    if (volProps != null) {
      if (volProps.getAcl() == null) {
        if (out != null) {
          out.addError(new OutputError(Errno.EINVAL,
              "Missing ACLs in VolumeProperties for Volume " + volumeName));
        }
        return false;
      }
      accessControlList = volProps.getAcl();
    }
    else {
      SecurityGetAclResponse getAclsResp = getAcls(creds, volumeName);
      if (getAclsResp.getStatus() != 0) {
        if (out != null) {
          out.addError(new OutputError(getAclsResp.getStatus(),
              getAclsResp.getErrorString()));
        }
        return false;
      }
      accessControlList = getAclsResp.getAcl();
    }

    ACL acl = new ACL(accessControlList);
    if (!acl.verifyPermissions(creds, volumeActions)) {
      if (out != null) {
        out.addError(new OutputError(Errno.E_NOT_ENOUGH_PRIVILEGES,
            "Authorization Error: Volume: " + volumeName +
            " VolumeActions: " + volumeActions));
      }
      return false;
    }

    return true;
  }

  public boolean canPerformVolumeAction(String volumeName, VolumeProperties volProps,
      int volumeActions, CredentialsMsg creds, OutputHierarchy out)
          throws CLIProcessingException
  {
    return canPerformVolumeAction(CLDBRpcCommonUtils.getInstance().getCurrentClusterName(),
        volumeName, volProps, volumeActions, creds, out);
  }

  /**
   * Checks if a user is authorized to perform specific cluster actions.
   *
   * @param creds  Credentials of the user performing the action.
   * @param clusterActions List of cluster actions the user wishes to perform
   * @param out    OutputHierarchy object to record error messages
   * @return       true, if the user is authorized to perform any of the
   *               actions ; false otherwise and in the case of errors
   * @throws CLIProcessingException
   */
  public boolean canPerformClusterActions(String clusterName, int clusterActions,
      CredentialsMsg creds, OutputHierarchy out) throws CLIProcessingException
  {
    ClientAuthorizationResponse resp = sendAuthRequestToCldb(clusterName,
        creds, 0 /* volumeActions */, null /* volumeName */, clusterActions);
    if (resp.getStatus() == 0) {
      /* Case 1 */
      if (resp.getIsAuthorized() == false) {
        if (out != null) {
          out.addError(new OutputError(Errno.E_NOT_ENOUGH_PRIVILEGES,
              Errno.toString(Errno.E_NOT_ENOUGH_PRIVILEGES)));
        }
      }
      return resp.getIsAuthorized();
    }
    if (resp.getStatus() != Errno.ERPCFAILED) {
      /* Case 2 */
      if (out != null) {
        out.addError(new OutputError(resp.getStatus(), resp.getErrMsg()));
      }
      return false;
    }

    /* Case 3 */
    if (LOG.isInfoEnabled()) {
      LOG.info("Authorization Not Supported at CLDB: Reverting to Obtaining ACLs");
    }

    SecurityGetAclResponse getAclsResp = getAcls(creds, null);
    if (getAclsResp.getStatus() != 0) {
      if (out != null) {
        out.addError(new OutputError(getAclsResp.getStatus(), getAclsResp.getErrorString()));
      }
      return false;
    }

    ACL acl = new ACL(getAclsResp.getAcl());
    if (!acl.verifyPermissions(creds, clusterActions)) {
      if (out != null) {
        out.addError(new OutputError(Errno.E_NOT_ENOUGH_PRIVILEGES,
            Errno.toString(Errno.E_NOT_ENOUGH_PRIVILEGES)));
      }
      return false;
    }

    return true;
  }

  public boolean canPerformClusterActions(int clusterActions, CredentialsMsg creds,
      OutputHierarchy out) throws CLIProcessingException
  {
    return canPerformClusterActions(CLDBRpcCommonUtils.getInstance().getCurrentClusterName(),
        clusterActions, creds, out);
  }

  private ClientAuthorizationResponse sendAuthRequestToCldb(String clusterName,
      CredentialsMsg creds, int volumeActions, String volumeName, int clusterActions)
          throws CLIProcessingException
  {
    /* Step 1: Construct the Request and Send it to the CLDB Server */
    ClientAuthorizationRequest.Builder authRequestBuilder =
        ClientAuthorizationRequest.newBuilder()
        .setCreds(creds)
        .setClusterActions(clusterActions)
        .setVolumeActions(volumeActions);
    if (volumeName != null) {
      authRequestBuilder.setVolumeName(volumeName);
    }

    byte[] data;
    try {
      data = CLDBRpcCommonUtils.getInstance().sendRequest(
          clusterName,
          Common.MapRProgramId.CldbProgramId.getNumber(),
          CLDBProg.ClientAuthorizationProc.getNumber(),
          authRequestBuilder.build(),
          ClientAuthorizationResponse.class);
    } catch (Exception e) {
      throw new CLIProcessingException("Error Sending RPC Request", e);
    }

    if (data == null) {
      return ClientAuthorizationResponse.newBuilder()
          .setStatus(Errno.ERPCFAILED)
          .setErrMsg("RPC Rejected by CLDB Server")
          .build();
    }

    /* Step 2: Parse the response */
    ClientAuthorizationResponse authResponse;
    try {
      authResponse = ClientAuthorizationResponse.parseFrom(data);
    }
    catch (InvalidProtocolBufferException e) {
      throw new CLIProcessingException("Error Parsing RPC Response", e);
    }

    return authResponse;
  }

  private SecurityGetAclResponse getAcls(CredentialsMsg creds, String volumeName)
      throws CLIProcessingException
  {
    SecurityGetAclRequest.Builder aclRequestBuilder =
        SecurityGetAclRequest.newBuilder()
        .setCreds(creds)
        .setObjectType(SecureObjectType.OBJECT_TYPE_CLUSTER);
    if (volumeName != null) {
      aclRequestBuilder.setName(volumeName)
      .setObjectType(SecureObjectType.OBJECT_TYPE_VOLUME);
    }

    byte[] data;
    try {
      data = CLDBRpcCommonUtils.getInstance().sendRequest(
          Common.MapRProgramId.CldbProgramId.getNumber(),
          CLDBProg.SecurityGetAclProc.getNumber(),
          aclRequestBuilder.build(),
          SecurityGetAclResponse.class);
    } catch (Exception e) {
      throw new CLIProcessingException(e);
    }

    if (data == null) {
      return SecurityGetAclResponse.newBuilder()
          .setStatus(Errno.ERPCFAILED)
          .setErrorString("RPC Rejected by CLDB Server")
          .build();
    }

    SecurityGetAclResponse getAclsResp = null;
    try {
      getAclsResp = SecurityGetAclResponse.parseFrom(data);
    } catch (InvalidProtocolBufferException e) {
      throw new CLIProcessingException(
          "InvalidProtocolBufferException " + "Exception", e);
    }

    return getAclsResp;
  }
}
