package com.mapr.cli.bulk;

import org.apache.log4j.Logger;

import com.mapr.baseutils.Errno;
import com.mapr.cli.MapRCliUtil;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.fs.MapRFileSystem;
import com.mapr.fs.cldb.proto.CLDBProto.VolumeProperties;
import com.mapr.fs.proto.Common.FidMsg;
import com.mapr.fs.proto.Common.MapRClusterDefaults;

public class VolumeUnMountTask extends AbstractTask {
  private static Logger LOG = Logger.getLogger(VolumeUnMountTask.class);
  private static final String TASK_NAME = "VolumeUnMount";

  private VolumeProperties volProps;
  private String cluster;
  private String user;

  public VolumeUnMountTask(VolumeProperties volProps, String cluster, String user) {
    super(TASK_NAME);
    this.volProps = volProps;
    this.cluster = cluster;
    this.user = user;
  }

  @Override
  public TaskResult executeTask() {
    String volumeName = this.volProps.getVolumeName();
    TaskResult result = new TaskResult();
    result.setEntityName(volumeName);

    try {
      MapRFileSystem fs = MapRCliUtil.getMapRFileSystem();
      FidMsg parentFid = this.volProps.getParentFid();
      int status = fs.unmountVolume(
              this.cluster,
              volumeName,
              this.volProps.getMountDir(),
              this.user,
              parentFid.getCid(),
              parentFid.getCinum(),
              parentFid.getUniq());

      result.setStatus(Errno.SUCCESS);
      if (status != Errno.SUCCESS) {
        String failureMsg =
            "Volume unmount failed for volume: " + volumeName
                + ". Error: " + (status > 0 ? Errno.toString(status) : "");
        result.setFailureMsg(failureMsg);
        LOG.warn(failureMsg);
      }
    } catch (CLIProcessingException e) {
      LOG.error("Exception while unmounting volume: " + volumeName, e);
    }
    return result;
  }

  @Override
  public TaskResult validateInput() {
    String volumeName = this.volProps.getVolumeName();
    TaskResult result = new TaskResult();
    result.setStatus(Errno.SUCCESS);
    result.setEntityName(volumeName);
    
    if (volumeName.equals(MapRClusterDefaults.getDefaultInstance().getClusterRootName())) {
      String failureMsg = "Cannot unmount volume " + volumeName + ". Error: " + Errno.toString(Errno.EINVAL);
      result.setStatus(Errno.EINVAL);
      result.setFailureMsg(failureMsg);
      LOG.warn(failureMsg);
      return result;
    }

    if (!this.volProps.getMounted()) {
      String failureMsg = "Volume " + volumeName + " is not mounted.";
      result.setStatus(Errno.EINVAL);
      result.setFailureMsg(failureMsg);
      LOG.warn(failureMsg);
    }
    return result;
  }
}
