package com.mapr.cli.bulk;

import org.apache.log4j.Logger;

import com.google.protobuf.InvalidProtocolBufferException;
import com.mapr.baseutils.Errno;
import com.mapr.cli.MapRCliUtil;
import com.mapr.fs.cldb.proto.CLDBProto.CLDBProg;
import com.mapr.fs.cldb.proto.CLDBProto.VolumeProperties;
import com.mapr.fs.cldb.proto.CLDBProto.VolumeRemoveRequest;
import com.mapr.fs.cldb.proto.CLDBProto.VolumeRemoveResponse;
import com.mapr.fs.proto.Common.MapRClusterDefaults;
import com.mapr.fs.proto.Security.CredentialsMsg;

public class VolumeRemoveTask extends AbstractTask {
  private static final Logger LOG = Logger.getLogger(VolumeRemoveTask.class);
  private static final String TASK_NAME = "VolumeRemove";

  private VolumeProperties properties;
  private boolean forceRemove;
  private CredentialsMsg userCreds;
  private String cluster;

  public VolumeRemoveTask(VolumeProperties properties, boolean forceRemove,
      CredentialsMsg userCreds, String cluster) {
    super(TASK_NAME);
    
    this.properties = properties;
    this.forceRemove = forceRemove;
    this.userCreds = userCreds;
    this.cluster = cluster;
  }

  @Override
  public TaskResult executeTask() {
    VolumeRemoveRequest volumeRemove =
        VolumeRemoveRequest.newBuilder().setVolumeName(this.properties.getVolumeName())
            .setCreds(this.userCreds).setForceRemove(this.forceRemove).build();

    VolumeRemoveResponse response = sendRpc(volumeRemove);
    TaskResult result = new TaskResult();
    if (response != null) {
      result.setEntityName(this.properties.getVolumeName());
      result.setStatus(response.getStatus());
      if (response.getStatus() != Errno.SUCCESS) {
        String failureMsg = "Volume remove failed: " + Errno.toString(response.getStatus());
        result.setFailureMsg(failureMsg);
        LOG.warn(failureMsg);
      }
    } else {
      String failureMsg = "Rpc failure during volume removal.";
      result.setFailureMsg(failureMsg);
      LOG.warn(failureMsg);
    }
    return result;
  }

  private VolumeRemoveResponse sendRpc(VolumeRemoveRequest volumeRemove) {
    VolumeRemoveResponse response = null;
    try {
      response = VolumeRemoveResponse.parseFrom(
          MapRCliUtil.sendRpc(volumeRemove, CLDBProg.VolumeRemoveProc.getNumber(), this.cluster,
              VolumeRemoveResponse.class));
    } catch (InvalidProtocolBufferException e) {
      LOG.error("Encountered exception while parsing VolumeRemoveResponse. Volume: " + volumeRemove.getVolumeName());
    } catch (Exception e) {
      LOG.error("Removal of volume: " + volumeRemove.getVolumeName() + " failed due to exception " + e);
    }
    return response;
  }

  @Override
  public TaskResult validateInput() {
    String volName = this.properties.getVolumeName();

    TaskResult result = new TaskResult();
    result.setEntityName(volName);

    if (volName.equals(MapRClusterDefaults.getDefaultInstance().getClusterRootName())) {
      String failureMsg = "Cannot remove volume " + volName + " " + Errno.toString(Errno.EINVAL);
      result.setStatus(Errno.EINVAL);
      result.setFailureMsg(failureMsg);
      LOG.warn(failureMsg);
      return result;
    }

    if ((this.properties.getNumSnapshots()) > 0 && (!this.forceRemove)) {
      String failureMsg = "Volume Remove: " + Errno.toString(Errno.ENOTEMPTY) + " Volume has "
      + this.properties.getNumSnapshots()
      + " snapshots. Please use -force option to remove Volume and Snapshots";
      result.setStatus(Errno.EINVAL);
      result.setFailureMsg(failureMsg);
      LOG.warn(failureMsg);
      return result;
    }

    result.setStatus(Errno.SUCCESS);
    return result;
  }
}
