package com.mapr.cli;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.MessageLite;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.baseutils.Errno;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CLIUsageOnlyCommand;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.BooleanInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.cliframework.util.FieldInfo;
import com.mapr.fs.proto.Common.VolumeType;
import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.cldb.proto.CLDBProto.MirrorDumpPermCheckRequest;
import com.mapr.fs.cldb.proto.CLDBProto.MirrorDumpPermCheckResponse;
import com.mapr.fs.cldb.proto.CLDBProto.MirrorInfo;
import com.mapr.fs.cldb.proto.CLDBProto.MirrorInfo.MirrorStatus;
import com.mapr.fs.cldb.proto.CLDBProto.MirrorStartRequest;
import com.mapr.fs.cldb.proto.CLDBProto.MirrorStartResponse;
import com.mapr.fs.cldb.proto.CLDBProto.MirrorStopRequest;
import com.mapr.fs.cldb.proto.CLDBProto.MirrorStopResponse;
import com.mapr.fs.cldb.proto.CLDBProto.MirrorType;
import com.mapr.fs.cldb.proto.CLDBProto.VolumeInfo;
import com.mapr.fs.cldb.proto.CLDBProto.VolumeInfoFields;
import com.mapr.fs.cldb.proto.CLDBProto.VolumeListRequest;
import com.mapr.fs.cldb.proto.CLDBProto.VolumeListResponse;
import com.mapr.fs.cldb.proto.CLDBProto.VolumeLookupRequest;
import com.mapr.fs.cldb.proto.CLDBProto.VolumeLookupResponse;
import com.mapr.fs.cldb.proto.CLDBProto.VolumeProperties;
import com.mapr.fs.cli.proto.CLIProto.FieldOp;
import com.mapr.fs.cli.proto.CLIProto.FieldVal;
import com.mapr.fs.cli.proto.CLIProto.Filter;
import com.mapr.fs.cli.proto.CLIProto.FilterOp;
import com.mapr.fs.cli.proto.CLIProto.Limiter;
import com.mapr.fs.proto.Common;
import com.mapr.fs.proto.Security.CredentialsMsg;
import com.mapr.security.JNISecurity;
import com.mapr.security.MaprSecurityException;

public class VolumeMirrorCommands extends CLIBaseClass implements CLIInterface {

  public class MirrorVolumeInfo {
    public String volumeName;
    public String clusterName;
    public int err;
    public VolumeProperties volProperties;

    public MirrorVolumeInfo(String volumeName, String clusterName) {
      this.volumeName = volumeName;
      this.clusterName = clusterName;
    }

  }

  private static final Logger LOG = Logger
      .getLogger(VolumeMirrorCommands.class);

  public static final String MIRROR_VOLUME_PARAM_NAME = "name";
  public static final String ROLLFWD_POST_MIRROR = "rollforward";
  public static final String DEL_SRCSNAP = "deletesourcesnap";
  public static final String MIRROR_VERBOSE_OUTPUT = "verbose";
  public static final String IS_FULL_MIRROR = "full";
  public static final String MULTI_ARG_SEP = ",";

  CredentialsMsg srcCreds;
  CredentialsMsg dstCreds;
  private String clusterName;
  boolean printVerboseMsg;
  static boolean isHardMount = false;

  public static final String startMirrorUsage = "volume mirror start -name volname [-full <true|false>] "
      + "[-cluster clustername]";

  public static final String stopMirrorUsage = "volume mirror stop -name volname "
      + "[-cluster clustername]";

  public static final String pushMirrorUsage = "volume mirror push -name volname "
      + "[-cluster clustername] "
      + "[-verbose <true|false> Default:true (if true then print the mirror progress)";

  public static Map<String, BaseInputParameter> baseParams = new ImmutableMap.Builder<String, BaseInputParameter>()
      .put(
          MapRCliUtil.CLUSTER_NAME_PARAM,
          new TextInputParameter(MapRCliUtil.CLUSTER_NAME_PARAM,
              "cluster_name", CLIBaseClass.NOT_REQUIRED, null)).build();

  public static final CLICommand mirrorStartCommand = new CLICommand("start",
      "", VolumeMirrorCommands.class, ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
          .putAll(VolumeMirrorCommands.baseParams)
          .put(
              VolumeMirrorCommands.MIRROR_VOLUME_PARAM_NAME,
              new TextInputParameter(
                  VolumeMirrorCommands.MIRROR_VOLUME_PARAM_NAME, "name",
                  CLIBaseClass.REQUIRED, null))
          .put(
              VolumeMirrorCommands.IS_FULL_MIRROR,
              new BooleanInputParameter(
                  VolumeMirrorCommands.IS_FULL_MIRROR, "<true|false>",
                  CLIBaseClass.NOT_REQUIRED, false/* default value */))
          .put(
              VolumeMirrorCommands.ROLLFWD_POST_MIRROR,
              new BooleanInputParameter(
                  VolumeMirrorCommands.ROLLFWD_POST_MIRROR, "rollforward",
                  CLIBaseClass.NOT_REQUIRED, true/* default value */)
                  .setInvisible(true))
          .put(
              VolumeMirrorCommands.DEL_SRCSNAP,
              new BooleanInputParameter(
                  VolumeMirrorCommands.DEL_SRCSNAP, "delsourcesnap",
                  CLIBaseClass.NOT_REQUIRED, true/* default value */)
                  .setInvisible(true)).build(), null)
      .setShortUsage(startMirrorUsage);

  public static final CLICommand mirrorStopCommand = new CLICommand("stop", "",
      VolumeMirrorCommands.class, ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
          .putAll(VolumeMirrorCommands.baseParams)
          .put(
              VolumeMirrorCommands.MIRROR_VOLUME_PARAM_NAME,
              new TextInputParameter(
                  VolumeMirrorCommands.MIRROR_VOLUME_PARAM_NAME, "name",
                  CLIBaseClass.REQUIRED, null)).build(), null)
      .setShortUsage(stopMirrorUsage);

  public static final CLICommand mirrorPushCommand = new CLICommand("push", "",
      VolumeMirrorCommands.class, ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
          .putAll(VolumeMirrorCommands.baseParams)
          .put(
              VolumeMirrorCommands.MIRROR_VOLUME_PARAM_NAME,
              new TextInputParameter(
                  VolumeMirrorCommands.MIRROR_VOLUME_PARAM_NAME, "name",
                  CLIBaseClass.REQUIRED, null))
          .put(
              VolumeMirrorCommands.MIRROR_VERBOSE_OUTPUT,
              new BooleanInputParameter(
                  VolumeMirrorCommands.MIRROR_VERBOSE_OUTPUT, "verbose",
                  CLIBaseClass.NOT_REQUIRED, true)).build(), null)
      .setShortUsage(pushMirrorUsage);

  public static CLICommand[] mirrorCommandsArray = { mirrorStartCommand,
      mirrorStopCommand, mirrorPushCommand, };

  public static CLICommand mirrorCommands = new CLICommand("mirror", "mirror",
      CLIUsageOnlyCommand.class, ExecutionTypeEnum.NATIVE, mirrorCommandsArray)
      .setShortUsage("mirror [start|stop|push]");

  public VolumeMirrorCommands(ProcessedInput input, CLICommand cliCommand) {
    super(input, cliCommand);
  }

  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException {

    if (LOG.isDebugEnabled()) {
      LOG.debug("ExecuteRealCommand" + cliCommand.getCommandName());
    }

    init();

    if (cliCommand.getCommandName().equalsIgnoreCase("start")) {
      try {
        return startMirror();
      } catch (Exception e) {
        /**
        * <MAPR_ERROR>
        * Message:Start Mirror Exception 
        * Function:VolumeMirrorCommands.executeRealCommand()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */

        throw new CLIProcessingException("Start Mirror Exception ", e);
      }
    } else if (cliCommand.getCommandName().equalsIgnoreCase("stop")) {
      try {
        return stopMirror();
      } catch (Exception e) {
        /**
        * <MAPR_ERROR>
        * Message:Stop Mirror Exception
        * Function:VolumeMirrorCommands.executeRealCommand()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        throw new CLIProcessingException("Stop Mirror Exception", e);
      }
    } else if (cliCommand.getCommandName().equalsIgnoreCase("push")) {
      try {
        return pushMirror();
      } catch (Exception e) {
        /**
        * <MAPR_ERROR>
        * Message:Push Mirror Exception
        * Function:VolumeMirrorCommands.executeRealCommand()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        throw new CLIProcessingException("Push Mirror Exception", e);
      }
    }
    return null;
  }

  private CommandOutput startMirror() throws CLIProcessingException {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    output.setOutput(out);

    List<String> volumeNames = new ArrayList<String>();
    List<MirrorVolumeInfo> volumeMirrorList = new ArrayList<MirrorVolumeInfo>();
    List<MirrorVolumeInfo> volumeNoEntList = new ArrayList<MirrorVolumeInfo>();
    List<MirrorVolumeInfo> volumeNoSrcVolList = new ArrayList<MirrorVolumeInfo>();
    List<MirrorVolumeInfo> volumeNotMirrorList = new ArrayList<MirrorVolumeInfo>();
    List<MirrorVolumeInfo> volumeFailedStartList = new ArrayList<MirrorVolumeInfo>();
    List<MirrorVolumeInfo> volumeNoCrossClusterMirrorList = new ArrayList<MirrorVolumeInfo>();
    List<MirrorVolumeInfo> volumeNoSrcLicenseList = new ArrayList<MirrorVolumeInfo>();
    List<MirrorVolumeInfo> volumeNoDestLicenseList = new ArrayList<MirrorVolumeInfo>();
    List<MirrorVolumeInfo> volumeNoSrcEntList = new ArrayList<MirrorVolumeInfo>();
    List<MirrorVolumeInfo> volumeNoSrcPermList = new ArrayList<MirrorVolumeInfo>();
    List<MirrorVolumeInfo> volumeNoDestPermList = new ArrayList<MirrorVolumeInfo>();
    List<MirrorVolumeInfo> volumeMirrorInProgressList = new ArrayList<MirrorVolumeInfo>();
    List<MirrorVolumeInfo> finalMirrorList = new ArrayList<MirrorVolumeInfo>();

    boolean rollForwardPostMirror;
    boolean delSrcSnapshot;
    boolean isFullMirror;
    String mirrorVolumeName;

    rollForwardPostMirror = getParamBooleanValue(
        VolumeMirrorCommands.ROLLFWD_POST_MIRROR, 0);

    delSrcSnapshot = getParamBooleanValue(
        VolumeMirrorCommands.DEL_SRCSNAP, 0);

    isFullMirror = getParamBooleanValue(
        VolumeMirrorCommands.IS_FULL_MIRROR, 0);

    mirrorVolumeName = getParamTextValue(
        VolumeMirrorCommands.MIRROR_VOLUME_PARAM_NAME, 0);

    // Split the input volumes to list of volumes
    if (!mirrorVolumeName.contains(MULTI_ARG_SEP)) {
      volumeNames.add(mirrorVolumeName);
    } else {
      volumeNames.addAll(Arrays.asList(mirrorVolumeName.split(MULTI_ARG_SEP)));
    }

    for (String volumeName : volumeNames) {
      MirrorVolumeInfo mInfo = new MirrorVolumeInfo(volumeName, clusterName);
      int status = ValidateMirrorVolume(mInfo, getUserCredentials());

      if (status == 0) {
        MirrorStatus mStatus = mInfo.volProperties.getMirrorInfo()
            .getMirrorStatus();
        if ((mStatus != MirrorStatus.STATE_MIRROR_COMPLETE)
            && (mStatus != MirrorStatus.STATE_MIRROR_FAILED) &&
            mStatus != MirrorStatus.STATE_CONVERT_COMPLETE) {
          LOG.info("Volume " + volumeName
              + " mirroring is already in progress " + " cluster "
              + clusterName);
          volumeMirrorInProgressList.add(mInfo);
        } else if (mInfo.volProperties.getMirrorInfo().getSrcVolumeId() == 0) {
          LOG.info("Volume " + volumeName + " doesn't have src volume id " + " cluster "
              + clusterName + ". User need to use volume modify command to set the src "
              + " volume information in the volume before starting the mirroring");
          volumeNoSrcVolList.add(mInfo);
        } else {
          LOG.info("Volume " + volumeName + " is a valid mirror "
              + "volume in cluster " + clusterName);
          volumeMirrorList.add(mInfo);
        }
      } else if (status == Errno.ENOENT) {
        /**
        * <MAPR_ERROR>
        * Message:Volume <volume> doesn't exist in cluster <cluster>
        * Function:VolumeMirrorCommands.startMirror()
        * Meaning:The specified volume does not exist.
        * Resolution:Check the volume name and the command syntax, and try again.
        * </MAPR_ERROR>
        */
        LOG.error("Volume " + volumeName + " doesn't exist in cluster "
            + clusterName);
        volumeNoEntList.add(mInfo);
      } else if (status == Errno.EINVAL) {
        /**
        * <MAPR_ERROR>
        * Message:Volume <volume> is not a mirror volume in <cluster>
        * Function:VolumeMirrorCommands.startMirror()
        * Meaning:The specified mirror volume does not exist.
        * Resolution:Check the volume name and the command syntax, and try again.
        * </MAPR_ERROR>
        */
        LOG.error("Volume " + volumeName + " is not a mirror volume in "
            + " cluster " + clusterName);
        volumeNotMirrorList.add(mInfo);
      } else if (status == Errno.ENOTLICENSED) {
        /**
        * <MAPR_ERROR>
        * Message:No mirror license in cluster <cluster> for mirroring of volume <volume>
        * Function:VolumeMirrorCommands.startMirror()
        * Meaning:The cluster does not have a license that supports mirroring.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        LOG.error("No mirror license in cluster " + clusterName
            + " for mirroring of volume " + volumeName);
        volumeNoDestLicenseList.add(mInfo);
      } else if (status == Errno.E_NOT_ENOUGH_PRIVILEGES) {
        /**
        * <MAPR_ERROR>
        * Message:No permission to mirror restore in cluster <cluster> for mirroring of volume <volume>
        * Function:VolumeMirrorCommands.startMirror()
        * Meaning:You do not have sufficient permissions to start mirroring the specified volume.
        * Resolution:Contact the cluster administrator or log in as a different user.
        * </MAPR_ERROR>
        */
        LOG.error("No permission to mirror restore in cluster " + clusterName
            + " for mirroring of volume " + volumeName);
        volumeNoDestPermList.add(mInfo);
      } else {
        /**
        * <MAPR_ERROR>
        * Message:Failed to get the mirror information for Volume <volume> from CLDB of cluster <cluster>. status code <status>
        * Function:VolumeMirrorCommands.startMirror()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        LOG.error("Failed to get the mirror information for Volume "
            + volumeName + " from CLDB of cluster " + clusterName
            + ". status code " + status);
        mInfo.err = status;
        volumeFailedStartList.add(mInfo);
      }
    }

    for (MirrorVolumeInfo mInfo : volumeMirrorList) {
      String volumeName = mInfo.volumeName;
      int volumeId= mInfo.volProperties.getVolumeId();
      int status = startMirror(clusterName, volumeName,
                      volumeId, rollForwardPostMirror, delSrcSnapshot, isFullMirror);

      if (status != 0) {
        /**
        * <MAPR_ERROR>
        * Message:Failed to start mirroring for volume <volume> in cluster <cluster>. Failure status code <status>
        * Function:VolumeMirrorCommands.startMirror()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        LOG.error("Failed to start mirroring for volume" + volumeName
            + " in cluster " + clusterName + ". Failure status code " + status);
        mInfo.err = status;
        volumeFailedStartList.add(mInfo);
      } else {
        finalMirrorList.add(mInfo);
      }
    }

    String msgString;
    // Log message for successful volumes
    if (finalMirrorList.size() > 0) {
      String successVolumeNames = "";

      MirrorVolumeInfo mInfo;
      int i = 0;
      for (i = 0; i < finalMirrorList.size() - 1; ++i) {
        mInfo = finalMirrorList.get(i);
        successVolumeNames += "'" + mInfo.volumeName + "', ";
      }
      mInfo = finalMirrorList.get(i);
      successVolumeNames += "'" + mInfo.volumeName + "' ";

      LOG.info("Starting mirroring operation for volumes : "
          + successVolumeNames);

      if (volumeMirrorList.size() > 20) {
        msgString = "Started mirror operation for " + volumeMirrorList.size()
            + " volumes.";
      } else {
        msgString = "Started mirror operation for volume(s) "
            + successVolumeNames;
      }
      out.addNode(new OutputNode("messages", msgString));
    }

    String s = "Start mirror operation for ";
    /**
    * <MAPR_ERROR>
    * Message:Volume not found
    * Function:VolumeMirrorCommands.startMirror()
    * Meaning:The specified volume could not be found.
    * Resolution:Check the volume name and the command syntax, and try again.
    * </MAPR_ERROR>
    */
    PrintErrorMsgForVolumeList(s, volumeNoEntList, "Volume not found", out);
    /**
    * <MAPR_ERROR>
    * Message:not a mirror volume
    * Function:VolumeMirrorCommands.startMirror()
    * Meaning:The specified volume is not a mirror volume.
    * Resolution:Check the volume name and the command syntax, and try again.
    * </MAPR_ERROR>
    */
    PrintErrorMsgForVolumeList(s, volumeNotMirrorList, "not a mirror volume",
        out);
    /**
    * <MAPR_ERROR>
    * Message:Operation not permitted
    * Function:VolumeMirrorCommands.startMirror()
    * Meaning:You do not have sufficient permissions to start mirroring the specified volume.
    * Resolution:Contact the cluster administrator or log in as a different user.
    * </MAPR_ERROR>
    */
    PrintErrorMsgForVolumeList(s, volumeNoDestPermList,
        "Operation not permitted", out);
    /**
    * <MAPR_ERROR>
    * Message:No license for mirroring
    * Function:VolumeMirrorCommands.startMirror()
    * Meaning:The cluster does not have a license that supports mirroring.
    * Resolution:Contact technical support.
    * </MAPR_ERROR>
    */
    PrintErrorMsgForVolumeList(s, volumeNoDestLicenseList,
        "No license for mirroring", out);
    /**
    * <MAPR_ERROR>
    * Message:Mirroring already in progress
    * Function:VolumeMirrorCommands.startMirror()
    * Meaning:Mirroring could not be started because it is already in progress for the specified volume.
    * Resolution:Wait until mirroring completes.
    * </MAPR_ERROR>
    */
    PrintErrorMsgForVolumeList(s, volumeMirrorInProgressList,
        "Mirroring already in progress", out);
    /**
    * <MAPR_ERROR>
    * Message:Source volume not found
    * Function:VolumeMirrorCommands.startMirror()
    * Meaning:The specified source volume could not be found.
    * Resolution:Check the volume name and the command syntax, and try again.
    * </MAPR_ERROR>
    */
    PrintErrorMsgForVolumeList(s, volumeNoSrcEntList,
        "Source volume not found", out);
    /**
    * <MAPR_ERROR>
    * Message:Operation not permitted on source volume
    * Function:VolumeMirrorCommands.startMirror()
    * Meaning:You do not have sufficient permissions to start mirroring the specified volume.
    * Resolution:Contact the cluster administrator or log on as a different user.
    * </MAPR_ERROR>
    */
    PrintErrorMsgForVolumeList(s, volumeNoSrcPermList,
        "Operation not permitted on source volume", out);
    /**
    * <MAPR_ERROR>
    * Message:Source volume not found
    * Function:VolumeMirrorCommands.startMirror()
    * Meaning:The specified source volume could not be found.
    * Resolution:Check the volume name and the command syntax, and try again.
    * </MAPR_ERROR>
    */
    PrintErrorMsgForVolumeList(s, volumeNoSrcEntList,
        "Source volume not found", out);
    /**
    * <MAPR_ERROR>
    * Message:No license for mirroring on source cluster
    * Function:VolumeMirrorCommands.startMirror()
    * Meaning:The cluster does not have a license that supports mirroring.
    * Resolution:Contact technical support.
    * </MAPR_ERROR>
    */
    PrintErrorMsgForVolumeList(s, volumeNoSrcLicenseList,
        "No license for mirroring on source cluster", out);

    PrintErrorMsgForVolumeList(s, volumeNoCrossClusterMirrorList,
        "Cross cluster mirroring from MapR security enabled "
        +"source cluster is not supported in this release", out);
    /**
    * <MAPR_ERROR>
    * Message:volume is a dump restored volume it is not associated with any source volume for mirroring
    * Function:VolumeMirrorCommands.startMirror()
    * Meaning:The cluster does not have a license that supports mirroring.
    * Resolution:Contact technical support.
    * </MAPR_ERROR>
    */
    PrintErrorMsgForVolumeList(s, volumeNoSrcVolList,
        "Not attached with a source volume for mirroring. "
        + " Use \"maprcli volume modify -source \" command to setup the source volume", out);

    if (volumeFailedStartList.size() > 0) {
      int err = volumeFailedStartList.get(0).err;
      PrintErrorMsgForVolumeList(s, volumeFailedStartList, "errcode " + err,
          out);
    }
    return output;
  }

  private void PrintErrorMsgForVolumeList(String msgPrefix,
      List<MirrorVolumeInfo> list, String errString, OutputHierarchy out) {

    String msgString;
    if (list.size() > 0) {
      String volumeNames = "";
      MirrorVolumeInfo mInfo;
      int i = 0;
      for (i = 0; i < list.size() - 1; ++i) {
        mInfo = list.get(i);
        volumeNames += "'" + mInfo.volumeName + "', ";
      }
      mInfo = list.get(i);
      volumeNames += "'" + mInfo.volumeName + "' ";

      if (list.size() > 20) {
        msgString = msgPrefix + list.size() + " volumes :" + errString;
      } else {
        msgString = msgPrefix + volumeNames + "failed : " + errString;
      }
      out.addError(new OutputError(Errno.EOPFAILED, msgString));
    }
  }

  private int startMirror(String clusterName, String volumeName,
      int volumeId,
      boolean rollForwardPostMirror,
      boolean delSrcSnapshot,
      boolean isFullMirror) throws CLIProcessingException {

    MirrorStartRequest.Builder req = MirrorStartRequest.newBuilder();
    MirrorStartResponse resp = null;
    byte[] data = null;
    req.setMirrorType(MirrorType.MIRROR_TYPE_LIVE);
    req.setVolumeName(volumeName);
    req.setVolumeId(volumeId);

    // Set the same credentials to be used for both
    // source and destination cluster of mirroring
    req.setSrcCreds(getUserCredentials());
    req.setDestCreds(getUserCredentials());

    if (clusterName != null) {
      req.setClusterName(clusterName);
    }

    req.setRollForwardPostMirror(rollForwardPostMirror);
    req.setDeleteSrcSnapshot(delSrcSnapshot);
    req.setIsFullMirror(isFullMirror);

    try {
      data = sendRequest(clusterName,
          Common.MapRProgramId.CldbProgramId.getNumber(),
          CLDBProto.CLDBProg.MirrorStartProc.getNumber(), req.build(),
          MirrorStartResponse.class, isHardMount);
      if (data == null) {
        /**
        * <MAPR_ERROR>
        * Message:Exception while processing RPC
        * Function:VolumeMirrorCommands.startMirror()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        throw new CLIProcessingException("Exception while processing RPC");
      }
      resp = MirrorStartResponse.parseFrom(data);
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
        "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      /**
      * <MAPR_ERROR>
      * Message:Exception while sending RPC to cluster
      * Function:VolumeMirrorCommands.startMirror()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      throw new CLIProcessingException("Exception while sending RPC to cluster"
          + clusterName, e);
    }

    if (resp.getStatus() != 0) {
      /**
      * <MAPR_ERROR>
      * Message:Mirror start RPC to CLDB for volume <volume>@<cluster> failed with status <status>
      * Function:VolumeMirrorCommands.startMirror()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      LOG.error("Mirror start RPC to CLDB for volume " + volumeName + "@"
          + clusterName + " failed with status " + resp.getStatus());
      return resp.getStatus();
    }
    return 0;
  }

  private CommandOutput stopMirror() throws CLIProcessingException {

    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    output.setOutput(out);

    List<String> volumeNames = new ArrayList<String>();
    List<MirrorVolumeInfo> volumeMirrorList = new ArrayList<MirrorVolumeInfo>();
    List<MirrorVolumeInfo> inProgressMirrorList = new ArrayList<MirrorVolumeInfo>();
    List<MirrorVolumeInfo> volumeNoEntList = new ArrayList<MirrorVolumeInfo>();
    List<MirrorVolumeInfo> volumeNotMirrorList = new ArrayList<MirrorVolumeInfo>();
    List<MirrorVolumeInfo> volumeFailedStartList = new ArrayList<MirrorVolumeInfo>();
    List<MirrorVolumeInfo> volumeNoDestLicenseList = new ArrayList<MirrorVolumeInfo>();
    List<MirrorVolumeInfo> volumeNoDestPermList = new ArrayList<MirrorVolumeInfo>();
    List<MirrorVolumeInfo> finalMirrorList = new ArrayList<MirrorVolumeInfo>();

    String mirrorVolumeName;
    mirrorVolumeName = getParamTextValue(VolumeCommands.RW_VOLUME_PARAM_NAME, 0);

    // Split the input volumes to list of volumes
    if (!mirrorVolumeName.contains(MULTI_ARG_SEP)) {
      volumeNames.add(mirrorVolumeName);
    } else {
      volumeNames.addAll(Arrays.asList(mirrorVolumeName.split(MULTI_ARG_SEP)));
    }

    for (String volumeName : volumeNames) {
      MirrorVolumeInfo mInfo = new MirrorVolumeInfo(volumeName, clusterName);
      int status = ValidateMirrorVolume(mInfo, getUserCredentials());

      if (status == 0) {
        MirrorStatus mStatus = mInfo.volProperties.getMirrorInfo()
            .getMirrorStatus();
        if ((mStatus == MirrorStatus.STATE_MIRROR_COMPLETE)
            || (mStatus == MirrorStatus.STATE_CONVERT_COMPLETE)
            || (mStatus == MirrorStatus.STATE_MIRROR_FAILED)) {
          LOG.info("Volume " + volumeName + " mirroring is already stopped"
              + " cluster " + clusterName + "status " + mStatus);
          finalMirrorList.add(mInfo);
        } else if (mInfo.volProperties.getMirrorInfo()
                   .getStopMirrorInProgress()) {
          LOG.info("Volume " + volumeName + " mirror stop is in progress"
              + " cluster " + clusterName + "status " + mStatus);
          inProgressMirrorList.add(mInfo);
        } else {
          LOG.info("Volume " + volumeName + " is a valid mirror "
              + "volume in cluster " + clusterName);
          volumeMirrorList.add(mInfo);
        }
      } else if (status == Errno.ENOENT) {
        /**
        * <MAPR_ERROR>
        * Message:Volume <volume> doesn't exist in cluster <cluster>
        * Function:VolumeMirrorCommands.stopMirror()
        * Meaning:The specified volume does not exist.
        * Resolution:Check the volume name and the command syntax, and try again.
        * </MAPR_ERROR>
        */
        LOG.error("Volume " + volumeName + " doesn't exist in cluster "
            + clusterName);
        volumeNoEntList.add(mInfo);
      } else if (status == Errno.EINVAL) {
        /**
        * <MAPR_ERROR>
        * Message:Volume <volume> is not a mirror volume in <cluster>
        * Function:VolumeMirrorCommands.stopMirror()
        * Meaning:The specified mirror volume does not exist.
        * Resolution:Check the volume name and the command syntax, and try again.
        * </MAPR_ERROR>
        */
        LOG.error("Volume " + volumeName + " is not a mirror volume in "
            + " cluster " + clusterName);
        volumeNotMirrorList.add(mInfo);
      } else if (status == Errno.ENOTLICENSED) {
        /**
        * <MAPR_ERROR>
        * Message:No mirror license in cluster <cluster> for mirroring of volume <volume>
        * Function:VolumeMirrorCommands.stopMirror()
        * Meaning:The cluster does not have a license that supports mirroring.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        LOG.error("No mirror license in cluster " + clusterName
            + " for mirroring of volume " + volumeName);
        volumeNoDestLicenseList.add(mInfo);
      } else if (status == Errno.E_NOT_ENOUGH_PRIVILEGES) {
        /**
        * <MAPR_ERROR>
        * Message:No permission to mirror restore in cluster <cluster> for mirroring of volume <volume>
        * Function:VolumeMirrorCommands.stopMirror()
        * Meaning:You do not have sufficient permissions to start mirroring the specified volume.
        * Resolution:Contact the cluster administrator or log on as a different user.
        * </MAPR_ERROR>
        */
        LOG.error("No permission to mirror restore in cluster " + clusterName
            + " for mirroring of volume " + volumeName);
        volumeNoDestPermList.add(mInfo);
      } else {
        /**
        * <MAPR_ERROR>
        * Message:Failed to get the mirror information for Volume <volume> from CLDB of cluster <cluster>. status code <status>
        * Function:VolumeMirrorCommands.stopMirror()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        LOG.error("Failed to get the mirror information for Volume "
            + volumeName + " from CLDB of cluster " + clusterName
            + ". status code " + status);
        mInfo.err = status;
        volumeFailedStartList.add(mInfo);
      }
    }
    for (MirrorVolumeInfo mInfo : volumeMirrorList) {
      String volumeName = mInfo.volumeName;
      int volumeId= mInfo.volProperties.getVolumeId();
      int status = stopMirror(clusterName, volumeName, volumeId);
      if (status != 0) {
        /**
        * <MAPR_ERROR>
        * Message:Failed to stop mirroring for volume <volume> in cluster <cluster>. Failure status code <status>
        * Function:VolumeMirrorCommands.stopMirror()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        LOG.error("Failed to stop mirroring for volume " + volumeName
            + " in cluster " + clusterName + ". Failure status code " + status);
        mInfo.err = status;
        volumeFailedStartList.add(mInfo);
      } else {
        finalMirrorList.add(mInfo);
      }
    }

    String msgString;
    // Log message for successful volumes
    if (finalMirrorList.size() > 0) {
      String successVolumeNames = "";
      MirrorVolumeInfo mInfo;
      int i = 0;
      for (i = 0; i < finalMirrorList.size() - 1; ++i) {
        mInfo = finalMirrorList.get(i);
        successVolumeNames += "'" + mInfo.volumeName + "', ";
      }
      mInfo = finalMirrorList.get(i);
      successVolumeNames += "'" + mInfo.volumeName + "' ";
      LOG.info("Stopped mirroring operation for volumes : "
          + successVolumeNames);

      if (finalMirrorList.size() > 20) {
        msgString = "Stopped mirror operation for " + volumeMirrorList.size()
            + " volumes.";
      } else {
        msgString = "Stopped mirror operation for " + successVolumeNames;
      }
      out.addNode(new OutputNode("messages", msgString));
    }
    String s = "Stop mirror operation for ";
    /**
    * <MAPR_ERROR>
    * Message:Volume not found
    * Function:VolumeMirrorCommands.stopMirror()
    * Meaning:The specified volume mirror stop is in progress.
    * Resolution:Wait for next start of mirror process.
    * </MAPR_ERROR>
    */
    PrintErrorMsgForVolumeList(s, inProgressMirrorList,
                               "mirror stop in progress", out);
    /**
    * <MAPR_ERROR>
    * Message:Volume not found
    * Function:VolumeMirrorCommands.stopMirror()
    * Meaning:The specified volume could not be found.
    * Resolution:Check the volume name and the command syntax, and try again.
    * </MAPR_ERROR>
    */
    PrintErrorMsgForVolumeList(s, volumeNoEntList, "Volume not found", out);
    /**
    * <MAPR_ERROR>
    * Message:not a mirror volume
    * Function:VolumeMirrorCommands.stopMirror()
    * Meaning:The specified volume is not a mirror volume.
    * Resolution:Check the volume name and the command syntax, and try again.
    * </MAPR_ERROR>
    */
    PrintErrorMsgForVolumeList(s, volumeNotMirrorList, "not a mirror volume",
        out);
    /**
    * <MAPR_ERROR>
    * Message:Operation not permitted
    * Function:VolumeMirrorCommands.stopMirror()
    * Meaning:You do not have sufficient permissions to start mirroring the specified volume.
    * Resolution:Contact the cluster administrator or log on as a different user.
    * </MAPR_ERROR>
    */
    PrintErrorMsgForVolumeList(s, volumeNoDestPermList,
        "Operation not permitted", out);
    /**
    * <MAPR_ERROR>
    * Message:No license for mirroring
    * Function:VolumeMirrorCommands.stopMirror()
    * Meaning:The cluster does not have a license that supports mirroring.
    * Resolution:Contact technical support.
    * </MAPR_ERROR>
    */
    PrintErrorMsgForVolumeList(s, volumeNoDestLicenseList,
        "No license for mirroring", out);

    if (volumeFailedStartList.size() > 0) {
      int err = volumeFailedStartList.get(0).err;
      if (err == Errno.EALREADY)
        PrintErrorMsgForVolumeList(s, volumeFailedStartList, "Operation " +
                                   "already in progress", out);
      else if (err == Errno.EINPROGRESS)
        PrintErrorMsgForVolumeList(s, volumeFailedStartList, "Other Operation " +
                                   "already in progress. Please check logs.", out);
      else
        PrintErrorMsgForVolumeList(s, volumeFailedStartList, "errcode " + err,
          out);
    }
    return output;
  }

  private int stopMirror(String clusterName, String volumeName, int volumeId)
      throws CLIProcessingException {

    LOG.info("Stopping mirror for volume " + volumeName
               + "@" + clusterName + " volId " + volumeId);

    MirrorStopRequest.Builder req = MirrorStopRequest.newBuilder();
    MirrorStopResponse resp = null;
    byte[] data = null;

    req.setVolumeName(volumeName);
    req.setVolumeId(volumeId);
    req.setSrcCreds(getUserCredentials());
    req.setDestCreds(getUserCredentials());

    if (clusterName != null) {
      req.setClusterName(clusterName);
    }

    try {
      data = sendRequest(clusterName,
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProto.CLDBProg.MirrorStopProc.getNumber(), req.build(),
            MirrorStopResponse.class, isHardMount);
      if (data == null) {
        /**
        * <MAPR_ERROR>
        * Message:Exception while processing RPC
        * Function:VolumeMirrorCommands.stopMirror()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        throw new CLIProcessingException("Exception while processing RPC");
      }
      resp = MirrorStopResponse.parseFrom(data);
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
        "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      /**
      * <MAPR_ERROR>
      * Message:Exception while sending RPC to cluster
      * Function:VolumeMirrorCommands.stopMirror()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      throw new CLIProcessingException("Exception while sending RPC to cluster"
          + clusterName, e);
    }

    if (resp.getStatus() != 0) {
      /**
      * <MAPR_ERROR>
      * Message:Mirror stop RPC to CLDB for volume <volume>@<cluster> failed with status <status>
      * Function:VolumeMirrorCommands.stopMirror()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      LOG.error("Mirror stop RPC to CLDB for volume " + volumeName + "@"
          + clusterName + " failed with status " + resp.getStatus());
      return resp.getStatus();
    }
    return resp.getStatus();
  }

  void init() throws CLIProcessingException {
    if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
      clusterName = getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0);
      if (!CLDBRpcCommonUtils.getInstance().isValidClusterName(clusterName))
        throw new CLIProcessingException("Invalid cluster: " + clusterName);
    } else {
      clusterName = CLDBRpcCommonUtils.getInstance().getCurrentClusterName();
    }

    if (LOG.isDebugEnabled()) {
      LOG.debug("Using the clustername : " + clusterName);
    }
    srcCreds = getUserCredentials();
    dstCreds = getUserCredentials();
  }

  /*
   * Validate that cluster name is in correct format Currently it accepts the
   * cluster name in format IP address a.b.c.d:port format.
   * 
   * We need to change this to understand the cluster name once we move to the
   * file with mapping from cluster name to CLDB IP address. Return true if the
   * cluster name is valid else false
   */
  public static boolean ValidateClusterName(String clusterName) {
    int index = clusterName.indexOf(':');
    if (index == -1 || index == 0) {
      return false;
    }

    try {
      Integer.parseInt(clusterName.substring(index + 1));
    } catch (NumberFormatException e) {
      return false;
    }
    return true;
  }

  /*
   * Verifies that given volume is a mirror volume in the cluster Returns 0 : If
   * the volume is a mirror volume ENOENT : If volume doesn't exist EINVAL : If
   * the volume is not a mirror volume EEXIST : If validateMirrorInProgress
   * isset and its not mirroring other status : If the CLDB is not accessible or
   * returns some error
   */
  public static int ValidateMirrorVolume(MirrorVolumeInfo mInfo,
      CredentialsMsg creds) throws CLIProcessingException {

    String volumeName = mInfo.volumeName;
    String clusterName = mInfo.clusterName;

    VolumeLookupRequest.Builder req = VolumeLookupRequest.newBuilder()
        .setCreds(creds).setVolumeName(volumeName);
    VolumeLookupResponse resp = null;
    byte[] data = null;

    try {
      data = sendRequest(clusterName,
          Common.MapRProgramId.CldbProgramId.getNumber(),
          CLDBProto.CLDBProg.VolumeLookupProc.getNumber(), req.build(),
          VolumeLookupResponse.class, isHardMount);

      if (data == null) {
        /**
        * <MAPR_ERROR>
        * Message:Exception while processing RPC
        * Function:VolumeMirrorCommands.ValidateMirrorVolume()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        throw new CLIProcessingException("Exception while processing RPC");
      }
      resp = VolumeLookupResponse.parseFrom(data);
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
        "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      /**
      * <MAPR_ERROR>
      * Message:Exception while sending RPC to cluster
      * Function:VolumeMirrorCommands.ValidateMirrorVolume()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      throw new CLIProcessingException("Exception while sending RPC to cluster"
          + clusterName, e);
    }

    if (resp.getStatus() != 0) {
      /**
      * <MAPR_ERROR>
      * Message:VolumeLookup RPC to CLDB for volume <volume>@<cluster> failed with status <status>
      * Function:VolumeMirrorCommands.ValidateMirrorVolume()
      * Meaning:Volume lookup failed. The destination volume might have been removed, or might be in an error state.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      LOG.error("VolumeLookup RPC to CLDB for volume " + volumeName + "@"
          + clusterName + " failed with status " + resp.getStatus());
      return resp.getStatus();
    }

    // Verify that the volume is a mirror volume
    if (!resp.getVolInfo().getVolProperties().getIsMirrorVol()) {
      /**
      * <MAPR_ERROR>
      * Message:volume <volume>@<cluster> is not a mirror volume
      * Function:VolumeMirrorCommands.ValidateMirrorVolume()
      * Meaning:The specified volume is nto a mirror volume.
      * Resolution:Check the volume name and the command syntax, and try again.
      * </MAPR_ERROR>
      */
      LOG.error("volume " + volumeName + "@" + clusterName
          + " is not a mirror volume");
      return Errno.EINVAL;
    }

    // return the volumeProperties to caller
    mInfo.volProperties = resp.getVolInfo().getVolProperties();
    return CheckMirrorPermission(mInfo.volProperties.getVolumeId(),
        mInfo.volProperties.getVolumetype(), clusterName, true, creds);
  }

  /*
   * Verifies that given volume is a mirror volume in the cluster Returns 0 : If
   * the volume is a mirror volume ENOENT : If volume doesn't exist EINVAL : If
   * the volume is not a mirror volume EEXIST : If validateMirrorInProgress
   * isset and its not mirroring other status : If the CLDB is not accessible or
   * returns some error
   */

  public static int ValidateSourceVolume(MirrorVolumeInfo mInfo,
      CredentialsMsg creds) throws CLIProcessingException {

    int volumeId = mInfo.volProperties.getMirrorInfo().getSrcVolumeId();
    String clusterName = mInfo.volProperties.getMirrorInfo()
        .getSrcClusterName();
    return CheckMirrorPermission(volumeId, VolumeType.VTRwConvertibleMirror,
      clusterName, false, creds);
  }

  public static int CheckMirrorPermission(int volumeId, VolumeType vt, 
      String clusterName,
      boolean isRestore, CredentialsMsg creds) throws CLIProcessingException {

    byte[] data = null;

    // Verify if user has permission to mirror the volume
    MirrorDumpPermCheckRequest.Builder mirrorDumpPermCheckReq = MirrorDumpPermCheckRequest
        .newBuilder();
    MirrorDumpPermCheckResponse mirrorDumpPermCheckResp = null;

    mirrorDumpPermCheckReq.setVolumeId(volumeId);
    mirrorDumpPermCheckReq.setVolumetype(vt);
    mirrorDumpPermCheckReq.setCreds(creds);
    if (isRestore) {
      mirrorDumpPermCheckReq.setCanRestore(true);
    } else {
      mirrorDumpPermCheckReq.setCanMirror(true);
    }
    try {
      data = sendRequest(clusterName,
          Common.MapRProgramId.CldbProgramId.getNumber(),
          CLDBProto.CLDBProg.MirrorDumpPermCheckProc.getNumber(),
          mirrorDumpPermCheckReq.build(), MirrorDumpPermCheckResponse.class, isHardMount);

      if (data == null) {
        /**
        * <MAPR_ERROR>
        * Message:Exception while processing RPC
        * Function:VolumeMirrorCommands.CheckMirrorPermission()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        throw new CLIProcessingException("Exception while processing RPC");
      }
      mirrorDumpPermCheckResp = MirrorDumpPermCheckResponse.parseFrom(data);
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
        "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      /**
      * <MAPR_ERROR>
      * Message:Exception while sending RPC to cluster
      * Function:VolumeMirrorCommands.CheckMirrorPermission()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      throw new CLIProcessingException("Exception while sending RPC to cluster"
          + clusterName, e);
    }

    return mirrorDumpPermCheckResp.getStatus();
  }

  private void PrintVerboseMsg(String msg) {
    if (printVerboseMsg) {
      System.out.println(msg);
    }
  }

  private CommandOutput pushMirror() 
            throws CLIProcessingException, InterruptedException {

    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    output.setOutput(out);

    // Set the hard mount flag as push mirror operation need to survive
    // even when the CLDB's are down
    isHardMount = true;

    String volumeName
        = getParamTextValue(VolumeCommands.RW_VOLUME_PARAM_NAME, 0);
    
    printVerboseMsg = getParamBooleanValue(MIRROR_VERBOSE_OUTPUT, 0);

    VolumeLookupRequest.Builder req = VolumeLookupRequest.newBuilder()
                                      .setCreds(getUserCredentials())
                                      .setVolumeName(volumeName);
    
    VolumeLookupResponse resp = null;
    byte[] data = null;
    try {
      data = sendRequest(clusterName,
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProto.CLDBProg.VolumeLookupProc.getNumber(),
            req.build(), VolumeLookupResponse.class, isHardMount);
      if (data == null) {
        /**
        * <MAPR_ERROR>
        * Message:Exception while processing RPC
        * Function:VolumeMirrorCommands.pushMirror()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        throw new CLIProcessingException("Exception while processing RPC");
      }
      resp = VolumeLookupResponse.parseFrom(data);
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
        "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      /**
      * <MAPR_ERROR>
      * Message:Exception while sending RPC to cluster
      * Function:VolumeMirrorCommands.pushMirror()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      throw new CLIProcessingException("Exception while sending RPC to cluster" 
          + clusterName, e);
    }  

    if (resp.getStatus() == Errno.ENOENT) {
      /**
      * <MAPR_ERROR>
      * Message:Volume Push command volume <volume>@<cluster> doesn't exist <status>
      * Function:VolumeMirrorCommands.pushMirror()
      * Meaning:Volume lookup failed. Most likely the volume is removed or is in an error state in the destination cluster.
      * Resolution:Check the volume name and the command syntax, and try again.
      * </MAPR_ERROR>
      */
      LOG.error("Volume Push command volume "
          + volumeName + "@" + clusterName
          + " doesn't exist" + resp.getStatus());
      /**
      * <MAPR_ERROR>
      * Message:volume <volume> doesn't exist
      * Function:VolumeMirrorCommands.pushMirror()
      * Meaning:Volume lookup failed. Most likely the volume is removed or is in an error state in the destination cluster.
      * Resolution:Check the volume name and the command syntax, and try again.
      * </MAPR_ERROR>
      */
      out.addError(new OutputError(Errno.ENOENT, "volume " 
          + volumeName + " doesn't exist"));
      return output;
    }
    
    if (resp.getStatus() != 0) {
      /**
      * <MAPR_ERROR>
      * Message:Volume Push command volume <volume>@<cluster> CLDB returned error in query volume properties, errcode <error>
      * Function:VolumeMirrorCommands.pushMirror()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      LOG.error("Volume Push command volume "
          + volumeName + "@" + clusterName
          + " CLDB returned error in query volume properties, errcode" 
          + resp.getStatus());
      /**
      * <MAPR_ERROR>
      * Message:Could not get volume properties for volume <volume>
      * Function:VolumeMirrorCommands.pushMirror()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      out.addError(new OutputError(resp.getStatus(), 
          "Could not get volume properties for volume " 
          + volumeName));
      return output;
    }

    int status = CheckMirrorPermission(resp.getVolInfo().getVolumeId(), 
        resp.getVolInfo().getVolProperties().getVolumetype(), clusterName, 
        false, getUserCredentials());

    if (status == Errno.ENOTLICENSED) {
      /**
      * <MAPR_ERROR>
      * Message:Volume Push command volume <volume>@<cluster> no mirror license found , errcode <error>
      * Function:VolumeMirrorCommands.pushMirror()
      * Meaning:The cluster does not have a license that supports mirroring.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      LOG.error("Volume Push command volume "
          + volumeName + "@" + clusterName
          + " no mirror license found , errcode" 
          + status);
      /**
      * <MAPR_ERROR>
      * Message: No mirror license for mirroring of <volume>
      * Function:VolumeMirrorCommands.pushMirror()
      * Meaning:The cluster does not have a license that supports mirroring.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      out.addError(new OutputError(status, 
          " No mirror license for mirroring of " 
          + volumeName));
      return output;
    }

    if (status == Errno.E_NOT_ENOUGH_PRIVILEGES) {
      /**
      * <MAPR_ERROR>
      * Message:Volume Push command volume <volume>@<cluster> not enough privileges, errcode <error>
      * Function:VolumeMirrorCommands.pushMirror()
      * Meaning:You do not have sufficient permissions to start mirroring the specified volume.
      * Resolution:Contact the cluster administrator or log on as a different user.
      * </MAPR_ERROR>
      */
      LOG.error("Volume Push command volume "
          + volumeName + "@" + clusterName
          + " not enough privileges, errcode" 
          + status);
      /**
      * <MAPR_ERROR>
      * Message:No permission to mirror <volume>
      * Function:VolumeMirrorCommands.pushMirror()
      * Meaning:You do not have sufficient permissions to start mirroring the specified volume.
      * Resolution:Contact the cluster administrator or log on as a different user.
      * </MAPR_ERROR>
      */
      out.addError(new OutputError(status, 
          " No permission to mirror " 
          + volumeName));
      return output;
    }

    status = PushToMirrors(volumeName, resp.getVolInfo().getVolumeId(), clusterName);
    if (status != 0) {
      /**
      * <MAPR_ERROR>
      * Message:Volume push failed for volume <volume> error <error>
      * Function:VolumeMirrorCommands.pushMirror()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      LOG.error("Volume push failed for volume " + volumeName 
          + " error " + status);
      /**
      * <MAPR_ERROR>
      * Message:Failed to push to mirrors for volume <volume>
      * Function:VolumeMirrorCommands.pushMirror()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      out.addError(new OutputError(status, 
        "Failed to push to mirrors for volume" + volumeName));
    } else {
      out.addNode(new OutputNode("Successfully completed mirror push to "
          + "all local mirrors of volume " + volumeName));
    }
    return output;
  }

  private int PushToMirrors(String volumeName, int volumeId, String cluster)
      throws CLIProcessingException, InterruptedException {

    int status;
    List<VolumeInfo> mirrorVolumes = new ArrayList<VolumeInfo>();

    status = getMirrorVolumes(volumeName, volumeId, cluster, mirrorVolumes /* out param */);

    if (status != 0) {
      return status;
    }
    List<MirrorPushThread> pushThreads = new ArrayList<MirrorPushThread>();
    for (VolumeInfo mirrorVolume : mirrorVolumes) {
      String mirrorVolumeName = mirrorVolume.getVolProperties().getVolumeName();
      MirrorPushThread t = new MirrorPushThread(mirrorVolumeName,
                            mirrorVolume.getVolProperties().getVolumeId(), cluster);
      pushThreads.add(t);
      t.start();
    }

    // Started all the threads not wait for them to complete
    for (MirrorPushThread t : pushThreads) {
      t.join();
      // Thread complete
      LOG.info("Completed mirroring of volume " + t.mirrorVolumeName
          + " status " + t.status);
      if (t.status != 0) {
        status = t.status;
      }
    }
    return status;
  }

  private int getMirrorVolumes(String volumeName, int volumeId, String cluster,
      List<VolumeInfo> retMirrorVolumes) throws CLIProcessingException {

    boolean hasMoreVolumes = true;
    int startIndex = 0;
    int Limit = 40; // Get 40 volumes in one request

    while (hasMoreVolumes) {

      VolumeListRequest.Builder req = VolumeListRequest.newBuilder();
      VolumeListResponse resp;
      List<Filter> filters = new ArrayList<Filter>();
      Filter.Builder filter1 = Filter.newBuilder();
      Filter.Builder filter2 = Filter.newBuilder();

      LOG.info("GetMirrorVolume for volume " + volumeName + " volId " + volumeId);
      filter1.setFieldId(VolumeInfoFields.mirrorSrcVolumeId.getNumber())
          .setFieldOp(FieldOp.EqualTo)
          .setFieldVal(FieldVal.newBuilder().setValSignedInteger32(volumeId))
          .setFilterOp(FilterOp.AND);

      filter2.setFieldId(VolumeInfoFields.mirrorSrcClusterName.getNumber())
          .setFieldOp(FieldOp.EqualTo)
          .setFieldVal(FieldVal.newBuilder().setValString(cluster))
          .setFilterOp(FilterOp.AND);

      filters.add(filter1.build());
      filters.add(filter2.build());

      Limiter.Builder limiter = Limiter.newBuilder();
      limiter.setStart(startIndex).setLimit(Limit);

      req.setColumns(0xffffffff);
      req.setCreds(getUserCredentials());
      req.addAllFilter(filters);
      req.setLimiter(limiter.build());

      byte[] data = null;
      try {
        data = sendRequest(cluster,
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProto.CLDBProg.VolumeListProc.getNumber(), req.build(),
            VolumeListResponse.class, isHardMount);

        if (data == null) {
          /**
          * <MAPR_ERROR>
          * Message:Exception while processing RPC
          * Function:VolumeMirrorCommands.getMirrorVolumes()
          * Meaning:An error occurred.
          * Resolution:Contact technical support.
          * </MAPR_ERROR>
          */
          throw new CLIProcessingException("Exception while processing RPC");
        }
        resp = VolumeListResponse.parseFrom(data);
      } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
      } catch (Exception e) {
        /**
        * <MAPR_ERROR>
        * Message:Exception while sending RPC to cluster <cluster>
        * Function:VolumeMirrorCommands.getMirrorVolumes()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        throw new CLIProcessingException(
            "Exception while sending RPC to cluster " + clusterName, e);
      }
      // There are no mirror volumes for this volume
      // so continue
      if (resp.getStatus() == Errno.ENOENT) {
        hasMoreVolumes = false;
        continue;
      }

      if (resp.getStatus() != 0) {
        /**
        * <MAPR_ERROR>
        * Message:Volume Push command : Failed to get the list  of mirror volumes for volume <volume>@<cluster> , err <status>
        * Function:VolumeMirrorCommands.getMirrorVolumes()
        * Meaning:Volume lookup failed. The volume on the destination cluster might be removed or in an error state.
        * Resolution:Check the volume name and the command syntax, and try again.
        * </MAPR_ERROR>
        */
        LOG.error("Volume Push command : Failed to get the list "
            + " of mirror volumes for volume " + volumeName + "@" + cluster
            + " , err " + resp.getStatus());
        return resp.getStatus();
      }
      retMirrorVolumes.addAll(resp.getVolumesList());
      if (resp.getVolumesCount() >= Limit) {
        hasMoreVolumes = true;
        startIndex += resp.getVolumesCount();
      } else {
        hasMoreVolumes = false;
      }
    }
    return 0;
  }

  public static byte[] sendRequest(String clusterName, int programId, int procedureId,
      MessageLite request, Class<? extends MessageLite> responseClass, 
      boolean isHardMount) throws Exception
  {
    byte[] resp;
    int sleepTimeMillis = 5*1000;
    int i = 0;
    do {
      if (clusterName != null) {
        resp = CLDBRpcCommonUtils.getInstance().sendRequest(
            clusterName, programId, procedureId, request, responseClass);
      } else {
        resp = CLDBRpcCommonUtils.getInstance().sendRequest(
            programId, procedureId, request, responseClass);
      }
      if (isHardMount == false) {
        return resp;
      }
      if (resp == null) {
        i++;
        if (i > 6) {
          i = 6;
        }
        // Keep maxinum time between retry to 5 mins
        Thread.sleep(sleepTimeMillis * (1 << i));
      }
    } while (resp == null);
    return resp;
  }

  public class MirrorPushThread extends Thread {
    public int status;
    public String errString;
    public int mirrorId;
    public String mirrorVolumeName;
    public String mirrorClusterName;
    public int mirrorVolumeId;

    public MirrorPushThread(String volumeName, int volumeId, String clusterName) {
      mirrorVolumeName = volumeName;
      mirrorVolumeId = volumeId;
      mirrorClusterName = clusterName;
    }

    public void run() {
      try {
        // Wait For mirroring to complete
        WaitForMirroringToComplete(true);
        if (status != 0)
          return;

        PrintVerboseMsg("Starting mirroring of volume " + mirrorVolumeName);
        status = startMirror(mirrorClusterName, mirrorVolumeName, mirrorVolumeId,
                             true, true /*delSrcSnap*/, false /*isFullMirror*/);
        if (status != 0)
          return;

        WaitForMirroringToComplete(false);
        if (status != 0)
          return;

        // Push to all downstream volumes
        status = PushToMirrors(mirrorVolumeName, mirrorVolumeId, mirrorClusterName);
      } catch (Exception e) {
        status = Errno.ECOMM;
        /**
        * <MAPR_ERROR>
        * Message:Exception while processing mirroring for volume <volume> <error>
        * Function:VolumeMirrorCommands.run()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        errString = "Exception while processing mirroring for volume "
            + mirrorVolumeName + " " + e.toString();
        return;
      }
      return;
    }

    // when call is successful then retObj returns the
    // current mirrorid of the volume
    private void WaitForMirroringToComplete(boolean justWaitForMirroringStop) throws CLIProcessingException {
      do {
        VolumeLookupRequest.Builder req = VolumeLookupRequest.newBuilder()
            .setCreds(getUserCredentials()).setVolumeName(mirrorVolumeName);

        VolumeLookupResponse resp = null;
        byte[] data = null;
        try {
          data = sendRequest(clusterName,
              Common.MapRProgramId.CldbProgramId.getNumber(),
              CLDBProto.CLDBProg.VolumeLookupProc.getNumber(), req.build(),
              VolumeLookupResponse.class, isHardMount);

          if (data == null) {
            /**
            * <MAPR_ERROR>
            * Message:
            * Function:VolumeMirrorCommands.WaitForMirroringToComplete()
            * Meaning:An error occurred.
            * Resolution:Contact technical support.
            * </MAPR_ERROR>
            */
            throw new CLIProcessingException("Exception while processing RPC");
          }
          resp = VolumeLookupResponse.parseFrom(data);
        } catch (MaprSecurityException e) {
          throw new CLIProcessingException(
            "MaprSecurityException " + "Exception", e);
        } catch (Exception e) {
          /**
          * <MAPR_ERROR>
          * Message:Exception while sending the RPC to CLDB <error>
          * Function:VolumeMirrorCommands.WaitForMirroringToComplete()
          * Meaning:An error occurred.
          * Resolution:Contact technical support.
          * </MAPR_ERROR>
          */
          errString = "Exception while sending the RPC to CLDB " + e.toString();
          status = Errno.ECOMM;
          return;
        }

        if (resp.getStatus() != 0) {
          /**
          * <MAPR_ERROR>
          * Message:Mirroring failed for volume <volume>@<cluster> CLDB returned error in volume lookup response, errcode <error>
          * Function:VolumeMirrorCommands.WaitForMirroringToComplete()
          * Meaning:Volume lookup failed. The volume on the destination cluster might be removed or in an error state.
          * Resolution:Contact technical support.
          * </MAPR_ERROR>
          */
          LOG.error("Mirroring failed for volume " + mirrorVolumeName + "@"
              + clusterName
              + " CLDB returned error in volume lookup response, errcode "
              + resp.getStatus());
          status = resp.getStatus();
          PrintVerboseMsg("Mirroring failed for volume " + mirrorVolumeName
              + " error code : " + status);
          return;
        }

        MirrorInfo mirrorInfo = resp.getVolInfo().getVolProperties()
            .getMirrorInfo();

        if (mirrorInfo.getMirrorStatus() == MirrorStatus.STATE_MIRROR_FAILED) {
          /**
          * <MAPR_ERROR>
          * Message:Mirroring failed for volume <volume>@<cluster> as mirroring is in failed state
          * Function:VolumeMirrorCommands.WaitForMirroringToComplete()
          * Meaning:An error occurred.
          * Resolution:Contact technical support.
          * </MAPR_ERROR>
          */
          LOG.error("Mirroring failed for volume " + mirrorVolumeName + "@"
              + mirrorClusterName + " as mirroring is in failed state");
          PrintVerboseMsg("Mirroring failed for volume " + mirrorVolumeName
              + " because it is already in failed state: ");
          status = Errno.EIO;
          return;
        }

        if (justWaitForMirroringStop) {
          if (mirrorInfo.getMirrorStatus() == MirrorStatus.STATE_MIRROR_COMPLETE) {
            mirrorId = mirrorInfo.getMirrorId();
            LOG.info("Initial mirror id of volume " + mirrorVolumeName + " is "
                + mirrorId);
            return;
          }
        } else {
          // If the volume has move forward the startMirrorId then complete
          if (mirrorInfo.getMirrorId() > mirrorId) {
            LOG.info("Mirror volume " + mirrorVolumeName
                + " mirroring complete mirrorId " + mirrorInfo.getMirrorId()
                + " mirrorState " + mirrorInfo.getMirrorStatus().name());
            PrintVerboseMsg("Mirroring complete for volume " + mirrorVolumeName);
            return;
          }
        }
        // Wait for few seconds before retrying
        LOG.info("Mirror volume " + mirrorVolumeName + " mirrorId "
            + mirrorInfo.getMirrorId() + " mirrorState "
            + mirrorInfo.getMirrorStatus().name());

        try {
          Thread.sleep(10 * 1000);
        } catch (InterruptedException e) {
        }

      } while (true);
    }
  }
}
