/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.cli;


import org.apache.hadoop.fs.Path;
import org.apache.log4j.Logger;

import com.google.common.collect.ImmutableMap;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CLIUsageOnlyCommand;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.TextCommandOutput;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.cliframework.base.inputparams.BooleanInputParameter;
import com.mapr.fs.MapRFileSystem;
import com.mapr.fs.cldb.proto.CLDBProto.VolumeLookupResponse;
import com.mapr.fs.cldb.proto.CLDBProto.VolumeProperties;


public class VolumeLinkCommands extends CLIBaseClass implements CLIInterface {

	private static final Logger LOG = Logger.getLogger(VolumeLinkCommands.class);
  
	public static final String VOLUMELINK_NAME_PARAM_NAME = "volume";
	public static final String VOLUMELINK_PATH_PARAM_NAME = "path";
	public static final String VOLUMELINK_TYPE_PARAM_NAME = "type";
	public static final String VOLUMELINK_HIDDEN_PARAM_NAME = "hidden";
	
  public static final String volumeLinkCreateUsage = 
                    "volume link -volume name -path path [-type writeable|mirror]" +
                    "[-cluster clustername]";

  public static String volumeLinkRemoveUsage = "volume link remove -link";
  
  
  static final CLICommand volumeLinkCreateCommand =
    new CLICommand(
        "create", 
        "usage : " + volumeLinkCreateUsage,
        VolumeLinkCommands.class, ExecutionTypeEnum.NATIVE,
        new ImmutableMap.Builder<String, BaseInputParameter>()
          .putAll(VolumeCommands.baseParams)
          .put(VOLUMELINK_NAME_PARAM_NAME,
              new TextInputParameter(
                  VOLUMELINK_NAME_PARAM_NAME,
                  "volume", 
                  CLIBaseClass.REQUIRED, 
                  null))
          .put(VOLUMELINK_TYPE_PARAM_NAME,
              new TextInputParameter(
                  VOLUMELINK_TYPE_PARAM_NAME,
                  "type <writeable|mirror>", 
                  CLIBaseClass.REQUIRED, 
                  null))
          .put(VOLUMELINK_PATH_PARAM_NAME,
              new TextInputParameter(
                  VOLUMELINK_PATH_PARAM_NAME,
                  "path", 
                  CLIBaseClass.REQUIRED, 
                  null))
          .put(VOLUMELINK_HIDDEN_PARAM_NAME,
              new BooleanInputParameter(
                  VOLUMELINK_HIDDEN_PARAM_NAME,
                  "hidden", 
                  CLIBaseClass.NOT_REQUIRED, 
                  0).setInvisible(true))
          .build(), 
          null)
    .setShortUsage(volumeLinkCreateUsage);
  
  static final CLICommand volumeLinkRemoveCommand = new CLICommand(
      "remove", 
      "usage : " + volumeLinkRemoveUsage,
      VolumeLinkCommands.class, ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
        .putAll(VolumeCommands.baseParams)
        .put(VOLUMELINK_PATH_PARAM_NAME,
            new TextInputParameter(
                VOLUMELINK_PATH_PARAM_NAME,
                "vollink", 
                CLIBaseClass.REQUIRED, 
                null))
        .build(), 
        null)
  .setShortUsage(volumeLinkRemoveUsage);
  
    
  public static CLICommand VolumeLinkCommands = new CLICommand(
      "link", 
      "link", 
      CLIUsageOnlyCommand.class,
      ExecutionTypeEnum.NATIVE, 
      new CLICommand [] {         /* array of subcommands */
        volumeLinkCreateCommand,
        volumeLinkRemoveCommand
       }
  ).setShortUsage("link [create|remove]");
  
  
  public VolumeLinkCommands(ProcessedInput input, CLICommand cliCommand) 
      throws CLIProcessingException {
    super(input, cliCommand);
  } 
  
  
  public CommandOutput executeRealCommand() throws CLIProcessingException {
    
    if (cliCommand.getCommandName().equalsIgnoreCase("create")) {
      try {   	    
        return volumeLinkCreate();
      } catch (Exception e) {
        throw new CLIProcessingException("Send request Exception", e);
      } 
    } else if (cliCommand.getCommandName().equalsIgnoreCase("remove")) {
      try {
        return volumeLinkRemove();
      } catch (Exception e) {
        throw new CLIProcessingException("Send request Exception", e);
      } 
    } 
     
    return new TextCommandOutput(("Volume link command failed").getBytes());
  }  
  
  CommandOutput volumeLinkRemove() throws CLIProcessingException {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    output.setOutput(out);
    
    String path = getParamTextValue(VOLUMELINK_PATH_PARAM_NAME, 0);
    if ((path == null) || path.isEmpty() || !path.matches("/.+")) {                
        out.addError(new OutputError(Errno.EINVAL, "Invalid link")
                           .setField(VOLUMELINK_PATH_PARAM_NAME)
                           .setFieldValue(path));
        return output;
    }
    
    String cluster = null;
    if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
      cluster = getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0);
      if (!CLDBRpcCommonUtils.getInstance().isValidClusterName(cluster)) {
        out.addError(new OutputError(Errno.EUCLUSTER, "Invalid cluster: " + cluster));
        return output;
      }
    }
    
    MapRFileSystem fs = MapRCliUtil.getMapRFileSystem();
    try {    
       int error = fs.deleteVolLink(cluster, path);      
       if (error != 0) {
         out.addError(new OutputError(error, "Failed to remove vollink " + path + 
             ", error: " + Errno.toString(error)));
      }
    } catch (Exception e) {
       out.addError(new OutputError(Errno.EOPFAILED, "Failed to remove vollink " + 
                    path));
       return output;
    }          
    
    return output;
  }  
   
  
  CommandOutput volumeLinkCreate() throws CLIProcessingException {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    output.setOutput(out);
    
    String name = getParamTextValue(VOLUMELINK_NAME_PARAM_NAME, 0).trim();
    if (name.isEmpty()) {
        out.addError(new OutputError(Errno.EINVAL, "Invalid volume name")
                           .setField(VOLUMELINK_NAME_PARAM_NAME)
                           .setFieldValue(name));
        return output;
    }
    
    String path = getParamTextValue(VOLUMELINK_PATH_PARAM_NAME, 0).trim();;
    if (path.isEmpty() || !path.matches("/.+")) {
      out.addError(new OutputError(Errno.EINVAL, "Invalid link")
                         .setField(VOLUMELINK_PATH_PARAM_NAME)
                         .setFieldValue(path));
      return output;
    }
    
    String type = getParamTextValue(VOLUMELINK_TYPE_PARAM_NAME, 0).trim();
    if (type.trim().isEmpty() || 
        (!type.equalsIgnoreCase("writeable") && !type.equalsIgnoreCase("mirror"))) {                
        out.addError(new OutputError(Errno.EINVAL, 
                               "Invalid type, should be writeable OR mirror")
                           .setField(VOLUMELINK_TYPE_PARAM_NAME)
                           .setFieldValue(type));
        return output;
    }
    
   boolean hidden = false;
   if (isParamPresent(VOLUMELINK_HIDDEN_PARAM_NAME))
      hidden = getParamBooleanValue(VOLUMELINK_HIDDEN_PARAM_NAME, 0);

    String cluster = null;
    if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
      cluster = getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0);
      if (!CLDBRpcCommonUtils.getInstance().isValidClusterName(cluster)) {
        out.addError(new OutputError(Errno.EUCLUSTER, "Invalid cluster: " + cluster));
        return output;
      }
    }
      
    VolumeLookupResponse resp = 
      VolumeCommands.volumeLookup(cluster, getUserCredentials(), name, null,isServerCall);
    if (resp.getStatus() != 0) {
      out.addError(new OutputError(Errno.EINVAL, 
          "Could not fetch volume info for volume " + name + 
          ", error: " + Errno.toString(resp.getStatus())));
      return output;
    }
    VolumeProperties vProps = resp.getVolInfo().getVolProperties();
    if (!vProps.getMounted()) {
      out.addError(new OutputError(Errno.EINVAL, 
          "Can only create vol-links to mounted and non-mirror volumes. Volume " + name + 
          " is not mounted"));
      return output;
    }
    if (vProps.getIsMirrorVol()) {
      out.addError(new OutputError(Errno.EINVAL, 
          "Can only create vol-links to mounted and non-mirror volumes. Volume " + name + 
          " is a mirror volume"));
      return output;
    }
        
    MapRFileSystem fs = MapRCliUtil.getMapRFileSystem();
    try {
      boolean writeable = ((type.charAt(0) == 'w') ? true : false);
      int error = fs.createVolLink(cluster, name, new Path(path), writeable,
                                   hidden);
      if (error != 0) {
         out.addError(new OutputError(error, "Failed to create vollink " + name + 
             ", error: " + Errno.toString(error)));
      }
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED, "Failed to create vollink " + name));
      return output;
    }
        
    return output;
  }
}
