/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.cli;

import java.io.IOException;
import java.util.Collections;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;
import com.mapr.security.MaprSecurityException;

import com.google.common.collect.ImmutableMap;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.baseutils.Errno;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CLIUsageOnlyCommand;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.BooleanInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.cliframework.base.inputparams.IntegerInputParameter;
import com.mapr.cliframework.base.inputparams.LongInputParameter;
import com.mapr.cliframework.base.TextCommandOutput;
import com.mapr.cliframework.util.FieldInfo;

import com.mapr.fs.Rpc;
import com.mapr.fs.cldb.conf.CLDBConfiguration;
import com.mapr.fs.cldb.proto.CLDBProto.*;
import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.proto.Common.ContainerReplType;
import com.mapr.fs.cldb.util.Util;
import com.mapr.fs.proto.Common.MapRProgramId;

import com.mapr.baseutils.Errno;
import com.mapr.baseutils.BinaryString;

public class VolumeContainerCommands extends CLIBaseClass implements CLIInterface {
  private static final String VOLUME_CID_PARAM_NAME = "cid";
  private static final String VOLUME_FROMFS_PARAM_NAME = "fromfileserverid";
  private static final String VOLUME_TOFS_PARAM_NAME = "tofileserverid";

  private static final Logger LOG = Logger
      .getLogger(VolumeContainerCommands.class);

  public static final String moveContainerUsage =
    "volume container move [ -cluster cluster_name ] " +
    "-cid cid -fromfileserverid fromfileserverid " +
    "[ -tofileserverid tofileserverid ]";

  public static final String switchMasterUsage =
      "volume container switchmaster -cid cid";

  public static final String switchIntermediateToTailUsage =
      "volume container switchtailwithupstream -cid cid";

  public static Map<String, BaseInputParameter> baseParams =
    new ImmutableMap.Builder<String, BaseInputParameter>()
    .put(MapRCliUtil.CLUSTER_NAME_PARAM, new TextInputParameter(
        MapRCliUtil.CLUSTER_NAME_PARAM, "cluster_name",
        CLIBaseClass.NOT_REQUIRED, null))
    .build();

  // Cmds
  public static final CLICommand moveContainerCommand =
    new CLICommand(
      "move",
      "",
      VolumeContainerCommands.class, ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
        .putAll(VolumeCommands.baseParams)
        .put(VolumeContainerCommands.VOLUME_CID_PARAM_NAME,
             new IntegerInputParameter(
               VolumeContainerCommands.VOLUME_CID_PARAM_NAME,
               VolumeContainerCommands.VOLUME_CID_PARAM_NAME,
               CLIBaseClass.REQUIRED, null))
        .put(VolumeContainerCommands.VOLUME_FROMFS_PARAM_NAME,
             new LongInputParameter(
               VolumeContainerCommands.VOLUME_FROMFS_PARAM_NAME,
               VolumeContainerCommands.VOLUME_FROMFS_PARAM_NAME,
               CLIBaseClass.REQUIRED, null))
        .put(VolumeContainerCommands.VOLUME_TOFS_PARAM_NAME,
             new LongInputParameter(
               VolumeContainerCommands.VOLUME_TOFS_PARAM_NAME,
               VolumeContainerCommands.VOLUME_TOFS_PARAM_NAME,
               CLIBaseClass.NOT_REQUIRED, null))
        .build(),
      null
    ).setShortUsage(moveContainerUsage);

  static final CLICommand switchMasterCommand = new CLICommand(
      "switchmaster",
      "usage : " + switchMasterUsage,
      VolumeContainerCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
        .putAll(VolumeContainerCommands.baseParams)
        .put(VolumeContainerCommands.VOLUME_CID_PARAM_NAME,
             new IntegerInputParameter(
                 VolumeContainerCommands.VOLUME_CID_PARAM_NAME,
                 VolumeContainerCommands.VOLUME_CID_PARAM_NAME,
                 CLIBaseClass.REQUIRED,
                 null))
        .build(),
       null
    ).setShortUsage(switchMasterUsage);

  static final CLICommand switchIntermediateToTailCommand = new CLICommand(
      "switchtailwithupstream",
      "usage : " + switchIntermediateToTailUsage,
      VolumeContainerCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
        .putAll(VolumeContainerCommands.baseParams)
        .put(VolumeContainerCommands.VOLUME_CID_PARAM_NAME,
             new IntegerInputParameter(
                 VolumeContainerCommands.VOLUME_CID_PARAM_NAME,
                 VolumeContainerCommands.VOLUME_CID_PARAM_NAME,
                 CLIBaseClass.REQUIRED,
                 null))
        .build(),
       null
    ).setShortUsage(switchIntermediateToTailUsage);

  public static CLICommand[] containerCommandsArray = {
      moveContainerCommand,
      switchMasterCommand,
      switchIntermediateToTailCommand
  };

  // main command
  public static CLICommand containerCommands =
      new CLICommand(
          "container", "containe",
          CLIUsageOnlyCommand.class,
          CLICommand.ExecutionTypeEnum.NATIVE,
          containerCommandsArray
      ).setShortUsage("volume container [move]");

  public VolumeContainerCommands(ProcessedInput input, CLICommand cliCommand) {
   super(input, cliCommand);
  }

  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException {
    if (LOG.isDebugEnabled()) {
      LOG.debug("ExecuteRealCommand" + cliCommand.getCommandName());
    }

    if (!super.validateInput()) {
      return output;
    }

    try {
      if (cliCommand.getCommandName().equalsIgnoreCase(moveContainerCommand.getCommandName())) {
        return moveContainer();
      } else if (cliCommand.getCommandName().equalsIgnoreCase(switchMasterCommand.getCommandName())) {
        return switchMaster();
      } else if (cliCommand.getCommandName().equalsIgnoreCase(switchIntermediateToTailCommand.getCommandName())) {
        return switchTailWithUpstream();
      } else {
        return new TextCommandOutput("volume unknown sub-command".getBytes());
      }
    } catch (Exception e) {
      /**
      * <MAPR_ERROR>
      * Message:Move Container Exception
      * Function:VolumeContainerCommands.executeRealCommand
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      throw new CLIProcessingException("Move Container Exception", e);
    }
  }

  private CommandOutput moveContainer() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    String cluster = null;
    if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
      cluster = getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0);
    }

    int cid = getParamIntValue(VOLUME_CID_PARAM_NAME, 0);
    long fromfsid = getParamLongValue(VOLUME_FROMFS_PARAM_NAME, 0);

    // Send container move rpc to cldbdb
    boolean done = false;
    ContainerMoveReplicaRequest.Builder reqb = ContainerMoveReplicaRequest.newBuilder()
                                    .setCid(cid)
                                    .setFromServer(fromfsid)
                                    .setCreds(getUserCredentials());
   if (isParamPresent(VOLUME_TOFS_PARAM_NAME)) {
     long tofsid = getParamLongValue(VOLUME_TOFS_PARAM_NAME, 0);
     reqb.setToServer(tofsid);
   }
   ContainerMoveReplicaRequest req = reqb.build();

    try {
      byte[] replyData;
      replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
                    MapRProgramId.CldbProgramId.getNumber(),
                    CLDBProto.CLDBProg.ContainerMoveReplicaProc.getNumber(),
                    req,
                    ContainerMoveReplicaResponse.class);

      if (replyData == null) {
        // most likely can not connect to CLDB
        LOG.error("Couldn't connect to the CLDB service");
        out.addError(new OutputError(Errno.ERPCFAILED, "cldb rpc failed"));
        return output;
      }

      ContainerMoveReplicaResponse resp;
      resp = ContainerMoveReplicaResponse.parseFrom(replyData);

      if (resp.getStatus() == 0) {
        done = true;
      } else {
        out.addError(new OutputError(resp.getStatus(),
                     "MoveContainer failed, Error : " +
                     Errno.toString(resp.getStatus())));
      }
    } catch (Exception e) {
      LOG.error("Container move failed");
      out.addError(new OutputError(Errno.ERPCFAILED, "cldb rpc failed"));
      return output;
    }
    if (done)
      return new TextCommandOutput(("MoveContainer done").getBytes());
    else
      return output;
  }

  static final CLICommand switchMaster = new CLICommand(
      "switchmaster",
      "usage : " + switchMasterUsage,
      VolumeContainerCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
        .putAll(VolumeContainerCommands.baseParams)
        .put(VolumeContainerCommands.VOLUME_CID_PARAM_NAME,
             new IntegerInputParameter(
                 VolumeContainerCommands.VOLUME_CID_PARAM_NAME,
                 VolumeContainerCommands.VOLUME_CID_PARAM_NAME,
                 CLIBaseClass.REQUIRED,
                 null))
        .build(),
       null
    ).setShortUsage(switchMasterUsage);

  private CommandOutput switchMaster() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    int cid = getParamIntValue(VOLUME_CID_PARAM_NAME, 0);

    // lookup container in cldb
    int dbHost, dbPort;
    byte[] replyData;
    String spid;
    ContainerLookupRequest creq = ContainerLookupRequest.newBuilder()
                                    .addContainerId(cid)
                                    .setCreds(getUserCredentials())
                                    .build();
    try {
      replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
                    MapRProgramId.CldbProgramId.getNumber(),
                    CLDBProto.CLDBProg.ContainerLookupProc.getNumber(),
                    creq,
                    ContainerLookupResponse.class);

      if (replyData == null) {
        // most likely can not connect to CLDB
        LOG.error("Couldn't connect to the CLDB service");
        out.addError(new OutputError(Errno.ERPCFAILED, "cldb rpc failed"));
        return output;
      }

      ContainerLookupResponse resp;
      resp = ContainerLookupResponse.parseFrom(replyData);
      if (resp.getStatus() == 0) {
        spid = resp.getContainers(0).getMServer().getSpInfo().getSpId();
      } else {
        out.addError(new OutputError(resp.getStatus(),
                     "Container lookup failed, Error : " +
                     Errno.toString(resp.getStatus())));
        return output;
      }
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      LOG.error("Container lookup failed");
      out.addError(new OutputError(Errno.ERPCFAILED, "cldb rpc failed"));
      return output;
    }


    // Send switch rpc to cldbdb
    boolean done = false;
    ContainerSwitchMasterRequest req = ContainerSwitchMasterRequest.newBuilder()
                                    .setCid(cid)
                                    .setMasterSpid(spid)
                                    .setCreds(getUserCredentials())
                                    .build();
    try {
      replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
                    MapRProgramId.CldbProgramId.getNumber(),
                    CLDBProto.CLDBProg.ContainerSwitchMasterProc.getNumber(),
                    req,
                    ContainerSwitchMasterResponse.class);

      if (replyData == null) {
        // most likely can not connect to CLDB
        LOG.error("Couldn't connect to the CLDB service");
        out.addError(new OutputError(Errno.ERPCFAILED, "cldb rpc failed"));
        return output;
      }

      ContainerSwitchMasterResponse resp;
      resp = ContainerSwitchMasterResponse.parseFrom(replyData);

      if (resp.getStatus() == 0) {
        done = true;
      } else {
        out.addError(new OutputError(resp.getStatus(),
                     "switchmaster failed, Error : " +
                     Errno.toString(resp.getStatus())));
      }
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      LOG.error("Container switchmaster failed");
      out.addError(new OutputError(Errno.ERPCFAILED, "cldb rpc failed"));
      return output;
    }
    if (done)
      return new TextCommandOutput(("Switched master").getBytes());
    else
      return output;
  }

  private CommandOutput switchTailWithUpstream() throws CLIProcessingException {
    OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    int cid = getParamIntValue(VOLUME_CID_PARAM_NAME, 0);

    // lookup container in cldb
    ContainerLookupRequest containerLookupReq = ContainerLookupRequest.newBuilder()
        .addContainerId(cid)
        .setCreds(getUserCredentials())
        .build();
   
    byte[] replyData;
    try {
      replyData = CLDBRpcCommonUtils.getInstance().sendRequest(MapRProgramId.CldbProgramId.getNumber(),
                    CLDBProto.CLDBProg.ContainerLookupProc.getNumber(), containerLookupReq, ContainerLookupResponse.class);

      if (replyData == null) {
        LOG.error("Couldn't connect to the CLDB service");
        out.addError(new OutputError(Errno.ERPCFAILED, "cldb rpc failed"));
        return output;
      }

      ContainerLookupResponse containerLookupResp = ContainerLookupResponse.parseFrom(replyData);
      if (containerLookupResp.getStatus() != 0) {
        out.addError(new OutputError(containerLookupResp.getStatus(), "ContainerLookup failed, Err: "
            + Errno.toString(containerLookupResp.getStatus())));
        return output;
      }
      if (containerLookupResp.getContainers(0).getType() != ContainerReplType.CASCADE) {
        out.addError(new OutputError(Errno.EINVAL, "Container replica type not supported."));
        return output;
      }
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException("MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      LOG.error("Container lookup failed for id: " + cid + ", error: " + e);
      out.addError(new OutputError(Errno.ERPCFAILED, "cldb rpc failed"));
      return output;
    }
    ContainerSwitchMasterRequest containerSwitchReq = ContainerSwitchMasterRequest.newBuilder()
        .setCid(cid)
        .setCreds(getUserCredentials())
        .build();
    try {
      replyData = CLDBRpcCommonUtils.getInstance().sendRequest(MapRProgramId.CldbProgramId.getNumber(),
                      CLDBProto.CLDBProg.ContainerSwitchTailWithUpstream.getNumber(), containerSwitchReq,
                      ContainerSwitchMasterResponse.class);

      if (replyData == null) {
        LOG.error("Couldn't connect to the CLDB service");
        out.addError(new OutputError(Errno.ERPCFAILED, "cldb rpc failed"));
        return output;
      }

      ContainerSwitchMasterResponse containerSwitchResp = ContainerSwitchMasterResponse.parseFrom(replyData);
      if (containerSwitchResp.getStatus() == 0) {
        return new TextCommandOutput(("switched tail replica to upstream.").getBytes());
      } else {
        out.addError(new OutputError(containerSwitchResp.getStatus(), "Container SwitchTailWithUpstream failed, Error : " +
            Errno.toString(containerSwitchResp.getStatus())));
        return output;
      }
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException("MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      LOG.error("Container switchmaster failed");
      out.addError(new OutputError(Errno.ERPCFAILED, "cldb rpc failed"));
      return output;
    }
  }
}
