/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.cli;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.regex.Pattern;

import org.apache.log4j.Logger;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.MessageLite;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.baseutils.fsrpcutils.Utils;
import com.mapr.baseutils.Errno;
import com.mapr.cli.common.ListCommand;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CLIUsageOnlyCommand;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.ProcessedInput.Parameter;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.IntegerInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.cliframework.util.FieldInfo;
import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.cldb.proto.CLDBProto.AddVirtualIpRequest;
import com.mapr.fs.cldb.proto.CLDBProto.AddVirtualIpResponse;
import com.mapr.fs.cldb.proto.CLDBProto.ListVirtualIpRequest;
import com.mapr.fs.cldb.proto.CLDBProto.ListVirtualIpResponse;
import com.mapr.fs.cldb.proto.CLDBProto.MoveVirtualIpRequest;
import com.mapr.fs.cldb.proto.CLDBProto.MoveVirtualIpResponse;
import com.mapr.fs.cldb.proto.CLDBProto.RemoveVirtualIpRequest;
import com.mapr.fs.cldb.proto.CLDBProto.RemoveVirtualIpResponse;
import com.mapr.fs.cldb.proto.CLDBProto.VirtualIPInfo;
import com.mapr.fs.cldb.util.Util;
import com.mapr.fs.cli.proto.CLIProto.Limiter;
import com.mapr.fs.proto.Common;
import com.mapr.fs.proto.Common.InterfaceInfo;
import com.mapr.security.MaprSecurityException;

public class VirtualIPCommands extends ListCommand
{
  private static final Logger LOG = Logger.getLogger(VirtualIPCommands.class);

  private static final int NUM_VIRTUAL_IPS_PER_RPC = 50;
  
  private static final String Macs       = "macs";
  private static final String NetMask    = "netmask";
  private static final String Gateway    = "gateway";
  private static final String VirtualIP    = "virtualip";
  private static final String VirtualIPEnd = "virtualipend";
  private static final String toMac      = "tomac";
  private static final String preferredMac      = "preferredmac";
  
  private static final String Range     = "range";
  private static final String NfsMacs   = "nfsmacs";

  private static final String Limit    = "limit";
  private static final String Start     = "start";
  private static final String Columns = "columns";
  private static final String Output   = "output";
  private static final String Filter    = "filter";

  private static Pattern s_spaceSplitter = Pattern.compile("\\s+");

  public static String longToIp(long i) {
    return ((i >> 24 ) & 0xFF) + "." +
           ((i >> 16 ) & 0xFF) + "." +
           ((i >>  8 ) & 0xFF) + "." +
           ( i        & 0xFF);
  }

  public static Map<CLDBProto.NodeInfo, FieldInfo> fieldTable =
    new ImmutableMap.Builder<CLDBProto.NodeInfo, FieldInfo>()
    .put (CLDBProto.NodeInfo.Ip, new FieldInfo (
        CLDBProto.NodeInfo.Ip.getNumber(), "ip", "ip", String.class))
    .put (CLDBProto.NodeInfo.Hostname, new FieldInfo(
        CLDBProto.NodeInfo.Hostname.getNumber(), "hn", "hn", String.class))
    .put (CLDBProto.NodeInfo.MacAddress, new FieldInfo(
        CLDBProto.NodeInfo.MacAddress.getNumber(), "mac", "mac", String.class))
    .put (CLDBProto.NodeInfo.VirtualIp, new FieldInfo(
        CLDBProto.NodeInfo.VirtualIp.getNumber(), "vip", "vip", String.class))
    .put (CLDBProto.NodeInfo.VirtualIpEnd, new FieldInfo(
        CLDBProto.NodeInfo.VirtualIpEnd.getNumber(), "vipe", "vipe", String.class))
    .build();

  public static Map<String, BaseInputParameter> baseParams =
    new ImmutableMap.Builder<String, BaseInputParameter>()
    .put(MapRCliUtil.CLUSTER_NAME_PARAM,
      new TextInputParameter(MapRCliUtil.CLUSTER_NAME_PARAM,
          "cluster name",
          CLIBaseClass.NOT_REQUIRED,
          null))
         .build();

  private static final CLICommand listCommand =
    new CLICommand (
        "list",
        "usage: virtualip list -type [assignment|range|nfsmacs]" +
          "-cluster clustername",
        VirtualIPCommands.class,
        ExecutionTypeEnum.NATIVE,
        new ImmutableMap.Builder<String, BaseInputParameter>().
          putAll(baseParams)
          .put (Range, new IntegerInputParameter(
                  Range, "range", CLIBaseClass.NOT_REQUIRED, 0))
          .put (NfsMacs, new IntegerInputParameter(
                  NfsMacs, "nfsmacs", CLIBaseClass.NOT_REQUIRED, 0))
          .put (Limit, new IntegerInputParameter(
                  Limit, "limit", CLIBaseClass.NOT_REQUIRED, Integer.MAX_VALUE))
          .put (Start, new IntegerInputParameter(
                  Start, "start", CLIBaseClass.NOT_REQUIRED, 0))
          .put (Columns, new TextInputParameter(
                  Columns, "columns", CLIBaseClass.NOT_REQUIRED, "none"))
          .put (Output, new TextInputParameter(
                  Output, "output", CLIBaseClass.NOT_REQUIRED, "terse"))
          .put (Filter, new TextInputParameter(
                  Filter, "filter",CLIBaseClass.NOT_REQUIRED, "none"))
        .build(),
        null)
    .setShortUsage("virtualip list -range range -nfsmacs nfsmacs" +
            " -cluster clustername");

  private static final CLICommand addCommand =
    new CLICommand (
        "add",
          "usage: virtualip add -virtualip vIP -virtualipend VirtualIpRangeEnd -netmask netmask " +
            "-macs macaddress1 macaddress2 -preferredmac preferredmac .. -cluster clustername",
        VirtualIPCommands.class,
        ExecutionTypeEnum.NATIVE,
        new ImmutableMap.Builder<String, BaseInputParameter>()
          .putAll(baseParams)
          .put (Macs, new TextInputParameter(
                  Macs, "macaddress",
                    CLIBaseClass.NOT_REQUIRED, null))
          .put  (NetMask, new TextInputParameter(
                  NetMask, "netmask",
                     CLIBaseClass.REQUIRED, null))
          .put  (Gateway, new TextInputParameter(
            Gateway, "gateway",
               CLIBaseClass.NOT_REQUIRED, null))
          .put (VirtualIP, new TextInputParameter(
                  VirtualIP, "virtualip",
                    CLIBaseClass.REQUIRED, null))
          .put (VirtualIPEnd, new TextInputParameter(
                  VirtualIPEnd, "VirtualIpRangeEnd",
                    CLIBaseClass.NOT_REQUIRED, null))
          .put (preferredMac, new TextInputParameter(
                  preferredMac, "preferredmac",
                    CLIBaseClass.NOT_REQUIRED, null))
                    
        .build(),
        null)
    .setShortUsage("virtualip add -virtualip vIP -virtualipend VirtualIpRangeEnd -netmask netmask " +
          "-macs macaddress1 -preferredmac macaddress.. -cluster clustername");

  private static final CLICommand editCommand =
      new CLICommand (
          "edit",
          "usage: virtualip edit -virtualip vIP -virtualipend VirtualIpRangeEnd -cluster clustername",
          VirtualIPCommands.class,
          ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
            .putAll(baseParams)
            .put (VirtualIP, new TextInputParameter(
                    VirtualIP, "virtualip",
                      CLIBaseClass.REQUIRED, null))
            .put (VirtualIPEnd, new TextInputParameter(
                    VirtualIPEnd, "VirtualIpRangeEnd",
                      CLIBaseClass.NOT_REQUIRED, null))
            .put  (NetMask, new TextInputParameter(
                  NetMask, "netmask",
                     CLIBaseClass.REQUIRED, null))
            .put (Macs, new TextInputParameter(
                    Macs, "macaddress",
                      CLIBaseClass.NOT_REQUIRED, null))
            .put (preferredMac, new TextInputParameter(
                  preferredMac, "preferredmac",
                    CLIBaseClass.NOT_REQUIRED, null))
         
          .build(),
          null)
      .setShortUsage ("virtualip edit -virtualip vIP -virtualipend VirtualIpRangeEnd -cluster clustername"
              + " -macs macaddress1 -preferredmac macaddress..");

  private static final CLICommand moveCommand =
		    new CLICommand (
		        "move",
		        "usage: virtualip move -virtualip vIP -virtualipend VirtualIpRangeEnd -cluster clustername",
		        VirtualIPCommands.class,
		        ExecutionTypeEnum.NATIVE,
		        new ImmutableMap.Builder<String, BaseInputParameter>()
		          .putAll(baseParams)
		          .put (VirtualIP, new TextInputParameter(
		                  VirtualIP, "virtualip",
		                    CLIBaseClass.REQUIRED, null))
		          .put (VirtualIPEnd, new TextInputParameter(
		                  VirtualIPEnd, "VirtualIpRangeEnd",
		                    CLIBaseClass.NOT_REQUIRED, null))
		          .put (toMac, new TextInputParameter(
                    toMac, "tomac",
                      CLIBaseClass.REQUIRED, null))
		        .build(),
		        null)
		    .setShortUsage ("virtualip move -virtualip vIP -virtualipend VirtualIpRangeEnd -cluster clustername");

  private static final CLICommand removeCommand =
    new CLICommand (
        "remove",
        "usage: virtualip remove -virtualip vIP -virtualipend VirtualIpRangeEnd -cluster clustername",
        VirtualIPCommands.class,
        ExecutionTypeEnum.NATIVE,
        new ImmutableMap.Builder<String, BaseInputParameter>()
          .putAll(baseParams)
          .put (VirtualIP, new TextInputParameter(
                  VirtualIP, "virtualip",
                    CLIBaseClass.REQUIRED, null))
          .put (VirtualIPEnd, new TextInputParameter(
                  VirtualIPEnd, "VirtualIpRangeEnd",
                    CLIBaseClass.NOT_REQUIRED, null))
        .build(),
        null)
    .setShortUsage ("virtualip remove -virtualip vIP -virtualipend VirtualIpRangeEnd -cluster clustername");

  // main command
  public static final CLICommand virtualIpsCommand =
    new CLICommand (
         "virtualip", "virtualip [add|remove|edit|move|list]",
         CLIUsageOnlyCommand.class,
         ExecutionTypeEnum.NATIVE,
         new CLICommand[] { addCommand, removeCommand, editCommand, moveCommand, listCommand }
       )
    .setShortUsage("virtualip [add|remove|edit|move|list]");

  public VirtualIPCommands (ProcessedInput input, CLICommand cliCommand)
  {
    super(input, cliCommand);
  }

  public CommandOutput executeRealCommand() throws CLIProcessingException
  {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    output.setOutput(out);

    if (!super.validateInput()) {
      return output;
    }


    if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
      String cluster = getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0);
      if (!CLDBRpcCommonUtils.getInstance().isValidClusterName(cluster)) {
        out.addError(new OutputError(Errno.EUCLUSTER, "Invalid cluster: " + cluster));
        return output;
      }
    }

    if (cliCommand.getCommandName().equalsIgnoreCase("add")) {
      addVirtualIps(out);
    } else if (cliCommand.getCommandName().equalsIgnoreCase("list")) {
      list(out);
    } else if (cliCommand.getCommandName().equalsIgnoreCase("remove")) {
      removeVirtualIps(out);
    } else if (cliCommand.getCommandName().equalsIgnoreCase("move")) {
      moveVirtualIps(out);
    } else if (cliCommand.getCommandName().equalsIgnoreCase("edit")) {
      editVirtualIps(out);
    }

    return output;
  }

  // netmask:
  // binary number containing '1' bits followed by '0' bits. 1 cannot follow a 0
  private boolean isNetmaskValid (int netmask)
  {
    // can't have a netmask of all 0s or with the lsb=1
    // 255.255.255.255 is a valid netmask
    boolean valid = netmask != 0 && 
      (netmask == 0xffffffff || (netmask & 0x1) == 0);

    /**
      * slower way
    if (valid) {
      while ((netmask & 0x80000000) != 0x0)
        netmask <<= 1;

      // cant have any 1 bits from here on..
      return (netmask == 0) ? true : false;
    }
     */

    // faster way:
    if (valid) {
      long x = ~netmask + 1;

      // should be a power of 2 now.
      valid = (x > 0) && ((x & (x - 1)) == 0);
    }    
    return valid;
  }
  
  // Verify that the string is of type 
  // aa:bb:cc:dd:ee:ff
  // Infiniband can be of 20 octets.
  private boolean isValidMacAddress(String mac) {
    if ((mac.length() != 17) && (mac.length() != 59)) {
      return false;
    }
    for (int i=2; i < mac.length(); i+=3) {
      if (mac.charAt(i) != ':') {
        return false;
      }
    }
    return true;
  }
  
  private void addVirtualIps (OutputHierarchy out)
      throws CLIProcessingException
  {
    String ip = getParamTextValue(VirtualIP, 0);
    String nMask = getParamTextValue(NetMask, 0);
    long netMask = Util.ipToLong(nMask);
    if (! isNetmaskValid((int)netMask)) {
      /**
      * <MAPR_ERROR>
      * Message:Netmask invalid: <netmask>
      * Function:VirtualIPCommands.addVirtualIps()
      * Meaning:An incorrect netmask was specified.
      * Resolution:Check the command syntax and try again.
      * </MAPR_ERROR>
      */
      out.addError (new OutputError(Errno.EINVAL, "Netmask invalid: " + nMask));
      LOG.error("Netmask invalid: " + nMask);
      return;
    }

    Parameter param = input.getParameterByName (Macs);
    List<String> values = param != null ?
          param.getParamValues():
              new ArrayList<String>(0);

    VirtualIPInfo.Builder req = VirtualIPInfo.newBuilder();
    InterfaceInfo.Builder ifb = InterfaceInfo.newBuilder();

    // fist set the vIpInfo
    long vIpStart = Util.ipToLong(ip);
    ifb.setNetmask(netMask);
    req.setNetmask(netMask);

    if (isParamPresent(Gateway)) {
      String gateway = getParamTextValue(Gateway, 0);
      ifb.setGateway(Util.ipToLong(gateway));
      req.setGateway (Util.ipToLong(gateway));
    }

    // Now add in the allowed MacAddresses
    for (String macaddr: values) {
      // The UI -> maprcli comes in as a single space separated string.
      String[] addrs = s_spaceSplitter.split((macaddr));
      for (String addr: addrs) {
        InterfaceInfo.Builder mb = InterfaceInfo.newBuilder();
        mb.setMacaddress(addr.toLowerCase());
        req.addDevInfo(mb.build()); // add each Mac
      }
    }
    
    if (isParamPresent(preferredMac)) {
      String mac = getParamTextValue(preferredMac, 0);
      
      if (isValidMacAddress(mac) == false) {        
        out.addError (new
          OutputError(Errno.EINVAL, "preferred mac address should of format xx:xx:xx:xx:xx:xx or 20 octet address"));
        return;        
      }
      
      InterfaceInfo.Builder mb = InterfaceInfo.newBuilder();
      mb.setMacaddress(mac.toLowerCase());
      req.addPreferredDevInfo(mb.build()); // add the Mac      
    }

    long vIpEnd = vIpStart;
    String vIpRangeEnd = "";
    if (isParamPresent(VirtualIPEnd)) {
      vIpRangeEnd = getParamTextValue(VirtualIPEnd, 0);
      vIpRangeEnd = vIpRangeEnd.trim();

      if (vIpRangeEnd.length() > 0) {
        if (!checkIPRange (ip, vIpRangeEnd)) {
          /**
          * <MAPR_ERROR>
          * Message:RangeEnd must be greater than the beginning of the range.
          * Function:VirtualIPCommands.addVirtualIps()
          * Meaning:When specifying a VIP range (more than one VIP), you must
          * specify an end IP address greater than the starting IP address for the range.
          * Resolution:Check the specified VIP range and try again.
          * </MAPR_ERROR>
          */
          out.addError (new
            OutputError(Errno.EINVAL, "RangeEnd must be greater than the beginning of the range."));
          return;
        }

        vIpEnd = Util.ipToLong(vIpRangeEnd);
      }
    }

    req.setVIpStart(vIpStart);
    req.setVIpEnd(vIpEnd);
    req.setVIpInfo(ifb.build());

    AddVirtualIpResponse resp;
    AddVirtualIpRequest.Builder aBuilder = AddVirtualIpRequest.newBuilder()
                                                              .setCreds(getUserCredentials());
    aBuilder.setVIpConfig(req.build());

    try {
      byte[] data;
      if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
    	  data = CLDBRpcCommonUtils.getInstance().sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
    			  Common.MapRProgramId.CldbProgramId.getNumber(),
    			  CLDBProto.CLDBProg.AddVirtualIpProc.getNumber(),
                  aBuilder.build(), AddVirtualIpResponse.class);
      } else {
    	  data = CLDBRpcCommonUtils.getInstance().sendRequest(
    			  Common.MapRProgramId.CldbProgramId.getNumber(),
    			  CLDBProto.CLDBProg.AddVirtualIpProc.getNumber(),
                  aBuilder.build(), AddVirtualIpResponse.class);
      }

      if (data == null) {
        /**
        * <MAPR_ERROR>
        * Message:Exception while processing RPC
        * Function:VirtualIPCommands.addVirtualIps()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
    	  out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
    	  return;
      }

      // success
      resp = AddVirtualIpResponse.parseFrom(data);
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
        "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      /**
      * <MAPR_ERROR>
      * Message:Add VirtualIP: <error>
      * Function:VirtualIPCommands.addVirtualIps()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      out.addError (new OutputError(Errno.EOPFAILED, "Add VirtualIP: " +
            Errno.toString(Errno.EOPFAILED))
          );
      /**
      * <MAPR_ERROR>
      * Message:Exception during Add VirtualIP <error>
      * Function:VirtualIPCommands.addVirtualIps()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      LOG.error("Exception during Add VirtualIP ", e);
      return;
    }

    int status = resp.getStatus();
    if (status != Errno.SUCCESS) {
      if (status == Errno.EEXIST) {
        /**
        * <MAPR_ERROR>
        * Message:Virtual IP <IP address> is already added to the cluster. Please specify a different IP address.
        * Function:VirtualIPCommands.addVirtualIps()
        * Meaning:When creating a VIP or VIP range, you must specify virtual IP addresses that are not currently in use on the cluster.
        * Resolution:Choose a different VIP or VIP range and try again.
        * </MAPR_ERROR>
        */
        String msg = "Virtual IP " + ip + " is already added to the cluster. Please specify a different IP address.";
        if (!vIpRangeEnd.isEmpty() && !ip.equals(vIpRangeEnd)) {
          /**
          * <MAPR_ERROR>
          * Message:One or more Virtual IPs in the range <IP address> to <IP address> are already added to the cluster.  Please specify a different IP address range.
          * Function:VirtualIPCommands.addVirtualIps()
          * Meaning:When creating a VIP or VIP range, you must specify virtual IP addresses that are not currently in use on the cluster.
          * Resolution:Choose a different VIP or VIP range and try again.
          * </MAPR_ERROR>
          */
          msg = "One or more Virtual IPs in the range " + ip 
          + " to " + vIpRangeEnd + " are already added to the cluster.  Please specify a different IP address range.";
        }
        out.addError(new OutputError(status, msg));
      } else {
        out.addError(new OutputError(status, resp.hasErrMsg() ? resp.getErrMsg() : Errno.toString(status)));
      }
      return;
    }
    
    // Successfully completed the command -- check if the CLDB processed 
    // preferrredmac option or not. If not then add a message to the 
    // output.
    if (isParamPresent(preferredMac) && !resp.hasPreferredDevSupported()) {
      out.addMessage("Successfully added VIPs but could not set the preferredmac "
                     + "as CLDB is of older version. Upgrade CLDB to latest version and "
                     + "update the preferredmac using \"maprcli virtualip edit \" command");
    }
  }

  private void moveVirtualIps (OutputHierarchy out)
      throws CLIProcessingException
  {
    String vipStartStr = getParamTextValue(VirtualIP, 0);
    String mac = getParamTextValue(toMac, 0);
    String vipEndStr = vipStartStr;
    if (isParamPresent(VirtualIPEnd)) {
      vipEndStr = getParamTextValue(VirtualIPEnd, 0);
    }      

    if (! checkIPRange (vipStartStr, vipEndStr)) {
      /**
       * <MAPR_ERROR>
       * Message:RangeEnd must be greater than the beginning of the range.
       * Function:VirtualIPCommands.moveVirtualIps()
       * Meaning:When specifying a VIP range (more than one VIP), you must
       * specify an end IP address greater than the starting IP address for the range.
       * Resolution:Check the specified VIP range and try again.
       * </MAPR_ERROR>
       */
      out.addError (new
          OutputError(Errno.EINVAL, "RangeEnd must be greater than the beginning of the range.")
          );
      return;
    }
    
    if (isValidMacAddress(mac) == false) {        
      out.addError (new
        OutputError(Errno.EINVAL, "mac address should of format xx:xx:xx:xx:xx:xx or 20 octet address"));
      return;        
    }
    long vIpStart = Util.ipToLong(vipStartStr);
    long vIpEnd = Util.ipToLong(vipEndStr);

    MoveVirtualIpRequest.Builder req = MoveVirtualIpRequest.newBuilder();
    req.setCreds(getUserCredentials());
    req.setVIpStart(vIpStart);
    req.setVIpEnd(vIpEnd);
    InterfaceInfo.Builder mb = InterfaceInfo.newBuilder();
    mb.setMacaddress(mac.toLowerCase());
    req.setPreferredDevInfo(mb.build());       

    MoveVirtualIpResponse resp;
    
    try {
      byte[] data;
      if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        data = CLDBRpcCommonUtils.getInstance().sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProto.CLDBProg.MoveVirtualIpProc.getNumber(),
                  req.build(), MoveVirtualIpResponse.class);
      } else {
        data = CLDBRpcCommonUtils.getInstance().sendRequest(
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProto.CLDBProg.MoveVirtualIpProc.getNumber(),
                  req.build(), MoveVirtualIpResponse.class);
      }

      if (data == null) {
        /**
        * <MAPR_ERROR>
        * Message:Exception while processing RPC
        * Function:VirtualIPCommands.editVirtualIps()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
        out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
        return;
      }

      // success
      resp = MoveVirtualIpResponse.parseFrom(data);
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
        "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      /**
      * <MAPR_ERROR>
      * Message:Edit VirtualIP: <error>
      * Function:VirtualIPCommands.editVirtualIps()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      out.addError (new OutputError(Errno.EOPFAILED, "Move VirtualIP: " +
            Errno.toString(Errno.EOPFAILED))
          );
      /**
      * <MAPR_ERROR>
      * Message:Exception during Edit VirtualIP <error>
      * Function:VirtualIPCommands.editVirtualIps()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      LOG.error("Exception during Move VirtualIP ", e);
      return;
    }

    int status = resp.getStatus();
    if (status != 0) {
      out.addError(new OutputError(status, resp.hasErrMsg() ? resp.getErrMsg() : Errno.toString(status)));
    }
  }

  private void editVirtualIps (OutputHierarchy out)
      throws CLIProcessingException
  {
    String ip = getParamTextValue(VirtualIP, 0);

    Parameter param = input.getParameterByName (Macs);
    List<String> values = param != null ?
          param.getParamValues():
              new ArrayList<String>(0);

    VirtualIPInfo.Builder req = VirtualIPInfo.newBuilder();

    // fist set the vIpInfo
    long vIpStart = Util.ipToLong(ip);
    if (isParamPresent(Gateway)) {
      String gateway = getParamTextValue(Gateway, 0);
      req.setGateway (Util.ipToLong(gateway));
    }

    String nm = getParamTextValue(NetMask, 0);
    long netmask = Util.ipToLong(nm);
    req.setNetmask(netmask);

    InterfaceInfo.Builder ifb = InterfaceInfo.newBuilder();
    ifb.setNetmask(netmask);
    req.setVIpInfo(ifb.build());

    // Now add in the allowed MacAddresses
    for (String macaddr: values) {
      // The UI -> maprcli comes in as a single space separated string.
      String[] addrs = s_spaceSplitter.split((macaddr));
      for (String addr: addrs) {
        InterfaceInfo.Builder mb = InterfaceInfo.newBuilder();
        mb.setMacaddress(addr.toLowerCase());
        req.addDevInfo(mb.build()); // add each Mac
      }
    }
    
    if (isParamPresent(preferredMac)) {
      
      String mac = getParamTextValue(preferredMac, 0);
      if ((mac.compareToIgnoreCase("") != 0) && 
          (isValidMacAddress(mac) == false)) {        
        out.addError (new
          OutputError(Errno.EINVAL, "preferred mac address should of format "
                                  + "xx:xx:xx:xx:xx:xx or \"\" to remove "
                                  + "the preferred mac"));
        return;        
      }
      
      InterfaceInfo.Builder mb = InterfaceInfo.newBuilder();
      mb.setMacaddress(mac.toLowerCase());
      req.addPreferredDevInfo(mb.build()); // add the Mac      
    }


    long vIpEnd = vIpStart;
    if (isParamPresent(VirtualIPEnd)) {
      String vIpRangeEnd = getParamTextValue(VirtualIPEnd, 0);

      if (! checkIPRange (ip, vIpRangeEnd)) {
        /**
        * <MAPR_ERROR>
        * Message:RangeEnd must be greater than the beginning of the range.
        * Function:VirtualIPCommands.editVirtualIps()
        * Meaning:When specifying a VIP range (more than one VIP), you must
        * specify an end IP address greater than the starting IP address for the range.
        * Resolution:Check the specified VIP range and try again.
        * </MAPR_ERROR>
        */
        out.addError (new
            OutputError(Errno.EINVAL, "RangeEnd must be greater than the beginning of the range.")
          );
        return;
      }
      vIpEnd = Util.ipToLong(vIpRangeEnd);
    }

    req.setVIpStart(vIpStart);
    req.setVIpEnd(vIpEnd);

    AddVirtualIpResponse resp;
    AddVirtualIpRequest.Builder aBuilder = AddVirtualIpRequest.newBuilder()
                                                              .setCreds(getUserCredentials());
    aBuilder.setVIpConfig(req.build());
    aBuilder.setModify (true);

    try {
      byte[] data;
      if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
    	  data = CLDBRpcCommonUtils.getInstance().sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
    			  Common.MapRProgramId.CldbProgramId.getNumber(),
    			  CLDBProto.CLDBProg.ModifyVirtualIpProc.getNumber(),
                  aBuilder.build(), AddVirtualIpResponse.class);
      } else {
    	  data = CLDBRpcCommonUtils.getInstance().sendRequest(
    			  Common.MapRProgramId.CldbProgramId.getNumber(),
    			  CLDBProto.CLDBProg.ModifyVirtualIpProc.getNumber(),
                  aBuilder.build(), AddVirtualIpResponse.class);
      }

      if (data == null) {
        /**
        * <MAPR_ERROR>
        * Message:Exception while processing RPC
        * Function:VirtualIPCommands.editVirtualIps()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
    	  out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
    	  return;
      }

      // success
      resp = AddVirtualIpResponse.parseFrom(data);
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
        "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      /**
      * <MAPR_ERROR>
      * Message:Edit VirtualIP: <error>
      * Function:VirtualIPCommands.editVirtualIps()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      out.addError (new OutputError(Errno.EOPFAILED, "Edit VirtualIP: " +
            Errno.toString(Errno.EOPFAILED))
          );
      /**
      * <MAPR_ERROR>
      * Message:Exception during Edit VirtualIP <error>
      * Function:VirtualIPCommands.editVirtualIps()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      LOG.error("Exception during Edit VirtualIP ", e);
      return;
    }

    int status = resp.getStatus();
    if (status != 0) {
      out.addError(new OutputError(status, resp.hasErrMsg() ? resp.getErrMsg() : Errno.toString(status)));
      return;
    }
    
    // Successfully completed the command -- check if the CLDB processed 
    // preferrredmac option or not. If not then add a message to the 
    // output.
    if (isParamPresent(preferredMac) && !resp.hasPreferredDevSupported()) {
      out.addMessage("Successfully edited VIPs but could not update the preferredmac "
                     + "as CLDB is of older version. Upgrade CLDB to latest version "
                     + "and try again");
                     
    }

  }

  private boolean checkIPRange (String ip, String ipEnd)
  {
    // Split ip ranges
    String[] ip1 = ip.split("\\.");
    String[] ip2 = ipEnd.split("\\.");
    if (ip1.length != ip2.length && ip1.length != 4) {
      // Ip octal lengths do not match and do not have 4 octals
      return false;
    }

    try {
      for (int i = 0; i < ip1.length; i++) {
        // Check if octal of ip start is greater than end. If so, then return false
        int i1 = Integer.parseInt(ip1[i]);
        int i2 = Integer.parseInt(ip2[i]);
        if (i1 < i2) {
          // If start ip octal less than end octal, then startIp < endIp
          return true;
        } else if (i1 > i2) {
          // If start ip octal greater than end octal, then startIp > endIp
          return false;
        }
      }
    } catch (NumberFormatException e) {
      // Ips are not formatted properly
      LOG.error("Ip's are not in proper format. Ip Start: " + ip + " Ip End: " + ipEnd);
      return false;
    }
    // Loop ended, means IPs are equal
    // IP start and end can be same number
    return true;
  }
  
  private static int formatStatus(String s) {
    if (s.equalsIgnoreCase("ACTIVE")) {
      // Healthy - Green
      return 0;
    }
    if (s.equalsIgnoreCase("DEAD") || s.equalsIgnoreCase("INACTIVE")) {
      // Critical - Red
      return 4;
    }
    return 5;
  }

  private void listNfsServers (List<VirtualIPInfo> list, OutputHierarchy out) {
    for (VirtualIPInfo vInfo : list) {
      int status = formatStatus(vInfo.getState());

      int nDevs = vInfo.getDevInfoCount();
      for (int j=0; j < nDevs; ++j) {
        OutputNode row = new OutputNode();
        out.addNode(row);

        OutputNode a = new OutputNode("h", status); // health
        row.addChild(a);

        InterfaceInfo ifInfo = vInfo.getDevInfo(j);
        a = new OutputNode("hn", ifInfo.getHostname());
        row.addChild(a);

        a = new OutputNode("ip", longToIp(ifInfo.getIp()));
        row.addChild(a);

        a = new OutputNode("mac", ifInfo.getMacaddress());
        row.addChild(a);
      }
    }
  }

  private void listAssigments (List<VirtualIPInfo> list, 
      OutputHierarchy out, boolean printAssignable) {
    for (VirtualIPInfo vInfo : list) {
      InterfaceInfo vIf = vInfo.getVIpInfo();
      OutputNode row = new OutputNode();

      OutputNode ip = new OutputNode("vip", longToIp(vIf.getIp()));
      row.addChild(ip);

      if (vInfo.hasAssignedDev()) {
        InterfaceInfo ifInfo = vInfo.getAssignedDev();
        OutputNode a = new OutputNode("hn", ifInfo.getHostname());
        row.addChild(a);

        a = new OutputNode("ip", longToIp(ifInfo.getIp()));
        row.addChild(a);

        a = new OutputNode("mac", ifInfo.getMacaddress());
        row.addChild(a);
      }

      if (! printAssignable) {
        out.addNode(row);
        continue;
      }

      // print the assignables at the end ..
      int num = vInfo.getDevInfoCount();
      StringBuilder sb = new StringBuilder();

      for (int j = 0; j < num; ++j) {
        InterfaceInfo d = vInfo.getDevInfo(j);

        if (sb.length() > 0) {
          sb.append(", ");
        }

        sb.append(d.getMacaddress());
      }

      if (sb.length() > 0) {
        OutputNode o = new OutputNode ("AssignableTo", sb.toString());
        row.addChild(o);
      }
      
      // print the preferred macs 
      if (vInfo.getPreferredDevInfoCount() > 0) {
        OutputNode o = new OutputNode ("PreferredMac", 
        		vInfo.getPreferredDevInfo(0).getMacaddress());
        row.addChild(o);
        o = new OutputNode ("Preferredip", 
        		Utils.longToIp(vInfo.getPreferredDevInfo(0).getIp()));
        row.addChild(o);
        o = new OutputNode ("Preferredhn", 
        		vInfo.getPreferredDevInfo(0).getHostname());
        row.addChild(o);
      }
      
      out.addNode(row);
    }
  }

  private void listConfig (List<VirtualIPInfo> list, OutputHierarchy out) {
    for (VirtualIPInfo vInfo : list) {

      OutputNode row = new OutputNode();
      out.addNode(row);

      OutputNode ip = new OutputNode("vip", longToIp(vInfo.getVIpStart()));
      row.addChild(ip);

      if (vInfo.hasVIpEnd()) {
        ip = new OutputNode("vipe", longToIp(vInfo.getVIpEnd()));
        row.addChild(ip);
      }

      if (vInfo.hasNetmask()) {
        ip = new OutputNode("nm", longToIp(vInfo.getNetmask()));
        row.addChild(ip);
      }

      if (vInfo.hasGateway()) {
        ip = new OutputNode("gw", longToIp(vInfo.getGateway()));
        row.addChild(ip);
      }

      int num = vInfo.getDevInfoCount();

      if (num == 0) {
        // TODO (smarella): There should be a better way to report an empty
        // JSON array than this.
        OutputNode wRow = new OutputNode("assignables");
        OutputNode subRow = new OutputNode("assignables");
        row.addChild(subRow);
        row.addChild(wRow);
      }
      for (int j = 0; j < num; ++j) {
        OutputNode subRow = new OutputNode("assignables");
        row.addChild(subRow);
        
        InterfaceInfo d = vInfo.getDevInfo(j);
        OutputNode o = new OutputNode("hn", d.getHostname());
        subRow.addChild(o);

        //String ipstr = d.hasIp() ? longToIp(d.getIp()): "";
        String ipstr = longToIp(d.getIp()); // print 0.0.0.0 if it doesn't exist
        o = new OutputNode("ip", ipstr);
        subRow.addChild(o);

        o = new OutputNode("mac", d.getMacaddress());
        subRow.addChild(o);

        if (num == 1) {
          // workaround for bug 1723: add an empty row ..
          OutputNode wRow = new OutputNode("assignables");
          row.addChild(wRow);
        }
      }  
    }
  }

  private void removeVirtualIps (OutputHierarchy out)
      throws CLIProcessingException
  {
    String start = getParamTextValue(VirtualIP, 0);
    long vIpStart = Util.ipToLong(start);

    RemoveVirtualIpResponse resp;
    RemoveVirtualIpRequest.Builder aBuilder = RemoveVirtualIpRequest.newBuilder()
                                                                    .setCreds(getUserCredentials());
    aBuilder.setVIpStart(vIpStart);

    long vIpEnd = vIpStart;
    if (isParamPresent(VirtualIPEnd)) {
      String end = getParamTextValue(VirtualIPEnd, 0);
      if (! checkIPRange (start, end)) {
        /**
        * <MAPR_ERROR>
        * Message:RangeEnd must be greater than the beginning of the range.
        * Function:VirtualIPCommands.removeVirtualIps()
        * Meaning:When specifying a VIP range (more than one VIP), you must
        * specify an end IP address greater than the starting IP address for the range.
        * Resolution:Check the specified VIP range and try again.
        * </MAPR_ERROR>
        */
        out.addError (new
            OutputError(Errno.EINVAL, "RangeEnd must be greater than the beginning of the range.")
          );
        return;
      }

      vIpEnd = Util.ipToLong(end);
    }
    aBuilder.setVIpEnd(vIpEnd);

    try {
    	
      byte[] data;
      if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
      	  data = CLDBRpcCommonUtils.getInstance().sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
      			  Common.MapRProgramId.CldbProgramId.getNumber(),
      			CLDBProto.CLDBProg.RemoveVirtualIpProc.getNumber(),
                aBuilder.build(), RemoveVirtualIpResponse.class);
        } else {
      	  data = CLDBRpcCommonUtils.getInstance().sendRequest(
      			  Common.MapRProgramId.CldbProgramId.getNumber(),
      			CLDBProto.CLDBProg.RemoveVirtualIpProc.getNumber(),
                aBuilder.build(), RemoveVirtualIpResponse.class);
        }

      if (data == null) {
        /**
        * <MAPR_ERROR>
        * Message:Exception while processing RPC
        * Function:VirtualIPCommands.removeVirtualIps()
        * Meaning:An error occurred.
        * Resolution:Contact technical support.
        * </MAPR_ERROR>
        */
    	  out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
    	  return;
      }

      // success
      resp = RemoveVirtualIpResponse.parseFrom(data);
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
        "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      /**
      * <MAPR_ERROR>
      * Message:Remove VirtualIP: <error>
      * Function:VirtualIPCommands.removeVirtualIps()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      out.addError (new OutputError(Errno.EOPFAILED, "Remove VirtualIP: " +
            Errno.toString(Errno.EOPFAILED))
          );
      /**
      * <MAPR_ERROR>
      * Message:Exception during Remove VirtualIP <error>
      * Function:VirtualIPCommands.removeVirtualIps()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      LOG.error("Exception during Remove VirtualIP ", e);
      return;
    }

    int status = resp.getStatus();
    if (status != 0) {
      out.addError(new OutputError(status, resp.hasErrMsg() ? resp.getErrMsg() : Errno.toString(status)));
    }
  }


  public String getCommandUsage()
  {
    return "virtualip add|edit|remove|move|list";
  }

  private ListVirtualIpRequest.ListType getListType() throws CLIProcessingException {
    ListVirtualIpRequest.ListType lType;
    int range = getParamIntValue(Range, 0);
    int nfsmacs = getParamIntValue(NfsMacs, 0);
    if (range != 0) {
      lType = ListVirtualIpRequest.ListType.RangeConfigured;
    } else if (nfsmacs != 0) {
      lType = ListVirtualIpRequest.ListType.NfsMacs;
    } else {
      lType = ListVirtualIpRequest.ListType.Assignments;
    }
    return lType;
  }
  
  private Limiter getLimiter() throws CLIProcessingException {
    return getNextLimiter(getParamIntValue(Start, 0), 0, 
        getParamIntValue(Start, 0), getParamIntValue(Limit, 0),
        NUM_VIRTUAL_IPS_PER_RPC);
    }
  
  private ListVirtualIpResponse getListVirtualIpResponse(byte[] data) throws CLIProcessingException {
    try {
      return ListVirtualIpResponse.parseFrom(data);
    } catch (InvalidProtocolBufferException ipbe) {
      /**
      * <MAPR_ERROR>
      * Message:Exception while parsing the RPC response data into ListVirtualIpResponse proto object.
      * Function:VirtualIPCommands.getListVirtualIpResponse()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      throw new CLIProcessingException("Exception while parsing the RPC " +
          "response data into ListVirtualIpResponse proto object.", ipbe);
    }
  }
  
  private int getRecordCount(ListVirtualIpResponse resp) throws CLIProcessingException {
    int count = resp.getVIpInfosCount();
    ListVirtualIpRequest.ListType lType = getListType();
    if (lType == ListVirtualIpRequest.ListType.NfsMacs) {
      count = 0;
      for (VirtualIPInfo info : resp.getVIpInfosList()) {
        count += info.getDevInfoCount();
      }
    }
    LOG.info("getRecordCount returned count: " + count);
    return count;
  }
  
  @Override
  public MessageLite buildNextRequest(MessageLite prevReq, MessageLite prevResp)
      throws CLIProcessingException {
    ListVirtualIpRequest.Builder newReqBuilder = null;
    if (prevReq != null) {
      newReqBuilder = ListVirtualIpRequest.newBuilder((ListVirtualIpRequest) prevReq);
    } else {
      newReqBuilder = ListVirtualIpRequest.newBuilder()
                                          .setLimiter(getLimiter())
                                          .setListType(getListType())
                                          .addAllFilter(getFilters(fieldTable, Filter))
                                          .setCreds(getUserCredentials());
    }
    
    if (prevResp != null) {
      int prevStart = newReqBuilder.getLimiter().getStart();
      int prevCount = getRecordCount((ListVirtualIpResponse) prevResp);
      int origStart = getParamIntValue(Start, 0);
      int origLimit = getParamIntValue(Limit, 0);
      newReqBuilder.setLimiter(getNextLimiter(prevStart, prevCount, 
          origStart, origLimit, NUM_VIRTUAL_IPS_PER_RPC));
    }
    
    return newReqBuilder.build();
  }

  @Override
  public boolean hasMore(MessageLite prevReq, MessageLite prevResp) throws CLIProcessingException {
    return hasMore(getParamIntValue(Start, 0), 
        getParamIntValue(Limit, 0), 
        ((ListVirtualIpRequest) prevReq).getLimiter().getStart(),
        getRecordCount((ListVirtualIpResponse) prevResp));
  }

  @Override
  public void processResponse(OutputHierarchy out, MessageLite response) throws CLIProcessingException {
    ListVirtualIpResponse resp = (ListVirtualIpResponse) response;
    ListVirtualIpRequest.ListType lType = getListType();
    if (lType == ListVirtualIpRequest.ListType.RangeConfigured) {
      listConfig(resp.getVIpInfosList(), out);
    } else if (lType == ListVirtualIpRequest.ListType.NfsMacs) {
      listNfsServers (resp.getVIpInfosList(), out);
    } else {
      listAssigments(resp.getVIpInfosList(), out, true);
    }
    if (resp.hasTotal()) {
      out.setTotal(resp.getTotal());
    } else  {
      /**
      * <MAPR_ERROR>
      * Message:total field not populated in ListVirtualIpResponse
      * Function:VirtualIPCommands.processResponse()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      LOG.error("total field not populated in ListVirtualIpResponse");
    }
  }

  @Override
  public MessageLite sendRequest(MessageLite request) throws CLIProcessingException {
    ListVirtualIpRequest req = (ListVirtualIpRequest) request;
    byte[] data = null;
    try {
      if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        data = CLDBRpcCommonUtils.getInstance().sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
            Common.MapRProgramId.CldbProgramId.getNumber(),
          CLDBProto.CLDBProg.ListVirtualIpProc.getNumber(),
              req, ListVirtualIpResponse.class);
      } else {
        data = CLDBRpcCommonUtils.getInstance().sendRequest(
            Common.MapRProgramId.CldbProgramId.getNumber(),
          CLDBProto.CLDBProg.ListVirtualIpProc.getNumber(),
              req, ListVirtualIpResponse.class);
      }
    } catch (Exception e) {
      throw new CLIProcessingException(e);
    }
    if (data != null) {
      return getListVirtualIpResponse(data);
    } else {
      /**
      * <MAPR_ERROR>
      * Message:RPC Request to list virtual ips failed. No data returned
      * Function:VirtualIPCommands.sendRequest()
      * Meaning:An error occurred.
      * Resolution:Contact technical support.
      * </MAPR_ERROR>
      */
      LOG.error("RPC Request to list virtual ips failed. No data returned");
      return null;
    }

  }
}
