/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.cli;

import java.util.Properties;

import org.apache.log4j.Logger;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.InvalidProtocolBufferException;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CLIUsageOnlyCommand;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.cldb.proto.CLDBProto.AeKey;
import com.mapr.fs.cldb.proto.CLDBProto.AeLookupRequest;
import com.mapr.fs.cldb.proto.CLDBProto.AeLookupResponse;
import com.mapr.fs.cldb.proto.CLDBProto.AeProperties;
import com.mapr.fs.proto.Common;
import com.mapr.fs.proto.License.EulaShowAcceptResponse;

public class UserConfigCommands extends CLIBaseClass implements CLIInterface {

    public static final String USER_PARAM_NAME = "username";
    public static final String SUPERUSER_PARAM = "mapr.fs.permissions.superuser";

    private static final Logger LOG = Logger.getLogger(UserConfigCommands.class);
    Properties CLDBProperties = new Properties();
    long binding = 0;
    
    public UserConfigCommands(ProcessedInput input, CLICommand cliCommand) {
        super(input, cliCommand);
    }
    
    static final CLICommand loadCmd = new CLICommand(
        "load",
        "",
        UserConfigCommands.class, 
        ExecutionTypeEnum.NATIVE,
        /* Add parameters in a hash map key is string,  value is BaseInputParameter */
        new ImmutableMap.Builder<String, BaseInputParameter>()
           /* parameter 1 */
           .put(UserConfigCommands.USER_PARAM_NAME,
                new TextInputParameter(
                     UserConfigCommands.USER_PARAM_NAME,
                     "username",
                     CLIBaseClass.REQUIRED,
                     ""))
           .put(MapRCliUtil.CLUSTER_NAME_PARAM,
               new TextInputParameter(MapRCliUtil.CLUSTER_NAME_PARAM,
          "cluster name",
          CLIBaseClass.NOT_REQUIRED,
          null))
           .build(),
            null        /* sub-commands */
    ).setShortUsage("userconfig load -username <username>");

    /* Define sub command */
    public static final CLICommand userconfigCmds = new CLICommand(
        "userconfig",
        "",
        CLIUsageOnlyCommand.class, 
        ExecutionTypeEnum.NATIVE,
        new CLICommand [] {         /* array of subcommands */
           loadCmd,
        }
    ).setUsageInVisible(true).setShortUsage("userconfig [load]");
        
    void fetchUserInfo (String user, OutputHierarchy out) 
        throws CLIProcessingException {
      OutputNode node = new OutputNode();
      node.addChild(new OutputNode("username", user));
      
      String cluster = null;
      if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        cluster = getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0);
        if (!CLDBRpcCommonUtils.getInstance().isValidClusterName(cluster)) {
          out.addError(new OutputError(Errno.EUCLUSTER, "Invalid cluster: " + cluster));
          return;
        }
      }

      MapRCliUtil.fetchCLDBParams(cluster, CLDBProperties, getUserCredentials());      
      int fsadmin = 0;
      String fsadminStr = CLDBProperties.getProperty(SUPERUSER_PARAM);
      if (fsadminStr != null && fsadminStr.equalsIgnoreCase(user)) {
        fsadmin = 1;
      }
      node.addChild(new OutputNode("fsadmin", fsadmin));
      
      // Fetch mradmin from JT
      node.addChild(new OutputNode("mradmin", 1));      
      
      // Fetch Eula information
      EulaShowAcceptResponse resp = EulaCommands.fetchEula(cluster);
      if ( resp == null ) {
      	out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
      	return;
      }
      int status = resp.getStatus();
      if (status != 0) {
        out.addError(new OutputError(status, Errno.toString(status)));       
      } else if (resp.hasAcceptedUser()) {
        node.addChild(new OutputNode("eulauser", resp.getAcceptedUser()));
        node.addChild(new OutputNode("eulatime", resp.getAcceptedTime()));
      }
      
      out.addNode(node);
    }
    
    /* This is a function that does the real work. */
    @Override
    public CommandOutput executeRealCommand() throws CLIProcessingException {
      OutputHierarchy out = new OutputHierarchy();  /* Root of comamnd output's hierarchy tree */
      CommandOutput output = new CommandOutput();
      output.setOutput(out);
        
      String cmd = cliCommand.getCommandName();
      if (cmd.equalsIgnoreCase("load")) {
        String user = getParamTextValue(USER_PARAM_NAME, 0);
        if (user != null && !user.isEmpty())
          fetchUserInfo(user, out);
      }
      return output;
    }
}
