/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.cli;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.TextFormat;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CLIUsageOnlyCommand;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.CommandOutput.*;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.*;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.BooleanInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.cldb.proto.CLDBProto.ClusterUpgradeRequest;
import com.mapr.fs.cldb.proto.CLDBProto.ClusterUpgradeResponse;
import com.mapr.fs.proto.Common;
import org.apache.log4j.Logger;
import com.mapr.security.MaprSecurityException;

import java.util.Map;

/**
 * Upgrade support
 */
public class UpgradeCommands extends CLIBaseClass
{
  private static final Logger LOG = Logger.getLogger(UpgradeCommands.class);
  private static final String TARGET_PARAM_NAME = "targetversion";
  private static final String NODE_PARAM_NAME = "node";
  
  public static Map<String, BaseInputParameter> baseParams =
    new ImmutableMap.Builder<String, BaseInputParameter>()
    .put(MapRCliUtil.CLUSTER_NAME_PARAM,
      new TextInputParameter(MapRCliUtil.CLUSTER_NAME_PARAM,
          "cluster name",
          CLIBaseClass.NOT_REQUIRED,
          null))
         .build();

  public static final CLICommand StartCmd =
    new CLICommand (
         "start", 
         "",
         UpgradeCommands.class,
         CLICommand.ExecutionTypeEnum.NATIVE,
         new ImmutableMap.Builder<String, BaseInputParameter>()
         .putAll(baseParams)
         .put(TARGET_PARAM_NAME, 
             new TextInputParameter(TARGET_PARAM_NAME,
                 "targetversion", 
                 CLIBaseClass.NOT_REQUIRED, 
                 null))
         .put(NODE_PARAM_NAME, 
             new TextInputParameter(NODE_PARAM_NAME,
                 "node", 
                 CLIBaseClass.NOT_REQUIRED, 
                 null))        
         .build(),
         null)
    .setShortUsage("notifyupgrade start -targetversion <version> [-cluster <cluster_name>]");
  
  public static final CLICommand FinishCmd =
    new CLICommand (
         "finish", 
         "",
         UpgradeCommands.class,
         CLICommand.ExecutionTypeEnum.NATIVE,
         new ImmutableMap.Builder<String, BaseInputParameter>()
         .putAll(baseParams)
         .put(NODE_PARAM_NAME, 
             new TextInputParameter(NODE_PARAM_NAME,
                 "node", 
                 CLIBaseClass.NOT_REQUIRED, 
                 null))        
         .build(),
         null)
    .setShortUsage("notifyupgrade finish [-cluster <cluster_name>]");
  
  public static final CLICommand CheckCmd =
    new CLICommand (
         "check", 
         "",
         UpgradeCommands.class,
         CLICommand.ExecutionTypeEnum.NATIVE,
         new ImmutableMap.Builder<String, BaseInputParameter>()
         .putAll(baseParams)
         .build(),
         null)
    .setShortUsage("notifyupgrade check [-cluster <cluster_name>]");
  
  // main command
  public static final CLICommand UpgradeCommands =
    new CLICommand (
         "notifyupgrade", 
         "",
         UpgradeCommands.class,
         CLICommand.ExecutionTypeEnum.NATIVE,
         new CLICommand[] {
             StartCmd, 
             FinishCmd,
             CheckCmd
           }
         )
    .setShortUsage("notifyupgrade start|finish|check")
    .setUsageInVisible(true);

  public UpgradeCommands (ProcessedInput input, CLICommand cliCommand)
  {
    super(input, cliCommand);
  }

  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException
  {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    output.setOutput(out);

    if (!super.validateInput()) {
      return output;
    }
    
    
    if (cliCommand.getCommandName().equalsIgnoreCase("start")) {
      ClusterUpgradeRequest.Builder req = ClusterUpgradeRequest.newBuilder()
                                                               .setCreds(getUserCredentials())
                                                               .setUpgradeInProgress(true);

      if (isParamPresent(NODE_PARAM_NAME)) {
        String node = getParamTextValue(NODE_PARAM_NAME, 0);
        req.setServer1(node);
      } else if (isParamPresent(TARGET_PARAM_NAME)) {
        String target = getParamTextValue(TARGET_PARAM_NAME, 0);
        req.setTargetVersion(target);
      }
      
      sendRpc(req.build(), out);
    } else if (cliCommand.getCommandName().equalsIgnoreCase("finish")) {
      ClusterUpgradeRequest.Builder req = ClusterUpgradeRequest.newBuilder()
                                                               .setCreds(getUserCredentials())
                                                               .setUpgradeInProgress(false);
      if (isParamPresent(NODE_PARAM_NAME)) {
        String node = getParamTextValue(NODE_PARAM_NAME, 0);
        req.setServer1(node);
      }
      sendRpc(req.build(), out);
    } else if (cliCommand.getCommandName().equalsIgnoreCase("check")) {
      ClusterUpgradeRequest req = ClusterUpgradeRequest.newBuilder()
                                                       .setCreds(getUserCredentials())
                                                       .setCheckProgress(false)
                                                       .build();
      ClusterUpgradeResponse resp = sendRpc(req, out);
      if (resp != null) {
        out.addNode(new OutputNode("InProgress", resp.getInProgress() ? 1 : 0));
      }
    }
    
    return output;
  }
  
  private ClusterUpgradeResponse sendRpc(ClusterUpgradeRequest req, OutputHierarchy out) throws CLIProcessingException {
    ClusterUpgradeResponse resp;
    byte[] data;

    try {
      if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        String cluster = getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0);
        if (!CLDBRpcCommonUtils.getInstance().isValidClusterName(cluster)) {
          out.addError(new OutputError(Errno.EUCLUSTER, "Invalid cluster: " + cluster));
          return null;
        }
        data = CLDBRpcCommonUtils.getInstance().sendRequest (
                    cluster,
                    Common.MapRProgramId.CldbProgramId.getNumber(),
                    CLDBProto.CLDBProg.ClusterUpgradeProc.getNumber(),
                    req, ClusterUpgradeResponse.class);
      } else {
        data = CLDBRpcCommonUtils.getInstance().sendRequest(
                    Common.MapRProgramId.CldbProgramId.getNumber(),
                    CLDBProto.CLDBProg.ClusterUpgradeProc.getNumber(),
                    req, ClusterUpgradeResponse.class);
      }

      if (data == null) {
        out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
        return null;
      }

      // success
      resp = ClusterUpgradeResponse.parseFrom(data);
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
        "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      out.addError (new CommandOutput.OutputHierarchy.OutputError (
            Errno.EOPFAILED, "notifyupgrade: " + Errno.toString(Errno.EOPFAILED)));
      LOG.error("Exception during ClusterUpgradeRequest", e);
      return null;
    }

    int status = resp.getStatus();
    if (status != 0) {
      out.addError(new OutputError(status, Errno.toString(status)));
      return null;
    }

    return resp;
  }
}
