package com.mapr.cli;

import java.io.IOException;
import java.util.Collections;
import java.util.List;
import java.util.ArrayList;

import org.apache.log4j.Logger;

import com.google.common.collect.ImmutableMap;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.cli.common.NodesCommonUtils;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CLIUsageOnlyCommand;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.IntegerInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.fs.Rpc;
import com.mapr.fs.proto.Common;
import com.mapr.fs.proto.Security.ServerKeyType;
import com.mapr.fs.proto.Security.TicketAndKey;
import com.mapr.fs.proto.Gtrace;
import com.mapr.fs.cldb.util.Util;
import com.mapr.security.MaprSecurityException;
import com.mapr.security.Security;
import com.mapr.security.MutableInt;
import com.mapr.security.JNISecurity.MutableErr;

import com.mapr.login.client.MapRLoginClient;
import com.mapr.login.client.MapRLoginHttpsClient;

import com.mapr.fs.proto.Fileserver;
import com.mapr.fs.proto.Fileserver.GetNumInstancesRequest;
import com.mapr.fs.proto.Fileserver.GetNumInstancesResponse;

public class TraceCommands extends CLIBaseClass implements CLIInterface {

  private static final Logger LOG = Logger.getLogger(TraceCommands.class);
  public static final String PORT_PARAM_NAME = "port";
  public static final String HOST_PARAM_NAME = "host";
  public static final String USER_MODE_PARAM_NAME = "isusermode";
  public static final String MODULE_PARAM_NAME = "module";
  public static final String LEVEL_PARAM_NAME = "level";
  public static final String SIZE_PARAM_NAME = "size";
  public static final String MODE_PARAM_NAME = "mode";

  static final CLICommand dumpCommand =
    new CLICommand(
        "dump",
        "usage: trace dump -host IP/Hostname -port PORT",
        TraceCommands.class,
        ExecutionTypeEnum.NATIVE,
        new ImmutableMap.Builder<String, BaseInputParameter>()
        .put(TraceCommands.PORT_PARAM_NAME,
             new IntegerInputParameter(TraceCommands.PORT_PARAM_NAME,
                                    "port",
                                    CLIBaseClass.NOT_REQUIRED,
                                    5660))
        .put(TraceCommands.HOST_PARAM_NAME,
             new TextInputParameter(TraceCommands.HOST_PARAM_NAME,
                                    "ip/hostname",
                                    CLIBaseClass.NOT_REQUIRED,
                                    "127.0.0.1"))
        .build(),        
        null)
    .setShortUsage("trace dump -host IP/Hostname -port PORT");
  
  static final CLICommand resizeCommand =
    new CLICommand(
        "resize",
        "usage: trace resize -host IP/Hostname -port PORT -size inKB",
        TraceCommands.class,
        ExecutionTypeEnum.NATIVE,
        new ImmutableMap.Builder<String, BaseInputParameter>()
        .put(TraceCommands.PORT_PARAM_NAME,
             new IntegerInputParameter(TraceCommands.PORT_PARAM_NAME,
                                    "port",
                                    CLIBaseClass.NOT_REQUIRED,
                                    5660))
        .put(TraceCommands.HOST_PARAM_NAME,
             new TextInputParameter(TraceCommands.HOST_PARAM_NAME,
                                    "ip/hostname",
                                    CLIBaseClass.NOT_REQUIRED,
                                    "127.0.0.1"))
        .put(TraceCommands.SIZE_PARAM_NAME,
             new IntegerInputParameter(TraceCommands.SIZE_PARAM_NAME,
                                       "size",
                                       CLIBaseClass.REQUIRED,
                                       1<<21))
        .build(),        
        null)
    .setShortUsage("trace resize -host IP/Hostname -port PORT -size inKB");


  static final CLICommand setLevelCommand =
    new CLICommand(
        "setlevel",
        "usage: trace setlevel -host IP/Hostname -port PORT -module <name/all> -level <level>",
        TraceCommands.class,
        ExecutionTypeEnum.NATIVE,
        new ImmutableMap.Builder<String, BaseInputParameter>()
        .put(TraceCommands.PORT_PARAM_NAME,
             new IntegerInputParameter(TraceCommands.PORT_PARAM_NAME,
                                    "port",
                                    CLIBaseClass.NOT_REQUIRED,
                                    5660))
        .put(TraceCommands.HOST_PARAM_NAME,
             new TextInputParameter(TraceCommands.HOST_PARAM_NAME,
                                    "ip/hostname",
                                    CLIBaseClass.NOT_REQUIRED,
                                    "127.0.0.1"))
        .put(TraceCommands.MODULE_PARAM_NAME,
             new TextInputParameter(TraceCommands.MODULE_PARAM_NAME,
                                    "module",
                                    CLIBaseClass.REQUIRED,
                                    "all"))
        .put(TraceCommands.LEVEL_PARAM_NAME,
             new TextInputParameter(TraceCommands.LEVEL_PARAM_NAME,
                                    "level",
                                    CLIBaseClass.REQUIRED,
                                    "Info"))
        .build(),        
        null)
    .setShortUsage("trace setlevel -host IP/Hostname -port PORT -module <name/all> -level <level>");
  

  static final CLICommand setModeCommand =
    new CLICommand(
        "setmode",
        "usage: trace setmode -host IP/Hostname -port PORT -mode MODE(default/continuous)",
        TraceCommands.class,
        ExecutionTypeEnum.NATIVE,
        new ImmutableMap.Builder<String, BaseInputParameter>()
        .put(TraceCommands.PORT_PARAM_NAME,
             new IntegerInputParameter(TraceCommands.PORT_PARAM_NAME,
                                    "port",
                                    CLIBaseClass.NOT_REQUIRED,
                                    5660))
        .put(TraceCommands.HOST_PARAM_NAME,
             new TextInputParameter(TraceCommands.HOST_PARAM_NAME,
                                    "ip/hostname",
                                    CLIBaseClass.NOT_REQUIRED,
                                    "127.0.0.1"))
        .put(TraceCommands.MODE_PARAM_NAME,
             new TextInputParameter(TraceCommands.MODE_PARAM_NAME,
                                    "ip",
                                    CLIBaseClass.REQUIRED,
                                    "default"))
        .build(),        
        null)
    .setShortUsage("trace setmode -host IP/Hostname -port PORT -mode MODE");
  
  static final CLICommand resetCommand =
    new CLICommand(
        "reset",
        "usage: trace reset -host IP/Hostname -port PORT",
        TraceCommands.class,
        ExecutionTypeEnum.NATIVE,
        new ImmutableMap.Builder<String, BaseInputParameter>()
        .put(TraceCommands.PORT_PARAM_NAME,
             new IntegerInputParameter(TraceCommands.PORT_PARAM_NAME,
                                    "port",
                                    CLIBaseClass.NOT_REQUIRED,
                                    5660))
        .put(TraceCommands.HOST_PARAM_NAME,
             new TextInputParameter(TraceCommands.HOST_PARAM_NAME,
                                    "ip/hostname",
                                    CLIBaseClass.NOT_REQUIRED,
                                    "127.0.0.1"))
        .build(),        
        null)
    .setShortUsage("trace reset -host IP/Hostname -port PORT");

  static final CLICommand infoCommand =
    new CLICommand(
        "info",
        "usage: trace info -host IP/Hostname -port PORT",
        TraceCommands.class,
        ExecutionTypeEnum.NATIVE,
        new ImmutableMap.Builder<String, BaseInputParameter>()
        .put(TraceCommands.PORT_PARAM_NAME,
             new IntegerInputParameter(TraceCommands.PORT_PARAM_NAME,
                                    "port",
                                    CLIBaseClass.NOT_REQUIRED,
                                    5660))
        .put(TraceCommands.HOST_PARAM_NAME,
             new TextInputParameter(TraceCommands.HOST_PARAM_NAME,
                                    "ip/hostname",
                                    CLIBaseClass.NOT_REQUIRED,
                                    "127.0.0.1"))
        .put(TraceCommands.USER_MODE_PARAM_NAME,
             new TextInputParameter(TraceCommands.USER_MODE_PARAM_NAME,
                                    "isusermode",
                                    CLIBaseClass.NOT_REQUIRED,
                                    "false"))
        .build(),        
        null)
    .setShortUsage("trace info -host IP/Hostname -port PORT");

  static final CLICommand printCommand =
    new CLICommand(
        "print",
        "usage: trace print -host IP/Hostname -port PORT -size inKB (max 64)",
        TraceCommands.class,
        ExecutionTypeEnum.NATIVE,
        new ImmutableMap.Builder<String, BaseInputParameter>()
        .put(TraceCommands.PORT_PARAM_NAME,
             new IntegerInputParameter(TraceCommands.PORT_PARAM_NAME,
                                    "port",
                                    CLIBaseClass.NOT_REQUIRED,
                                    5660))
        .put(TraceCommands.HOST_PARAM_NAME,
             new TextInputParameter(TraceCommands.HOST_PARAM_NAME,
                                    "ip/hostname",
                                    CLIBaseClass.NOT_REQUIRED,
                                    "127.0.0.1"))
        .put(TraceCommands.SIZE_PARAM_NAME,
             new IntegerInputParameter(TraceCommands.SIZE_PARAM_NAME,
                                       "size",
                                       CLIBaseClass.REQUIRED,
                                       1<<12))
        .build(),        
        null)
    .setShortUsage("trace print -host IP/Hostname -port PORT -size inKB (max 64)");

  /* main command */
  public static final CLICommand traceCommands = 
    new CLICommand(
                   "trace", "trace [dump|setmode|setlevel|resize|reset|info|print]",
                   CLIUsageOnlyCommand.class,
                   ExecutionTypeEnum.NATIVE,
                   new CLICommand[] { setModeCommand, setLevelCommand, 
                                      dumpCommand, resizeCommand, resetCommand,
                                      infoCommand, printCommand})
    .setShortUsage("trace [dump|setmode|setlevel|resize|reset|info|print] -host x.x.x.x -port <port>");

  String clusterName = null;

  public TraceCommands(ProcessedInput input, CLICommand cliCommand) {
    super(input, cliCommand);
    clusterName = CLDBRpcCommonUtils.getInstance().getCurrentClusterName();
  }

  /** 
   * @return
   */
  List<Integer> getMfsInstancesPorts(int hostIp, int port) throws CLIProcessingException
  {
    List<Integer> portsList = new ArrayList<Integer>();
    
    long binding = Rpc.createBindingFor(hostIp, port, 
        clusterName, ServerKeyType.ServerKey.getNumber());
    byte[] replyData;
    try {
      /* The request protobuf does not have any field, so pass the default instance */
      replyData = Rpc.sendRequest(binding,
          Common.MapRProgramId.FileServerProgramId.getNumber(),
          Fileserver.FSProg.GetNumInstances.getNumber(),
          GetNumInstancesRequest.getDefaultInstance());
      GetNumInstancesResponse resp = GetNumInstancesResponse.parseFrom(replyData);
      if (resp != null) {
        portsList = resp.getPortsList();
        /*
         * If the mfs server sends a null list in the response, treat it
         * as an error in the caller and return an error from the caller.
         */
      }
      else {
        /* 
         * Unsupported RPC. Must be a non-multimfs server. Just issue the 
         * command to the original port.
         */
        portsList.add(port);        
      }
    } catch (Exception e) {
      LOG.error("Exception while obtaining the list of MFS ports ");
      throw new CLIProcessingException(e);
    }
    
    return portsList;
  }

  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    output.setOutput(out);

    if (!super.validateInput()) {
      return output;
    }

    LOG.info("RPC Client Initialize");
    try {
      int port = Rpc.initialize(0, 0, null); // Client
      if (port < 0) {
        throw new IOException("Error in RPC init");
      }
    } catch (Exception e) {
    	LOG.error("Exception in Rpc.initialize");
    	out.addError(new OutputError(Errno.ERPCFAILED, "Exception in Rpc.initialize"));
    }

    try {
      //ensure authenticated
      MapRLoginClient client = new MapRLoginHttpsClient();
      client.authenticateIfNeeded(clusterName);
    } catch (Exception e) {
      LOG.error("Exception unable to authenticate ", e);
      out.addError(new OutputError(Errno.ERPCFAILED, "authentication failed"));
    }

    String hostName = getParamTextValue(HOST_PARAM_NAME, 0);
    if ( !( hostName.equalsIgnoreCase("localhost") || hostName.equalsIgnoreCase("127.0.0.1")) ) {
		List<String> ips = NodesCommonUtils.convertHostToIp(Collections.singletonList(hostName));
		if ( ips.isEmpty() ) {
		  out.addError(new OutputError(Errno.EINVAL, "Can not get valid IP address out of provided name: " + hostName));
		  return output;
	    }
		hostName = ips.get(0);
    }
    int hostip = Util.ipToInt(hostName);

    /*
     * At this point, we have the IP to talk to and also have authenticated.
     * The port information specified by the user will be used to obtain
     * the number of instances and issue command to all those ports.
     */
    
    if (cliCommand.getCommandName().equalsIgnoreCase("dump")) {
      
      /* Step 1: Get the list of all ports */
      List<Integer> mfsPorts = getMfsInstancesPorts(hostip, getParamIntValue(PORT_PARAM_NAME, 0));
      if (mfsPorts == null) {
        out.addError(new OutputError(Errno.EINVAL, "Received null for the list of MFS ports"));
        return output;
      }
      for (Integer port : mfsPorts) {
        long binding = Rpc.createBindingFor(hostip, port, clusterName, ServerKeyType.ServerKey.getNumber());
        Gtrace.GTraceRequest.Builder req = Gtrace.GTraceRequest.newBuilder();
        req.setReqType(Gtrace.GTraceRequestType.dump);
        byte[] replyData;
        try {
          replyData = Rpc.sendRequest(binding,
              Common.MapRProgramId.GTraceProgramId.getNumber(),
              Gtrace.GTraceProg.GTraceProc.getNumber(),
              req.build());
        } catch (MaprSecurityException e) {
          throw new CLIProcessingException(
              "MaprSecurityException " + "Exception", e);
        } catch (Exception e) {
          // usually, dont care about error
        }
      }
    } else if (cliCommand.getCommandName().equalsIgnoreCase("setlevel")) {
      int port = getParamIntValue(PORT_PARAM_NAME, 0);
      long binding = Rpc.createBindingFor(hostip, port, clusterName, ServerKeyType.ServerKey.getNumber());
      Gtrace.GTraceRequest.Builder req = Gtrace.GTraceRequest.newBuilder();
      req.setReqType(Gtrace.GTraceRequestType.setLevel);
      req.setModule(getParamTextValue(MODULE_PARAM_NAME, 0));
      req.setLevel(getParamTextValue(LEVEL_PARAM_NAME, 0));
      byte[] replyData;
      try {
        replyData = Rpc.sendRequest(binding,
            Common.MapRProgramId.GTraceProgramId.getNumber(),
            Gtrace.GTraceProg.GTraceProc.getNumber(),
            req.build());
        Gtrace.GTraceResponse resp = Gtrace.GTraceResponse.parseFrom(replyData);
        checkError(resp.getStatus(), out);
      } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
            "MaprSecurityException " + "Exception", e);
      } catch (Exception e) {
        // usually, dont care about error
      }
    } else if (cliCommand.getCommandName().equalsIgnoreCase("setmode")) {
      int port = getParamIntValue(PORT_PARAM_NAME, 0);
      long binding = Rpc.createBindingFor(hostip, port, clusterName, ServerKeyType.ServerKey.getNumber());
      Gtrace.GTraceRequest.Builder req = Gtrace.GTraceRequest.newBuilder();
      req.setReqType(Gtrace.GTraceRequestType.setMode);
      req.setMode(getParamTextValue(MODE_PARAM_NAME, 0));
      byte[] replyData;
      try {
        replyData = Rpc.sendRequest(binding,
            Common.MapRProgramId.GTraceProgramId.getNumber(),
            Gtrace.GTraceProg.GTraceProc.getNumber(),
            req.build());
        Gtrace.GTraceResponse resp = Gtrace.GTraceResponse.parseFrom(replyData);
        checkError(resp.getStatus(), out);
      } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
            "MaprSecurityException " + "Exception", e);
      } catch (Exception e) {
        // usually, dont care about error
      }
    } else if (cliCommand.getCommandName().equalsIgnoreCase("resize")) {
      int port = getParamIntValue(PORT_PARAM_NAME, 0);
      long binding = Rpc.createBindingFor(hostip, port, clusterName, ServerKeyType.ServerKey.getNumber());
      Gtrace.GTraceRequest.Builder req = Gtrace.GTraceRequest.newBuilder();
      req.setReqType(Gtrace.GTraceRequestType.setSize);
      req.setSize(getParamIntValue(SIZE_PARAM_NAME, 0));
      byte[] replyData;
      try {
        replyData = Rpc.sendRequest(binding,
            Common.MapRProgramId.GTraceProgramId.getNumber(),
            Gtrace.GTraceProg.GTraceProc.getNumber(),
            req.build());
        Gtrace.GTraceResponse resp = Gtrace.GTraceResponse.parseFrom(replyData);
        checkError(resp.getStatus(), out);
      } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
            "MaprSecurityException " + "Exception", e);
      } catch (Exception e) {
        // usually, dont care about error
      }
    } else if (cliCommand.getCommandName().equalsIgnoreCase("reset")) {
      int port = getParamIntValue(PORT_PARAM_NAME, 0);
      long binding = Rpc.createBindingFor(hostip, port, clusterName, ServerKeyType.ServerKey.getNumber());
      Gtrace.GTraceRequest.Builder req = Gtrace.GTraceRequest.newBuilder();
      req.setReqType(Gtrace.GTraceRequestType.reset);
      byte[] replyData;
      try {
        replyData = Rpc.sendRequest(binding,
            Common.MapRProgramId.GTraceProgramId.getNumber(),
            Gtrace.GTraceProg.GTraceProc.getNumber(),
            req.build());
      } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
            "MaprSecurityException " + "Exception", e);
      } catch (Exception e) {
        // usually, dont care about error
      }
    } else if (cliCommand.getCommandName().equalsIgnoreCase("info")) {
      // If NFS is running in user mode, create a ticket using the user key.
      if (isParamPresent(USER_MODE_PARAM_NAME) &&
          getParamTextValue(USER_MODE_PARAM_NAME, 0).equalsIgnoreCase("true")) {
        MutableInt err = CreateNFSUserTicket(clusterName);

        if (err.GetValue() != 0) {
          out.addError(new OutputError(Errno.INTERROR,
                                     "Cannot create the ticket for nfs in user mode"));
          return output;
        }
      }

      List<Integer> mfsPorts = getMfsInstancesPorts(hostip, getParamIntValue(PORT_PARAM_NAME, 0));
      if (mfsPorts == null) {
        out.addError(new OutputError(Errno.EINVAL, "Received null for the list of MFS ports"));
        return output;
      }
      for (Integer port: mfsPorts) {
        long binding = Rpc.createBindingFor(hostip, port, clusterName, ServerKeyType.ServerKey.getNumber());
        Gtrace.GTraceRequest.Builder req = Gtrace.GTraceRequest.newBuilder();
        req.setReqType(Gtrace.GTraceRequestType.info);
        byte[] replyData;
        try {
          replyData = Rpc.sendRequest(binding,
              Common.MapRProgramId.GTraceProgramId.getNumber(),
              Gtrace.GTraceProg.GTraceProc.getNumber(),
              req.build());
          Gtrace.GTraceResponse resp = Gtrace.GTraceResponse.parseFrom(replyData);
          System.out.println("** [Port No: " + port + "] Trace is in " + resp.getMode() + " mode.");
          System.out.println("**Allowed Trace Levels are: ");
          for (String l : resp.getLevelsList()) {
            System.out.println(l);
          }
          System.out.println("**Trace buffer size: " + resp.getSize());
          System.out.println("**Modules and levels: ");
          for (Gtrace.ModuleDetail md : resp.getModulesList()) {
            System.out.println(md.getName() + " : " + md.getLevel());
          }
        } catch (MaprSecurityException e) {
          throw new CLIProcessingException(
              "MaprSecurityException " + "Exception", e);
        } catch (Exception e) {
          // usually, dont care about error
        }
      }
    } else if (cliCommand.getCommandName().equalsIgnoreCase("print")) {
      List<Integer> mfsPorts = getMfsInstancesPorts(hostip, getParamIntValue(PORT_PARAM_NAME, 0));
      if (mfsPorts == null) {
        out.addError(new OutputError(Errno.EINVAL, "Received null for the list of MFS ports"));
        return output;
      }
      for (Integer port: mfsPorts) {
        long binding = Rpc.createBindingFor(hostip, port, clusterName, ServerKeyType.ServerKey.getNumber());
        Gtrace.GTraceRequest.Builder req = Gtrace.GTraceRequest.newBuilder();
        req.setReqType(Gtrace.GTraceRequestType.print);
        req.setSize(getParamIntValue(SIZE_PARAM_NAME, 0));
        byte[] replyData;
        try {
          replyData = Rpc.sendRequest(binding,
              Common.MapRProgramId.GTraceProgramId.getNumber(),
              Gtrace.GTraceProg.GTraceProc.getNumber(),
              req.build());
          Gtrace.GTraceResponse resp = Gtrace.GTraceResponse.parseFrom(replyData);
          System.out.println("-----------------------------------------------------");
          System.out.println("[Port No: " + port + "] " + resp.getTracedata().toStringUtf8());
          System.out.println("-----------------------------------------------------");
          System.out.println("DONE");
        } catch (MaprSecurityException e) {
          throw new CLIProcessingException(
              "MaprSecurityException " + "Exception", e);
        } catch (Exception e) {
          // usually, dont care about error
        }
      }
    }
    return output;
  }

  private MutableInt CreateNFSUserTicket(String clusterName) {
    MutableInt err = new MutableInt();
    TicketAndKey ticketAndKey = Security.GetTicketAndKeyForCluster(
        ServerKeyType.ServerKey, clusterName, err);
    if (err.GetValue() != 0) {
      return err;
    }
  
    // Update the security layer to use the user key as server key.
    Security.SetKey(ServerKeyType.ServerKey, ticketAndKey.getUserKey());

    // some random gid. this will not be used anywhere.
    String userName = ticketAndKey.getUserCreds().getUserName();
    int[] gids = new int[1];
    gids[0] = userName.length();
    
    ticketAndKey = Security.GenerateTicketAndKey(ServerKeyType.ServerKey,
        userName, userName.length(), gids, Security.MAX_EXPIRY_TIME,
        0 /*non-renewal*/, true, false /*canUserImpersonate*/, err);
    if (err.GetValue() != 0) {
      return err;
    }

    // Set the new ticket as the server ticket.
    Security.SetTicketAndKey(ServerKeyType.ServerKey,
        clusterName, ticketAndKey);
    return err;
  }
 
  void checkError(int status, OutputHierarchy out) {
    /* TODO when linux error -> java error is done add error messages */
    if (status == 22) {
    	out.addError(new OutputError(status, "Operation failed :(EINVAL) Invalid parameters"));
    } 
    if (status == 12) {
      System.out.println("Operation failed :(ENOMEM) could not allocate memory");
      out.addError(new OutputError(status, "Operation failed :(ENOMEM) could not allocate memory"));
    } 
  }

  @Override
  public String getCommandUsage() {
    return "trace [dump|setmode|setlevel|resize|reset|info|print]";
  }
}
