package com.mapr.cli;

import java.util.Map;
import org.apache.log4j.Logger;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.InvalidProtocolBufferException;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CLIUsageOnlyCommand;
import com.mapr.cliframework.base.CommandLineOutput;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.TextCommandOutput;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.NoValueInputParameter;
import com.mapr.cliframework.base.inputparams.IntegerInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;

import com.mapr.fs.proto.Common;
import com.mapr.fs.cldb.proto.CLDBProto.CLDBProg;
import com.mapr.fs.cldb.proto.CLDBProto.TedActionRequest;
import com.mapr.fs.cldb.proto.CLDBProto.TedActionResponse;
import com.mapr.fs.proto.Security.CredentialsMsg;
import com.mapr.fs.proto.Security.SecurityPrincipal;
import com.mapr.security.UnixUserGroupHelper;
import com.mapr.security.MaprSecurityException;

public class TedCommands extends CLIBaseClass implements CLIInterface {
  private static final Logger LOG = Logger.getLogger(TedCommands.class);
  static final String TED_ID_PARAM_NAME = "id";
  static final String OBJECT_PARAM_NAME = "object";
  static final String SET_VALUE_PARAM_NAME = "setValue";
  
  static final Map<String, BaseInputParameter> params = new ImmutableMap.Builder<String, BaseInputParameter>()
  .put(MapRCliUtil.CLUSTER_NAME_PARAM,
      new TextInputParameter(MapRCliUtil.CLUSTER_NAME_PARAM,
          "cluster name",
          CLIBaseClass.NOT_REQUIRED,
          null))
  .build();
  
  static final CLICommand enableCommand = new CLICommand(
      "enable",
      "enable tedaction for an object",
      TedCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
        .putAll(TedCommands.params)
        .put(TedCommands.TED_ID_PARAM_NAME,
            new IntegerInputParameter(TedCommands.TED_ID_PARAM_NAME,
                "tedaction to enable",
                CLIBaseClass.REQUIRED,
                null))
        .put(TedCommands.OBJECT_PARAM_NAME,
            new TextInputParameter(TedCommands.OBJECT_PARAM_NAME,
                "object on which to enable",
                CLIBaseClass.NOT_REQUIRED,
                null))
        .put(TedCommands.SET_VALUE_PARAM_NAME,
            new TextInputParameter(TedCommands.SET_VALUE_PARAM_NAME,
                "delta to apply",
                CLIBaseClass.NOT_REQUIRED,
                null))
        .build(),
      null)
    .setShortUsage("enable -id <tedaction> -object <ref>");
  
  static final CLICommand enableOnceCommand = new CLICommand(
      "enableonce",
      "enable tedaction once for an object",
      TedCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
        .putAll(TedCommands.params)
        .put(TedCommands.TED_ID_PARAM_NAME,
            new IntegerInputParameter(TedCommands.TED_ID_PARAM_NAME,
                "tedaction to enable",
                CLIBaseClass.REQUIRED,
                null))
        .put(TedCommands.OBJECT_PARAM_NAME,
            new TextInputParameter(TedCommands.OBJECT_PARAM_NAME,
                "object on which to enable",
                CLIBaseClass.NOT_REQUIRED,
                null))
        .put(TedCommands.SET_VALUE_PARAM_NAME,
            new TextInputParameter(TedCommands.SET_VALUE_PARAM_NAME,
                "absolute value to set",
                CLIBaseClass.NOT_REQUIRED,
                null))
        .build(),
      null)
    .setShortUsage("enableonce -id <tedaction> -object <ref>");
  
  static final CLICommand disableCommand = new CLICommand(
      "disable",
      "disable tedaction for an object",
      TedCommands.class,
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
        .putAll(TedCommands.params)
        .put(TedCommands.TED_ID_PARAM_NAME,
            new IntegerInputParameter(TedCommands.TED_ID_PARAM_NAME,
                "tedaction to disable",
                CLIBaseClass.REQUIRED,
                null))
        .put(TedCommands.OBJECT_PARAM_NAME,
            new TextInputParameter(TedCommands.OBJECT_PARAM_NAME,
                "object on which to disable (default:all)",
                CLIBaseClass.NOT_REQUIRED,
                null))
        .build(),
      null)
    .setShortUsage("disable -id <tedaction> -object <ref>");
  
  static final CLICommand[] subCommands = {
    enableCommand,
    enableOnceCommand,
    disableCommand,
  };
  
  static final String usageStr = "ted [enable|enableonce|disable] -id [tedaction] -object <ref>";
  public static final CLICommand commands = 
    new CLICommand(
        "ted",
        "usage: " + usageStr,
        CLIUsageOnlyCommand.class,
        ExecutionTypeEnum.NATIVE,
        subCommands)
    .setUsageInVisible(true)
    .setShortUsage(usageStr);

  UnixUserGroupHelper userInfo;
  
  public TedCommands(ProcessedInput input, CLICommand cliCommand) {
    super(input, cliCommand);
  }

  CommandOutput enableTedAction() throws CLIProcessingException {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();

    TedActionRequest.Builder reqB = TedActionRequest.newBuilder();
    reqB.setCmd(TedActionRequest.TedActionType.ENABLE);

    int id = getParamIntValue(TED_ID_PARAM_NAME, 0);
    reqB.setId(id);

    if (isParamPresent(OBJECT_PARAM_NAME)) {
      reqB.setObjectRef(getParamTextValue(OBJECT_PARAM_NAME, 0));
    }

    if (isParamPresent(SET_VALUE_PARAM_NAME)) {
      reqB.setValue(getParamTextValue(SET_VALUE_PARAM_NAME, 0));
    }

    reqB.setCreds(getUserCredentials());

    byte[] data = null;
    CLDBRpcCommonUtils cldbRpc = CLDBRpcCommonUtils.getInstance();

    try {
      if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
    	  data =  cldbRpc.sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0), 
                                    Common.MapRProgramId.CldbProgramId.getNumber(),
                                    CLDBProg.TedActionProc.getNumber(),
                                    reqB.build(),
                                    TedActionResponse.class);
      } else {
    	  data =  cldbRpc.sendRequest(Common.MapRProgramId.CldbProgramId.getNumber(),
                                    CLDBProg.TedActionProc.getNumber(),
                                    reqB.build(),
                                    TedActionResponse.class);
      }

      if (data == null) {
    	  out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
    	  output.setOutput(out);
    	  return output;
      }

      TedActionResponse resp = TedActionResponse.parseFrom(data);
      int status = resp.getStatus();
      if (status != 0) {
        out.addError(new OutputError(Errno.EOPFAILED, 
                                     "enabling tedaction failed with error - " 
                                     + resp.getErrMsg()));
        output.setOutput(out);
        return output;          
      }
    } catch (InvalidProtocolBufferException e) {
        throw new CLIProcessingException(
            "InvalidProtocolBufferException " + "Exception", e);
    } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
            "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED, 
          "Error while trying to enable tedaction"));
        output.setOutput(out);
        return output;
    }
    
    return output;
  }

  CommandOutput enableTedActionOnce() throws CLIProcessingException {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();

    TedActionRequest.Builder reqB = TedActionRequest.newBuilder();
    reqB.setCmd(TedActionRequest.TedActionType.ENABLEONCE);

    int id = getParamIntValue(TED_ID_PARAM_NAME, 0);
    reqB.setId(id);

    if (isParamPresent(OBJECT_PARAM_NAME)) {
      reqB.setObjectRef(getParamTextValue(OBJECT_PARAM_NAME, 0));
    }

    if (isParamPresent(SET_VALUE_PARAM_NAME)) {
      reqB.setValue(getParamTextValue(SET_VALUE_PARAM_NAME, 0));
    }

    reqB.setCreds(getUserCredentials());

    byte[] data = null;
    CLDBRpcCommonUtils cldbRpc = CLDBRpcCommonUtils.getInstance();

    try {
      if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
    	  data =  cldbRpc.sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0), 
                                    Common.MapRProgramId.CldbProgramId.getNumber(),
                                    CLDBProg.TedActionProc.getNumber(),
                                    reqB.build(),
                                    TedActionResponse.class);
      } else {
    	  data =  cldbRpc.sendRequest(Common.MapRProgramId.CldbProgramId.getNumber(),
                                    CLDBProg.TedActionProc.getNumber(),
                                    reqB.build(),
                                    TedActionResponse.class);
      }

      if (data == null) {
    	  out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
    	  output.setOutput(out);
    	  return output;
      }

      TedActionResponse resp = TedActionResponse.parseFrom(data);
      int status = resp.getStatus();
      if (status != 0) {
        out.addError(new OutputError(Errno.EOPFAILED, 
                                     "enabling tedaction once failed with error - " 
                                     + resp.getErrMsg()));
        output.setOutput(out);
        return output;          
      }
    } catch (InvalidProtocolBufferException e) {
        throw new CLIProcessingException(
            "InvalidProtocolBufferException " + "Exception", e);
    } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
            "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED, 
          "Error while trying to enable tedaction once"));
        output.setOutput(out);
        return output;
    }
    
    return output;
  }
  
  CommandOutput disableTedAction() throws CLIProcessingException {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();

    TedActionRequest.Builder reqB = TedActionRequest.newBuilder();
    reqB.setCmd(TedActionRequest.TedActionType.DISABLE);

    int id = getParamIntValue(TED_ID_PARAM_NAME, 0);
    reqB.setId(id);

    if (isParamPresent(OBJECT_PARAM_NAME)) {
      reqB.setObjectRef(getParamTextValue(OBJECT_PARAM_NAME, 0));
    }

    reqB.setCreds(getUserCredentials());

    byte[] data = null;
    CLDBRpcCommonUtils cldbRpc = CLDBRpcCommonUtils.getInstance();

    try {
      if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
    	  data =  cldbRpc.sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0), 
                                    Common.MapRProgramId.CldbProgramId.getNumber(),
                                    CLDBProg.TedActionProc.getNumber(),
                                    reqB.build(),
                                    TedActionResponse.class);
      } else {
    	  data =  cldbRpc.sendRequest(Common.MapRProgramId.CldbProgramId.getNumber(),
                                    CLDBProg.TedActionProc.getNumber(),
                                    reqB.build(),
                                    TedActionResponse.class);
      }

      if (data == null) {
    	  out.addError(new OutputError(Errno.ERPCFAILED, "Couldn't connect to the CLDB service"));
    	  output.setOutput(out);
    	  return output;
      }

      TedActionResponse resp = TedActionResponse.parseFrom(data);
      int status = resp.getStatus();
      if (status != 0) {
        out.addError(new OutputError(Errno.EOPFAILED, 
                                     "disabling tedaction failed with error - " 
                                     + resp.getErrMsg()));
        output.setOutput(out);
        return output;          
      }
    } catch (InvalidProtocolBufferException e) {
        throw new CLIProcessingException(
            "InvalidProtocolBufferException " + "Exception", e);
    } catch (MaprSecurityException e) {
        throw new CLIProcessingException(
            "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      out.addError(new OutputError(Errno.EOPFAILED, 
          "Error while trying to disable tedaction"));
        output.setOutput(out);
        return output;
    }
    
    return output;
  }
  
  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException {
    if (!super.validateInput()) {
      CommandOutput output = new CommandOutput();
      OutputHierarchy out = new OutputHierarchy();
      output.setOutput(out);
      return output;
    }
    userInfo = new UnixUserGroupHelper();

    String command = cliCommand.getCommandName();
    if (command.equalsIgnoreCase("enable")) {
      return enableTedAction();
    }
    
    if (command.equalsIgnoreCase("enableonce")) {
      return enableTedActionOnce();
    }
    
    if (command.equalsIgnoreCase("disable")) {
      return disableTedAction();
    }
    
    return new TextCommandOutput(("ted command failed: unknown command " + 
        command + " received.").getBytes());
  }
  
  @Override
  public String getCommandUsage() {
    return usageStr;
  }
}
