/* Copyright (c) 2009 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.cli;

import static org.junit.Assert.*;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.BitSet;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.io.IOException;

import org.apache.log4j.Logger;

import com.google.common.collect.ImmutableMap;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.MessageLite;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.fsrpcutils.FSRpcUtils;
import com.mapr.baseutils.fsrpcutils.GetMsgStatus;
import com.mapr.cli.common.ListCommand;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CLIUsageOnlyCommand;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.TextCommandOutput;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.IntegerInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.cliframework.util.FieldInfo;
import com.mapr.cliframework.util.FilterProcessingException;
import com.mapr.cliframework.util.FilterUtil;
import com.mapr.fs.MapRFileSystem;
import com.mapr.fs.Rpc;
import com.mapr.fs.proto.Common.Server;
import com.mapr.fs.AceHelper;
import com.mapr.fs.cldb.proto.CLDBProto.*;
import com.mapr.fs.cldb.VolumeUtils;
import com.mapr.fs.cldb.proto.CLDBProto;
import com.mapr.fs.cldb.proto.CLDBProto.ContainerInfo;
import com.mapr.fs.cldb.proto.CLDBProto.SnapshotInfo;
import com.mapr.fs.cldb.proto.CLDBProto.SnapshotInfoFields;
import com.mapr.fs.cldb.proto.CLDBProto.SnapshotListRequest;
import com.mapr.fs.cldb.proto.CLDBProto.SnapshotListResponse;
import com.mapr.fs.cldb.proto.CLDBProto.SnapshotRemoveRequest;
import com.mapr.fs.cldb.proto.CLDBProto.SnapshotRemoveResponse;
import com.mapr.fs.cldb.proto.CLDBProto.SnapshotsPreserveRequest;
import com.mapr.fs.cldb.proto.CLDBProto.SnapshotsPreserveResponse;
import com.mapr.fs.cldb.proto.CLDBProto.VolumeInfo;
import com.mapr.fs.cldb.proto.CLDBProto.VolumeLookupRequest;
import com.mapr.fs.cldb.proto.CLDBProto.VolumeLookupResponse;
import com.mapr.fs.cli.proto.CLIProto.Filter;
import com.mapr.fs.proto.Common;
import com.mapr.fs.proto.Fileserver;
import com.mapr.fs.proto.Fileserver.FSProg;
import com.mapr.fs.proto.Common.MapRProgramId;
import com.mapr.fs.proto.Security.ServerKeyType;
import com.mapr.fs.proto.Fileserver.GetVolumeAceRequest;
import com.mapr.fs.proto.Fileserver.GetVolumeAceResponse;
import com.mapr.fs.proto.Common.VolumeAceEntry;
import com.mapr.security.UnixUserGroupHelper;
import com.mapr.security.MaprSecurityException;
import com.mapr.fs.proto.Common.VolumeActions;
import com.mapr.fs.proto.Common.VolumeAces;

class GetVolumeAceResponseMsgStatus implements GetMsgStatus {
  GetVolumeAceResponse msg;

  public void init(byte[] data) throws InvalidProtocolBufferException {
    msg = GetVolumeAceResponse.parseFrom(data);
  }

  public int GetStatus() {
    return msg.getStatus();
  }

  public GetVolumeAceResponse GetMsg() {
    return msg;
  }
}

public class SnapshotCommands extends ListCommand
implements CLIInterface {

  private static final Logger LOG = Logger.getLogger(SnapshotCommands.class);
  private static final int NUM_SNAPSHOTS_PER_RPC = 40;

  public FSRpcUtils fsRpcUtils;
  public static final String SNAPSHOT_PARAM_VOL_NAME = "volume";
  public static final String SNAPSHOT_PARAM_VOL_PATH   = "path";
  public static final String RW_VOLUME_PARAM_NAME = "volume";
  public static final String RW_VOLUME_PARAM_MOUNTDIR = "path";
  public static final String SNAPSHOT_PARAM_NAME = "snapshotname";
  public static final String SNAPSHOTS_ID_PARAM_NAME = "snapshots";

  public static final String FILTER_PARAM_NAME = "filter";
  public static final String COLUMNS_PARAM_NAME   = "columns";
  public static final String SORT_PARAM_NAME = "sort";
  public static final String SORT_DIRECTION_PARAM_NAME = "dir";
  public static final String OUTPUT_PARAM_NAME = "output";
  public static final String START_PARAM_NAME = "start";
  public static final String LIMIT_PARAM_NAME = "limit";
  private static int MAX_SNAPFIELDINFO = 0;

  public static String snapshotListUsage = "volume snapshot list [-filter " +
      "filters -cluster clustername]";

  public static final String snapshotCreateUsage = 
      "volume snapshot -volume volName -snapshot snapshotName " +
          "[-cluster clustername]";

  public static String snapshotRemoveUsage = "volume snapshot remove [-volume" +
      " volumename] [-snapshotname snapshotName] [-snapshots comma separated IDs of snapshots]";

  public static String snapshotPreserveUsage = "volume snapshot preserve [-volume" +
      " volumenames] [-path volumepathes] [-snapshots comma separated IDs of snapshots]";

  static final CLICommand snapshotCreateCommand =
      new CLICommand(
          "create", "usage : " + snapshotCreateUsage,
          SnapshotCommands.class, ExecutionTypeEnum.NATIVE,
          new ImmutableMap.Builder<String, BaseInputParameter>()
          .putAll(VolumeCommands.baseParams)
          .put(SnapshotCommands.SNAPSHOT_PARAM_NAME,
              new TextInputParameter(
                  SnapshotCommands.SNAPSHOT_PARAM_NAME,
                  "snapshotName", CLIBaseClass.REQUIRED, null))
          .put(SnapshotCommands.RW_VOLUME_PARAM_NAME,
              new TextInputParameter(
                  SnapshotCommands.RW_VOLUME_PARAM_NAME,
                  "volume", CLIBaseClass.REQUIRED, null))       
          .build(), null)
      .setShortUsage(snapshotCreateUsage);

  static final CLICommand snapshotRemoveCommand = new CLICommand(
      "remove", "usage : " + snapshotRemoveUsage,
      SnapshotCommands.class, ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
      .put(MapRCliUtil.CLUSTER_NAME_PARAM,
          new TextInputParameter(MapRCliUtil.CLUSTER_NAME_PARAM,
              "cluster name",
              CLIBaseClass.NOT_REQUIRED,
              null))
      .put(SnapshotCommands.SNAPSHOT_PARAM_NAME,
          new TextInputParameter(
              SnapshotCommands.SNAPSHOT_PARAM_NAME,
              "snapshotName", CLIBaseClass.NOT_REQUIRED, null))
      .put(SnapshotCommands.RW_VOLUME_PARAM_NAME,
          new TextInputParameter(
              SnapshotCommands.RW_VOLUME_PARAM_NAME,
              "volumeName", CLIBaseClass.NOT_REQUIRED, null))
      .put(SnapshotCommands.SNAPSHOTS_ID_PARAM_NAME,
          new TextInputParameter(
              SnapshotCommands.SNAPSHOTS_ID_PARAM_NAME,
              "comma separated IDs of snapshots", CLIBaseClass.NOT_REQUIRED, null))

      .build(), null)
      .setShortUsage(snapshotRemoveUsage);

  static final CLICommand snapshotPreserveCommand = new CLICommand(
      "preserve", "usage : " + snapshotPreserveUsage,
      SnapshotCommands.class, ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
      .put(MapRCliUtil.CLUSTER_NAME_PARAM,
          new TextInputParameter(MapRCliUtil.CLUSTER_NAME_PARAM,
              "cluster name",
              CLIBaseClass.NOT_REQUIRED,
              null))
      .put(SnapshotCommands.SNAPSHOT_PARAM_VOL_NAME,
          new TextInputParameter(
              SnapshotCommands.SNAPSHOT_PARAM_VOL_NAME,
              "comma separated volume names to preserve snapshots for", CLIBaseClass.NOT_REQUIRED, null))
      .put(SnapshotCommands.SNAPSHOT_PARAM_VOL_PATH,
          new TextInputParameter(
              SnapshotCommands.SNAPSHOT_PARAM_VOL_PATH,
              "comma separated volume pathes to preserve snapshots for", CLIBaseClass.NOT_REQUIRED, null))       
      .put(SnapshotCommands.FILTER_PARAM_NAME,
          new TextInputParameter(
              SnapshotCommands.FILTER_PARAM_NAME,
              "filter to preserve snapshots for", CLIBaseClass.NOT_REQUIRED, null))       
      .put(SnapshotCommands.SNAPSHOTS_ID_PARAM_NAME,
          new TextInputParameter(
              SnapshotCommands.SNAPSHOTS_ID_PARAM_NAME,
              "comma separated IDs of snapshots to preserve", CLIBaseClass.NOT_REQUIRED, null))       
      .build(), null)
      .setShortUsage(snapshotPreserveUsage);

  static final CLICommand snapshotListCommand = new CLICommand(
      "list", "usage : " + snapshotListUsage,
      SnapshotCommands.class, 
      ExecutionTypeEnum.NATIVE,
      new ImmutableMap.Builder<String, BaseInputParameter>()
      .put(MapRCliUtil.CLUSTER_NAME_PARAM,
          new TextInputParameter(MapRCliUtil.CLUSTER_NAME_PARAM,
              "cluster name",
              CLIBaseClass.NOT_REQUIRED,
              null))
      .put(SnapshotCommands.SNAPSHOT_PARAM_VOL_NAME,
          new TextInputParameter(
              SnapshotCommands.SNAPSHOT_PARAM_VOL_NAME,
              "volume",
              CLIBaseClass.NOT_REQUIRED,
              null))
      .put(SnapshotCommands.SNAPSHOT_PARAM_VOL_PATH,
          new TextInputParameter(
              SnapshotCommands.SNAPSHOT_PARAM_VOL_PATH,
              "path",
              CLIBaseClass.NOT_REQUIRED,
              null))   
      .put(VolumeCommands.SORT_PARAM_NAME,
          new TextInputParameter(VolumeCommands.SORT_PARAM_NAME,
              "none",
              CLIBaseClass.NOT_REQUIRED,
              "none").setInvisible(true))
      .put(VolumeCommands.SORT_DIRECTION_PARAM_NAME,
          new TextInputParameter(VolumeCommands.SORT_DIRECTION_PARAM_NAME,
              "none",
              CLIBaseClass.NOT_REQUIRED,
              "ASC").setInvisible(true))
      .put(VolumeCommands.OUTPUT_PARAM_NAME,
          new TextInputParameter(VolumeCommands.OUTPUT_PARAM_NAME,
              "verbose",
              CLIBaseClass.NOT_REQUIRED,
              "verbose"))
      .put(VolumeCommands.START_PARAM_NAME,
          new IntegerInputParameter(VolumeCommands.START_PARAM_NAME,
              "start",
              CLIBaseClass.NOT_REQUIRED,
              0))
      .put(VolumeCommands.LIMIT_PARAM_NAME,
          new IntegerInputParameter(VolumeCommands.LIMIT_PARAM_NAME,
              "limit",
              CLIBaseClass.NOT_REQUIRED,
              Integer.MAX_VALUE))
      .put(VolumeCommands.FILTER_PARAM_NAME,
          new TextInputParameter(VolumeCommands.FILTER_PARAM_NAME,
              "none",
              CLIBaseClass.NOT_REQUIRED,
              "none"))
      .put(VolumeCommands.COLUMNS_PARAM_NAME,
          new TextInputParameter(VolumeCommands.COLUMNS_PARAM_NAME,
              "none",
              CLIBaseClass.NOT_REQUIRED,
              "none"))        
      .build(),
      null       
      ).setShortUsage(snapshotListUsage);

  public static Map<SnapshotInfoFields, FieldInfo> snapshotFieldTable = 
      new ImmutableMap.Builder<SnapshotInfoFields, FieldInfo>()
      .put(SnapshotInfoFields.rwVolumeId, new FieldInfo(
          SnapshotInfoFields.rwVolumeId.getNumber(), 
          "vid", "volumeid", Integer.class))
      .put(SnapshotInfoFields.snapshotId, new FieldInfo(
          SnapshotInfoFields.snapshotId.getNumber(), 
          "id", "snapshotid", Integer.class))
      .put(SnapshotInfoFields.snapshotName, new FieldInfo(
          SnapshotInfoFields.snapshotName.getNumber(), 
          "n", "snapshotname", String.class))
      .put(SnapshotInfoFields.rwVolumeName, new FieldInfo(
          SnapshotInfoFields.rwVolumeName.getNumber(), 
          "vn", "volumename", String.class))
      .put(SnapshotInfoFields.volumePath, new FieldInfo(
          SnapshotInfoFields.volumePath.getNumber(), 
          "vp", "volumepath", String.class))
      .put(SnapshotInfoFields.ownerName, new FieldInfo(
          SnapshotInfoFields.ownerName.getNumber(), 
          "on", "ownername", String.class))
      .put(SnapshotInfoFields.ownerType, new FieldInfo(
          SnapshotInfoFields.ownerType.getNumber(), 
          "ot", "ownertype", Integer.class))
      .put(SnapshotInfoFields.creationTime, new FieldInfo(
          SnapshotInfoFields.creationTime.getNumber(), 
          "ct", "creationtime", Long.class))
      .put(SnapshotInfoFields.expiryTime, new FieldInfo(
          SnapshotInfoFields.expiryTime.getNumber(), 
          "et", "expirytime", Long.class))
      .put(SnapshotInfoFields.cumulativeReclaimSizeMB, new FieldInfo(
          SnapshotInfoFields.cumulativeReclaimSizeMB.getNumber(), 
          "cs", "cumulativeReclaimSizeMB", Long.class))
      .build();

  static {
    SnapshotInfoFields [] values = SnapshotInfoFields.values();
    int max = 0;
    for ( SnapshotInfoFields value : values ) {
      if ( value.getNumber() > max )
        max = value.getNumber();
    }
    MAX_SNAPFIELDINFO = max;
  }

  public static CLICommand[] snapshotCommandsArray = {
      snapshotCreateCommand, 
      snapshotRemoveCommand,
      snapshotPreserveCommand,
      snapshotListCommand};

  public static CLICommand snapshotCommands = new CLICommand(
      "snapshot", "snapshot", CLIUsageOnlyCommand.class,
      ExecutionTypeEnum.NATIVE, 
      snapshotCommandsArray)
      .setShortUsage("snapshot [create|list|remove|preserve]");

  //  public static Map<SnapshotInfoFields, FieldInfo> snapshotFieldTable =
  //    new ImmutableMap.Builder<SnapshotInfoFields, FieldInfo>()
  //    .put(SnapshotInfoFields.rwVolumeName, new FieldInfo(SnapshotInfoFields.rwVolumeName.getNumber(), 
  //        "vn", "volumename", String.class))
  //    .put(SnapshotInfoFields.rwVolumeName, new FieldInfo(SnapshotInfoFields.rwVolumeName.getNumber(), 
  //        "vn", "volumename", String.class))    
  //    .build();
  UnixUserGroupHelper uInfo = null;

  public SnapshotCommands(ProcessedInput input, 
      CLICommand cliCommand) 
          throws CLIProcessingException {
    super(input, cliCommand);
    uInfo = new UnixUserGroupHelper();
  }

  private long getBindingForContainer(int cid) throws CLIProcessingException {
    // lookup container in cldb
    int dbHost, dbPort;
    byte[] replyData;
    ContainerLookupRequest creq = ContainerLookupRequest.newBuilder()
        .addContainerId(cid)
        .setCreds(getUserCredentials())
        .build();
    try {
      replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
          MapRProgramId.CldbProgramId.getNumber(),
          CLDBProto.CLDBProg.ContainerLookupProc.getNumber(),
          creq,
          ContainerLookupResponse.class);

      if (replyData == null) {
        // most likely can not connect to CLDB
        LOG.error("Couldn't connect to the CLDB service");
        return -1;
      }

      ContainerLookupResponse resp;
      resp = ContainerLookupResponse.parseFrom(replyData);
      if (resp.getStatus() == 0) {
        Server server = resp.getContainers(0).getMServer();
        dbHost = server.getIps(0).getHost();
        dbPort = server.getIps(0).getPort();
      } else {
        LOG.error("Container lookup failed : Error " + Errno.toString(resp.getStatus()));
        return -1;
      }
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      LOG.error("Container lookup failed");
      return -1;
    }

    String clusterName = null;
    clusterName = isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM) ?
        getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0) :
          CLDBRpcCommonUtils.getInstance().getCurrentClusterName();

        return Rpc.createBindingFor(dbHost, dbPort, clusterName, ServerKeyType.ServerKey.getNumber());
  }

  public CommandOutput executeRealCommand() throws CLIProcessingException {
    LOG.debug("Processing CLDBProcessing::executeRealCommand");    

    if (cliCommand.getCommandName().equalsIgnoreCase("list")) {
      OutputHierarchy out = new OutputHierarchy(); 
      CommandOutput output = new CommandOutput();
      output.setOutput(out);
      String clusterName = null;
      clusterName = isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM) ?
          getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0) :
            CLDBRpcCommonUtils.getInstance().getCurrentClusterName();
          fsRpcUtils = new FSRpcUtils(clusterName, getUserCredentials(), 1, ServerKeyType.ServerKey, 0);
          list(out);
          return output;
    } else if (cliCommand.getCommandName().equalsIgnoreCase("create")) {
      try {
        return snapshotCreate();
      } catch (Exception e) {
        throw new CLIProcessingException("Send request Exception", e);
      } 
    } else if (cliCommand.getCommandName().equalsIgnoreCase("remove")) {
      try {
        return snapshotRemove();
      } catch (Exception e) {
        throw new CLIProcessingException("Send request Exception", e);
      } 
    } else if (cliCommand.getCommandName().equalsIgnoreCase("preserve")) {
      try {
        return snapshotPreserve();
      } catch (Exception e) {
        throw new CLIProcessingException("Send request Exception", e);
      } 
    }    
    return new TextCommandOutput(("Volume command failed").getBytes());
  }  

  CommandOutput snapshotRemove() throws CLIProcessingException {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    output.setOutput(out);

    if (isParamPresent(SnapshotCommands.SNAPSHOTS_ID_PARAM_NAME)) {
      String snapshotIds = getParamTextValue(SnapshotCommands
          .SNAPSHOTS_ID_PARAM_NAME, 0);
      String[] ids = snapshotIds.split(VolumeCommands.MULTI_ARG_SEP);
      for (String id : ids) {
        SnapshotRemoveRequest.Builder snapshotRemove = SnapshotRemoveRequest
            .newBuilder();
        try {
          snapshotRemove.setSnapshotId(Integer.valueOf(id));
        } catch (NumberFormatException e) {
          out.addError(new OutputError(Errno.EINVAL, "Error parsing snapshot IDs"));
          break;
        }
        int status = snapshotRemoveInternal(snapshotRemove);
        if (status == 0) {
          LOG.info("Snapshot removed for " + id);
        } else {
          LOG.error("Snapshot remove failed with status " + status);
          if (status == Errno.EINPROGRESS) {
            out.addError(new OutputError(Errno.EOPFAILED,
                "Snapshot Remove: Operation failed, " + "snapshot can't be created "
                    + "or removed when mirroring is in progress for the volume"));
          } else {
            out.addError(new OutputError(status, "Snapshot Remove:  " + Errno.toString(status)));
          }  
        }
      }
    }

    if (isParamPresent(RW_VOLUME_PARAM_NAME) && isParamPresent(SNAPSHOT_PARAM_NAME)) {
      String rwVolumeName = getParamTextValue(
          SnapshotCommands.RW_VOLUME_PARAM_NAME, 0);
      String snapshotName = getParamTextValue(SNAPSHOT_PARAM_NAME, 0);
      SnapshotRemoveRequest.Builder snapshotRemove = SnapshotRemoveRequest
          .newBuilder();
      snapshotRemove.setRemoveNow(true)
      .setRwVolumeName(rwVolumeName)
      .setSnapshotName(snapshotName);
      int status = snapshotRemoveInternal(snapshotRemove);
      if (status == 0) {
        LOG.info("Snapshot removed for volume " + rwVolumeName + " snapshot name " +
            snapshotName);
      } else {
        LOG.error("Snapshot for volume " + rwVolumeName + " snapshot name " +
            snapshotName +" failed with status " + status);
        if (status == Errno.EINPROGRESS) {
          out.addError(new OutputError(Errno.EOPFAILED,
              "Snapshot Remove: Operation failed, " + "snapshot can't be created "
                  + "or removed when mirroring is in progress for the volume"));
        } else {
          out.addError(new OutputError(status, "Snapshot Remove:  " + Errno.toString(status)));
        }  
      }
    }
    if (!(isParamPresent(RW_VOLUME_PARAM_NAME) && isParamPresent(SNAPSHOT_PARAM_NAME)) &&
        !isParamPresent(SnapshotCommands.SNAPSHOTS_ID_PARAM_NAME)) {
      out.addError(new OutputError(Errno.EINVAL, "Please specify both snapshot name and volume name, or just snapshot id.\n" +
          snapshotRemoveUsage));
    }
    return output;
  }

  private int snapshotRemoveInternal(SnapshotRemoveRequest.Builder snapshotRemoveBuilder) throws CLIProcessingException {
    byte[] data = null;
    try {
      snapshotRemoveBuilder.setCreds(getUserCredentials());
      if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        data = CLDBRpcCommonUtils.getInstance().sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProto.CLDBProg.SnapshotRemoveProc.getNumber(), snapshotRemoveBuilder.build(), SnapshotRemoveResponse.class);
      } else {
        data = CLDBRpcCommonUtils.getInstance().sendRequest(
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProto.CLDBProg.SnapshotRemoveProc.getNumber(), snapshotRemoveBuilder.build(), SnapshotRemoveResponse.class);
      }

      if (data == null) {
        throw new CLIProcessingException("Exception while processing RPC");
      }
      SnapshotRemoveResponse resp = SnapshotRemoveResponse.parseFrom(data);
      if (resp.getStatus() == 0) {
        LOG.info("Snapshot removed succeded ");
        return 0;
      } else {
        LOG.error("Snapshot remove " + 
            " failed with status " + resp.getStatus());
        return resp.getStatus();
      }
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      LOG.error("Send request Exception", e);
      return Errno.EOPFAILED;
    }
  }

  CommandOutput snapshotPreserve() throws CLIProcessingException {     
    OutputHierarchy out = new OutputHierarchy(); 
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    boolean isVolumeParamPresent = isParamPresent(RW_VOLUME_PARAM_NAME);
    boolean isPathParamPresent = isParamPresent(SNAPSHOT_PARAM_VOL_PATH);
    boolean isFilterParamPresent = isParamPresent(FILTER_PARAM_NAME);
    boolean isSnapshotIdspParamPresent = isParamPresent(SNAPSHOTS_ID_PARAM_NAME);

    if ( isVolumeParamPresent ||
        isPathParamPresent || 
        isFilterParamPresent ||
        isSnapshotIdspParamPresent ) {
    } else {
      out.addError(new OutputError(Errno.EINVAL, "Neither of parameters is passed."));
      return output;
    }

    SnapshotsPreserveRequest.Builder snapPreserveReqBuilder = SnapshotsPreserveRequest.newBuilder();

    if ( isVolumeParamPresent ) {
      String volumeNamesStr = getParamTextValue(RW_VOLUME_PARAM_NAME, 0);
      String[] volumeNames = volumeNamesStr.split(",");
      if ( volumeNames == null ) {
        out.addError(new OutputError(Errno.EINVAL, "Volumes were not provided").setField(RW_VOLUME_PARAM_NAME));
        return output;
      }
      for ( String volumeName : volumeNames ) {
        SnapshotInfo sInfo = SnapshotInfo.newBuilder().setRwVolumeName(volumeName).build();
        snapPreserveReqBuilder.addSnapshotInfos(sInfo);
      }
    } else if ( isSnapshotIdspParamPresent ) {
      String snapshotStringIdsStr = getParamTextValue(SNAPSHOTS_ID_PARAM_NAME, 0);
      String[] snapshotStringIds = snapshotStringIdsStr.split(",");
      if ( snapshotStringIds == null ) {
        out.addError(new OutputError(Errno.EINVAL, "SnapshotIds were not provided").setField(SNAPSHOTS_ID_PARAM_NAME));
        return output;
      }

      for (String snapshotIdStr : snapshotStringIds ) {
        try {
          Integer snapId = Integer.valueOf(snapshotIdStr);
          SnapshotInfo sInfo = SnapshotInfo.newBuilder().setSnapshotId(snapId).build();
          snapPreserveReqBuilder.addSnapshotInfos(sInfo);
        } catch(NumberFormatException nfe) {
          LOG.error("SnapshotID is not Integer: " + snapshotIdStr + ". Processing the rest");
          out.addError(new OutputError(Errno.EINVAL, "SnapshotID is not Integer: " + snapshotIdStr + ". Processing the rest"));
          continue;
        }
      }
    } else if (isFilterParamPresent) {
      List<Filter> filters = new ArrayList<Filter>();
      String filterString = getParamTextValue(FILTER_PARAM_NAME, 0);
      if (filterString != null && (!filterString.equals("none"))) {
        try {
          List<String> filterStrings = new ArrayList<String>();
          filterStrings.add(filterString);
          filters = FilterUtil.compileFilter(snapshotFieldTable, filterStrings);
        } catch (FilterProcessingException fpe) {
          LOG.error("Exception while processing filter parameter: " + filterString, fpe);
          out.addError(new OutputError(Errno.EINVAL, "Exception while processing filter parameter: " + filterString).setField(FILTER_PARAM_NAME));
        }
      } 
      snapPreserveReqBuilder.addAllFilter(filters);
    } else if ( isPathParamPresent ) {
      String svolumePathesStr = getParamTextValue(SNAPSHOT_PARAM_VOL_PATH, 0);
      String[] volumePathes = svolumePathesStr.split(",");
      if ( volumePathes == null ) {
        out.addError(new OutputError(Errno.EINVAL, "Volume Paths were not provided").setField(SNAPSHOT_PARAM_VOL_PATH));
        return output;
      }

      snapPreserveReqBuilder.addAllRwVolumePath(Arrays.asList(volumePathes));
    } 

    byte[] data = null;
    try {
      snapPreserveReqBuilder.setCreds(getUserCredentials());
      if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        data = CLDBRpcCommonUtils.getInstance().sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProto.CLDBProg.SnapshotsPreserveProc.getNumber(),
            snapPreserveReqBuilder.build(), SnapshotsPreserveResponse.class);
      } else {
        data = CLDBRpcCommonUtils.getInstance().sendRequest(
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProto.CLDBProg.SnapshotsPreserveProc.getNumber(),
            snapPreserveReqBuilder.build(), SnapshotsPreserveResponse.class);
      }

      if (data == null) {
        LOG.error("SnapshotPreserve : Exception while processing RPC");
        out.addError(new OutputError(Errno.ERPCFAILED,"Couldn't connect to the CLDB service to do SnapshotPreserve"));
        return output;
      }
      SnapshotsPreserveResponse resp = SnapshotsPreserveResponse.parseFrom(data);
      if ((resp.getStatus() == 0) ||
          (resp.getStatus() == Errno.EBADF)) {
        // status == EBADF is partial success
        if (resp.getVolumesWithoutPermsCount() > 0) {
          String warnMsg = "Warning: Caller does not have permissions for volumes ";
          boolean addComma = false;
          for(String volName : resp.getVolumesWithoutPermsList()) {
            if (addComma) warnMsg = warnMsg + ", " + volName;
            else {
              warnMsg = warnMsg + volName;
              addComma = true;
            }
          }
          out.addMessage(warnMsg);
        }
      } else if (resp.getStatus() == Errno.E_NOT_ENOUGH_PRIVILEGES) {
        // could not preserve any snapshots since the caller does
        // not have permissions
        out.addError(new OutputError(Errno.EOPFAILED, "Failed to preserve snapshots since the caller does not have privileges"));
      } else {
        out.addError(new OutputError(Errno.EOPFAILED, "Snapshots Preserve Operation failed")); 
      }
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
    } catch(Exception e) {
      LOG.error("Exception while trying to preserve snapshots", e);
      out.addError(new OutputError(Errno.ERPCFAILED, "Exception while trying to preserve snapshots"));
    }

    return output;
  }

  private BitSet getColumns() throws CLIProcessingException {
    BitSet columns = new BitSet(snapshotFieldTable.size());
    // set all bits to 1 for whole filedMap
    columns.set(0, MAX_SNAPFIELDINFO+1);
    //long columns = 0xffffffff;

    String columnsString = getParamTextValue(COLUMNS_PARAM_NAME, 0);
    if (columnsString != null && (!columnsString.equals("none"))) {
      columns = FilterUtil.getColumns(snapshotFieldTable, columnsString.trim());
    } 
    return columns;
  }

  private SnapshotListRequest.Builder getSnapshotListRequestBuilder() throws CLIProcessingException {
    SnapshotListRequest.Builder snapList = SnapshotListRequest.newBuilder();
    snapList.addAllFilter(getFilters(snapshotFieldTable, FILTER_PARAM_NAME));
    snapList.setLimiter(getNextLimiter(getParamIntValue(START_PARAM_NAME, 0), 0, 
        getParamIntValue(START_PARAM_NAME, 0), getParamIntValue(LIMIT_PARAM_NAME, 0), NUM_SNAPSHOTS_PER_RPC));

    snapList.setCreds(getUserCredentials());
    if (isParamPresent(SNAPSHOT_PARAM_VOL_NAME)) {
      String volumeNames = getParamTextValue(SNAPSHOT_PARAM_VOL_NAME, 0);
      snapList.addAllRwVolumeNames(Arrays.asList(volumeNames.split(VolumeCommands.MULTI_ARG_SEP)));
    } else if (isParamPresent(SNAPSHOT_PARAM_VOL_PATH)) {
      String volumePaths = getParamTextValue(SNAPSHOT_PARAM_VOL_PATH, 0);
      snapList.addAllRwVolumePaths(Arrays.asList(volumePaths.split(VolumeCommands.MULTI_ARG_SEP)));
    }
    return snapList;
  }

  private SnapshotListResponse getSnapshotListResponse(byte[] replyData) throws CLIProcessingException {
    try {
      return SnapshotListResponse.parseFrom(replyData);
    } catch (InvalidProtocolBufferException ipbe) {
      throw new CLIProcessingException("Exception while parsing the RPC " +
          "response data into SnapshotListResponse proto object.", ipbe);
    }
  }

  @Override
  public SnapshotListRequest buildNextRequest(MessageLite prevReq, MessageLite prevResp) throws CLIProcessingException {
    SnapshotListRequest.Builder newReqBuilder = null;
    if (prevReq != null) {
      newReqBuilder = SnapshotListRequest.newBuilder((SnapshotListRequest) prevReq);
    } else {
      newReqBuilder = getSnapshotListRequestBuilder();
    }

    if (prevResp != null) {
      int prevStart = newReqBuilder.getLimiter().getStart();
      int prevCount = ((SnapshotListResponse) prevResp).getSnapshotsCount();
      int origStart = getParamIntValue(START_PARAM_NAME, 0);
      int origLimit = getParamIntValue(LIMIT_PARAM_NAME, 0);
      newReqBuilder.setLimiter(getNextLimiter(prevStart, prevCount, 
          origStart, origLimit, NUM_SNAPSHOTS_PER_RPC));
    }

    return newReqBuilder.build();
  }

  @Override
  public boolean hasMore(MessageLite prevReq, MessageLite prevResp) throws CLIProcessingException {
    return hasMore(getParamIntValue(START_PARAM_NAME, 0), 
        getParamIntValue(LIMIT_PARAM_NAME, 0), 
        ((SnapshotListRequest) prevReq).getLimiter().getStart(), 
        ((SnapshotListResponse) prevResp).getSnapshotsCount());
  }

  @Override
  public void processResponse(OutputHierarchy out, MessageLite response) throws CLIProcessingException {
    SnapshotListResponse resp = (SnapshotListResponse) response;
    boolean terse = getParamTextValue(OUTPUT_PARAM_NAME, 0).equals("terse");
    BitSet columns = getColumns();

    if (resp.getStatus() == 0) {
      for (SnapshotInfo s : resp.getSnapshotsList()) {
        try {
          OutputNode snapInfo = formatSnapshotInfo(s, terse, columns);
          if (s.getDeleteInProg() || s.getSnapshotInProgress()) {
            if (s.getDeleteInProg())
              snapInfo.addChild(new OutputNode("snapDeleteInProgress","1"));
            else
              snapInfo.addChild(new OutputNode("snapCreateInProgress","1"));
          } else {
            OutputNode volumeAcesNode = new OutputNode("volumeSnapshotAces");
            //Fetch volume aces for snapshot
            GetVolumeAceRequest req = GetVolumeAceRequest.newBuilder()
                .setCid(s.getRootContainerId())
                .build();
            GetVolumeAceResponseMsgStatus msgStatus =
                new GetVolumeAceResponseMsgStatus();
            int status = fsRpcUtils.SendRequestToCid(s.getRootContainerId(),
                false,
                Common.MapRProgramId.FileServerProgramId.getNumber(),
                FSProg.GetVolumeAceProc.getNumber(),
                req, msgStatus);
            //the fileserver doesnt support this proc.
            if (status == Errno.ENOSYS || status == Errno.EAGAIN) {
              LOG.debug("GetVolumeAce RPC returned " + status);
              out.addNode(snapInfo);
              continue;
            }
            if (status != 0) {
              LOG.error("Got null reply from RPC, status :" + status);
              out.addError(new OutputError(Errno.ERPCFAILED, "GetVolumeAce rpc failed"));
              return;
            }
            GetVolumeAceResponse getvolAceResp = msgStatus.GetMsg();
            if (getvolAceResp.getStatus() == 0) {
              if (getvolAceResp.getVolumeAces().getAcesCount() == 0) {
                LOG.debug("GetVolumeAce RPC returned 0 , using deafult.");
                volumeAcesNode.addNode(new OutputNode(VolumeCommands.VOL_READACE_PARAM, "p"));
                volumeAcesNode.addNode(new OutputNode(VolumeCommands.VOL_WRITEACE_PARAM, "p"));
              } else {
                for (VolumeAceEntry volumeAceEntry : getvolAceResp.getVolumeAces().getAcesList()) {
                  LOG.debug("GetVolumeAce: " + volumeAceEntry.getAccessType().name() + " : " +
                      AceHelper.toInfix(volumeAceEntry.getExpr().toStringUtf8()));
                  if (volumeAceEntry.getAccessType() == VolumeActions.VOLUME_READ) {
                    volumeAcesNode.addNode(new OutputNode(
                          VolumeCommands.VOL_READACE_PARAM,
                          AceHelper.toInfix(volumeAceEntry.getExpr().toStringUtf8())));
                  } else if (volumeAceEntry.getAccessType() == VolumeActions.VOLUME_WRITE) {
                    volumeAcesNode.addNode(new OutputNode(
                          VolumeCommands.VOL_WRITEACE_PARAM,
                          AceHelper.toInfix(volumeAceEntry.getExpr().toStringUtf8())));
                  } else {
                    assertNotNull(null);
                  }
                } //for
              }
            } else if (getvolAceResp.getStatus() == Errno.ENOENT) {
              LOG.debug("GetVolumeAce RPC returned ENOENT , using deafult.");
              volumeAcesNode.addNode(new OutputNode(VolumeCommands.VOL_READACE_PARAM, "p"));
              volumeAcesNode.addNode(new OutputNode(VolumeCommands.VOL_WRITEACE_PARAM, "p"));
            }
            snapInfo.addChild(volumeAcesNode);
          }
          out.addNode(snapInfo);
        } catch(CLIProcessingException e) {
          // Catch any errors trying to process snapshot columns
          out.addError(new OutputError(Errno.EINVAL,
              e.getMessage()));
        } catch(Exception e) {
          LOG.error("Exception while trying to list snapshots", e);
          out.addError(new OutputError(Errno.EINVAL, "Exception while trying to list snapshots"));
        }
      }
      if (resp.hasTotal()) {
        out.setTotal(resp.getTotal());
      }
    } else  {
      out.addError(new OutputError(resp.getStatus(), Errno.toString(resp.getStatus())));
    }
  }

  @Override
  public SnapshotListResponse sendRequest(MessageLite request) throws CLIProcessingException {
    SnapshotListRequest req = (SnapshotListRequest) request;
    byte[] replyData = null;
    try {
      if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        replyData = CLDBRpcCommonUtils.getInstance().sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProto.CLDBProg.SnapshotListProc.getNumber(),
            req, SnapshotListResponse.class);
      } else {
        replyData = CLDBRpcCommonUtils.getInstance().sendRequest(
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProto.CLDBProg.SnapshotListProc.getNumber(),
            req, SnapshotListResponse.class);
      }
    } catch (Exception e) {
      throw new CLIProcessingException(e);
    }

    if (replyData != null) {
      return getSnapshotListResponse(replyData);
    } else {
      LOG.error("RPC Request to list Nodes failed. No data returned");
      return null;
    }
  }

  CommandOutput snapshotCreate() throws CLIProcessingException {
    CommandOutput output = new CommandOutput();
    OutputHierarchy out = new OutputHierarchy();
    output.setOutput(out);
    List<String> rwVolumeNames = new ArrayList<String>();

    String name = getParamTextValue(
        SnapshotCommands.RW_VOLUME_PARAM_NAME, 0);
    if (name.contains(VolumeCommands.MULTI_ARG_SEP)) {
      rwVolumeNames.addAll(Arrays.asList(name.split(VolumeCommands.MULTI_ARG_SEP))); 
    } else {
      rwVolumeNames.add(name);
    }

    MapRFileSystem fs = MapRCliUtil.getMapRFileSystem();
    String cluster = null;
    if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
      cluster = getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0);
    }
    for (String rwVolumeName : rwVolumeNames) {
      String snapshotName = getParamTextValue(SNAPSHOT_PARAM_NAME, 0);
      // Lookup Volume
      VolumeInfo vInfo = volumeLookup(rwVolumeName);
      if (vInfo == null) {
        out.addError(new OutputError(Errno.ENOENT, "Snapshot Create: " +
            "Could not find information about Volume: '" +
            rwVolumeName + "'. "  + Errno.toString(Errno.ENOENT)));
        LOG.error("SnapshotCreate: Could not find Volume Info of " + 
            rwVolumeName);     
        continue;
      }
      // Check to see if name length exceeds
      if (!VolumeUtils.isValidSnapshotNameLength(snapshotName)) {
        out.addError(new OutputError(Errno.EINVAL,
            "Invalid Snapshot Name " + snapshotName +
            ", Exceeds allowed length of " + 
            VolumeUtils.validSnapshotNameLength() + " characters")
            .setField(SNAPSHOT_PARAM_NAME));      
        return output;
      }
      // Check if it is valid snapshot name
      if (!VolumeUtils.isValidSnapshotName(snapshotName)) {
        out.addError(new OutputError(Errno.EINVAL,
            "SnapshotCreate: Invalid Snapshot Name. "+
                " Allowed characters " + VolumeUtils.getValidName())
            .setField(SNAPSHOT_PARAM_NAME) );
        output.setOutput(out);
        continue;  
      }

      // Make sure we got Master for RootContainer
      ContainerInfo rootCInfo = vInfo.getRootContainer();
      if (!rootCInfo.hasMServer()) {
        out.addError(new OutputError(Errno.ENODATA, "Snapshot Create: " +
            "Could not find master server for root of Volume: " + 
            rwVolumeName + Errno.toString(Errno.ENODATA)));
        LOG.error("SnapshotCreate: Could not find master server for " +
            "Volume Info of " + rwVolumeName);     
        continue;
      }

      int status = fs.createSnapshot(cluster, rwVolumeName, 
          vInfo.getVolProperties().getVolumeId(), 
          vInfo.getRootContainer().getContainerId(),
          snapshotName,
          false,
          getUserLoginId());

      if (status == Errno.EINPROGRESS) {
        out.addError(new OutputError(Errno.EOPFAILED,
            "Snapshot Create: Operation failed, " + "snapshot can't be created "
                + "or removed when mirroring is in progress for the volume"));
        continue;
      }
      if (status != 0) {    
        out.addError(new OutputError(Errno.EOPFAILED,
            "Snapshot Create: Operation failed, " + Errno.toString(status)));
        continue;
      }
    }
    return output;
  }

  OutputNode formatSnapshotInfo(SnapshotInfo v, boolean terse, BitSet columns) throws CLIProcessingException {
    OutputNode snapshotInfo = new OutputNode();

    if (columns.get(SnapshotInfoFields.ownerName.getNumber())) {
      int ownerId = v.getOwnerId();
      String ownerName = null;
      try {
        ownerName = uInfo.getUsername(ownerId);
      } catch (SecurityException se) {
        ownerName = "Uid " + ownerId;
      }
      snapshotInfo.addChild(new OutputNode(
          snapshotFieldTable.get(SnapshotInfoFields.ownerName).getName(terse), 
          ownerName));
    }
    if (columns.get(SnapshotInfoFields.ownerType.getNumber())) {
      snapshotInfo.addChild(new OutputNode(
          snapshotFieldTable.get(SnapshotInfoFields.ownerType).getName(terse), 
          "1"));
    }
    if (columns.get(SnapshotInfoFields.rwVolumeId.getNumber())) {
      snapshotInfo.addChild(new OutputNode(
          snapshotFieldTable.get(SnapshotInfoFields.rwVolumeId).getName(terse), 
          String.valueOf(v.getRwVolumeId())));
    }

    if (columns.get(SnapshotInfoFields.rwVolumeName.getNumber())) {
      snapshotInfo.addChild(new OutputNode(
          snapshotFieldTable.get(SnapshotInfoFields.rwVolumeName).getName(terse), 
          v.getRwVolumeName()));
    }
    if (columns.get(SnapshotInfoFields.volumePath.getNumber())) {
      snapshotInfo.addChild(new OutputNode(
          snapshotFieldTable.get(SnapshotInfoFields.volumePath).getName(terse), 
          v.getMountDir()));
    }
    if (columns.get(SnapshotInfoFields.snapshotId.getNumber()) ) {
      snapshotInfo.addChild(new OutputNode(
          snapshotFieldTable.get(SnapshotInfoFields.snapshotId).getName(terse), 
          String.valueOf(v.getSnapshotId())));
    }
    if (columns.get(SnapshotInfoFields.snapshotName.getNumber()) ) {
      snapshotInfo.addChild(new OutputNode(
          snapshotFieldTable.get(SnapshotInfoFields.snapshotName).getName(terse), 
          v.getSnapshotName()));
    }
    if (columns.get(SnapshotInfoFields.creationTime.getNumber()) ) {
      String createTime = terse ? String.valueOf(v.getCreateTime()) : 
        new Date(v.getCreateTime()).toString();
      snapshotInfo.addChild(new OutputNode(
          snapshotFieldTable.get(SnapshotInfoFields.creationTime).getName(terse), 
          createTime));
    }
    if (columns.get(SnapshotInfoFields.expiryTime.getNumber()) ) {
      String expiryTime = terse ? String.valueOf(v.getDeleteTime()) : 
        new Date(v.getDeleteTime()).toString();
      if ( terse || v.getDeleteTime() != 0 ) {
        snapshotInfo.addChild(new OutputNode(
            snapshotFieldTable.get(SnapshotInfoFields.expiryTime).getName(terse), 
            expiryTime));
      }
    }
    if (columns.get(SnapshotInfoFields.cumulativeReclaimSizeMB.getNumber())) {
      snapshotInfo.addChild(new OutputNode(
          snapshotFieldTable.get(SnapshotInfoFields.cumulativeReclaimSizeMB).getName(terse),
          String.valueOf(v.getSnapshotCumulativeReclaimSizeMB())));
    }

    // Check if no columns are found, and throw exception if no columns exist
    if (snapshotInfo.getChildren().isEmpty()) {
      throw new CLIProcessingException("No columns found in output");
    }
    return snapshotInfo;
  }

  private VolumeInfo volumeLookup(String volumeName) throws CLIProcessingException {
    VolumeLookupResponse resp = null;
    byte[] data = null;

    VolumeLookupRequest req = VolumeLookupRequest.newBuilder()
        .setVolumeName(volumeName)
        .setCreds(getUserCredentials())
        .build();
    try {
      if ( isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
        data = CLDBRpcCommonUtils.getInstance().sendRequest(getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0),
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProto.CLDBProg.VolumeLookupProc.getNumber(), req, VolumeLookupResponse.class);
      } else {
        data = CLDBRpcCommonUtils.getInstance().sendRequest(
            Common.MapRProgramId.CldbProgramId.getNumber(),
            CLDBProto.CLDBProg.VolumeLookupProc.getNumber(), req, VolumeLookupResponse.class);
      }

      if (data == null) {
        LOG.error("VolumeLookup RPC to CLDB failed for volume " + volumeName);
        return null;
      }
      resp = VolumeLookupResponse.parseFrom(data);
      if (resp.getStatus() == 0) {
        return resp.getVolInfo();
      }
    } catch (MaprSecurityException e) {
      throw new CLIProcessingException(
          "MaprSecurityException " + "Exception", e);
    } catch (Exception e) {
      LOG.error("Exception during VolumeLookup RPC to CLDB " + 
          e.getLocalizedMessage());
      return null;
    }
    return null;
  }  
}
