/* Copyright (c) 2012 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.cli;

import java.io.IOException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.yarn.api.records.ApplicationId;
import org.apache.hadoop.yarn.api.records.ApplicationReport;
import org.apache.hadoop.yarn.client.api.YarnClient;
import org.apache.hadoop.yarn.conf.DefaultYarnConfiguration;
import org.apache.hadoop.yarn.exceptions.ApplicationNotFoundException;
import org.apache.hadoop.yarn.exceptions.YarnException;
import org.apache.hadoop.yarn.util.ConverterUtils;
import org.apache.hadoop.yarn.util.TaskLogUtil;
import org.apache.hadoop.yarn.webapp.log.DFSContainerLogsUtils;

/**
 * Sets up links for logs generated by a YARN application.
 *
 * The logs for each container can be classified into one or more ways.
 * This allows us to easily focus on the logs of interest.
 *
 * 1. Host: Container ran on that host
 * 2. Mapper: Container ran a map task
 * 3. Reducer: Container ran a reduce task
 *
 * So for a mapreduce app, the directory structure will look like this:
 *
 * <pre>
 * /path/to/log/dir/app_id/hosts
 * /path/to/log/dir/app_id/hosts/host1/container1/syslog
 * /path/to/log/dir/app_id/hosts/host1/container1/stdout
 * /path/to/log/dir/app_id/hosts/host1/container1/stderr
 *
 * /path/to/log/dir/app_id/hosts/host2/container2/syslog
 * /path/to/log/dir/app_id/hosts/host2/container2/stdout
 * /path/to/log/dir/app_id/hosts/host2/container2/stderr
 *
 * /path/to/log/dir/app_id/mappers
 * /path/to/log/dir/app_id/mappers/container1/syslog
 * /path/to/log/dir/app_id/mappers/container1/stdout
 * /path/to/log/dir/app_id/mappers/container1/stderr
 *
 * /path/to/log/dir/app_id/reducers
 * /path/to/log/dir/app_id/reducers/container1/syslog
 * /path/to/log/dir/app_id/reducers/container1/stdout
 * /path/to/log/dir/app_id/reducers/container1/stderr
 * </pre>
 */
public final class SetupYarnAppLogLinks
{
  private static final Log LOG = LogFactory.getLog(SetupYarnAppLogLinks.class);

  private final String appIdStr;

  private final String targetBaseDir;

  public SetupYarnAppLogLinks(String appIdStr, String targetBaseDir) {
    this.appIdStr = appIdStr;
    this.targetBaseDir = targetBaseDir;
  }

  public void run() throws YarnException, IOException {
    if (appIdStr == null) {
      System.err.println("ApplicationId cannot be null!");
      return;
    }

    ApplicationId appId = null;
    try {
      appId = ConverterUtils.toApplicationId(appIdStr);
    } catch (Exception e) {
      System.err.println("Invalid ApplicationId specified");
      return;
    }
    
    YarnClient yarnClient = createYarnClient();
    ApplicationReport appReport = null;

    try {
      appReport = yarnClient.getApplicationReport(appId);
    } catch (ApplicationNotFoundException e) {
      LOG.error("Application not found", e);
      System.err.println("Application not found. Check application id.");
      return;
    }

    if (!isAppDone(appReport)) {
      String msg = "Application has not yet completed." +
          " So you will be viewing partial logs only.";

      LOG.warn(msg);
      System.out.println("WARN: " + msg);
    }

    // TODO
    // Handle map reduce job
    /*
    List<ApplicationAttemptReport> appAttemptReports = yarnClient.getApplicationAttempts(appId);
    for (ApplicationAttemptReport appAttemptReport : appAttemptReports) {
      List<ContainerReport> containerReports = yarnClient.getContainers(appAttemptReport.getApplicationAttemptId());
      for (ContainerReport containerReport : containerReports) {
      }
    }
    */

    // Get all the possible paths where the app logs are stored
    FileSystem fs = FileSystem.get(DefaultYarnConfiguration.get());
    final Path[] appLogs = TaskLogUtil.getDFSLoggingHandler().getLogDir(appId);

    //boolean mapreduce = appReport.getApplicationType()
    //  .equalsIgnoreCase("mapreduce");
    boolean mapreduce = false;
    // Create the base directory structure
    Path targetAppDirPath = JobLogLinkUtils.setupTargetJobDir(
        fs,
        targetBaseDir,
        appIdStr,
        mapreduce);

    if (appLogs == null) {
      return;
    }

    // Create links for each container dir present under the app directory
    for (Path appLogDir : appLogs) {
      String hostName = appLogDir.toUri().getPath().split("/")[4];
      Path hostsDirPath = JobLogLinkUtils.getHostDirPath(targetAppDirPath);
      Path hostDirPath = new Path(hostsDirPath, hostName);

      if (LOG.isDebugEnabled()) {
        LOG.debug("Creating host dir: " + hostDirPath);
      }
      fs.mkdirs(hostDirPath);

      Path[] paths = DFSContainerLogsUtils.getFilesInDir(appLogDir);
      for (Path path : paths) {
        JobLogLinkUtils.createSymlinkIfAbsent(
            fs,
            path,
            new Path(hostDirPath,
              path.getName()));
      }
    }
  }

  private boolean isAppDone(ApplicationReport appReport) {
    switch (appReport.getYarnApplicationState()) {
      case NEW:
      case NEW_SAVING:
      case ACCEPTED:
      case SUBMITTED:
      case RUNNING:
        return false;

      default:
        return true;
    }
  }

  private YarnClient createYarnClient() {
    YarnClient yarnClient = YarnClient.createYarnClient();
    yarnClient.init(DefaultYarnConfiguration.get());
    yarnClient.start();

    return yarnClient;
  }
}
