/* Copyright (c) 2012 & onwards. MapR Tech, Inc., All rights reserved */

package com.mapr.cli;

import com.mapr.baseutils.Errno;

import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.ProcessedInput;

import java.io.IOException;

import java.net.URI;
import java.net.URISyntaxException;

import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.LogFactory;
import org.apache.commons.logging.Log;

import org.apache.hadoop.conf.Configuration;

import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;

import org.apache.hadoop.mapred.DefaultJobHistoryParser;
import org.apache.hadoop.mapred.JobHistory;
import org.apache.hadoop.mapred.JobHistory.JobInfo;
import org.apache.hadoop.mapred.JobHistory.Keys;
import org.apache.hadoop.mapred.JobHistory.Task;
import org.apache.hadoop.mapred.JobHistory.TaskAttempt;
import org.apache.hadoop.mapred.JobHistory.MapAttempt;
import org.apache.hadoop.mapred.JobHistory.ReduceAttempt;

public final class SetupJobLogLinks
  extends    CLIBaseClass
  implements CLIInterface
{
  private static final String CENTRAL_LOGVIEWDIR_PROP =
    "mapr.centrallogview.dir";

  private static final PathFilter jobLogFileFilter = new PathFilter() {
    public boolean accept(Path p) {
      final String pathName = p.getName();
      return !pathName.endsWith(".xml") && !pathName.endsWith("_metrics");
    }
  };
  private static final Log LOG = LogFactory.getLog(SetupJobLogLinks.class);

  private final Configuration conf = new Configuration();
  private Path   jobsDonePath;
  private String localVolumesPath;
  private String centralLogDirName;
  private String centralLogViewDirName;
  private FileSystem maprFS;

  public SetupJobLogLinks(ProcessedInput input, CLICommand cliCommand) {
    super(input, cliCommand);
  }

  private void createLogLinks(
    String jobNamePattern)
    throws IOException
  {
    final Path historyLogDir = maprFS.makeQualified(jobsDonePath);
    if (LOG.isDebugEnabled()) {
      LOG.debug("Parsing history at "
              + historyLogDir
              + " for jobid "
              + jobNamePattern);
    }

    final Path jobGlob = new Path(
      historyLogDir,
        "[^_]*_[0-9]*_"                                         /* host_hash */
      + jobNamePattern                  /* job_<jobTrackerStartTime>_id_user */
      +  "*");                             /* match potentially empty suffix */

    final Path[] jobLogs =
      FileUtil.stat2Paths(maprFS.globStatus(jobGlob, jobLogFileFilter));

    if (LOG.isInfoEnabled()) {
      LOG.info("Found " + jobLogs.length + " matches for " + jobGlob);
    }

    for (int i = 0; i < jobLogs.length; i++) {
      final String fileName =
        JobInfo.decodeJobHistoryFileName(jobLogs[i].toString());
      final String[] jobDetails =  fileName.split("_");

      if (jobDetails.length < 5
       || !"job".equals(jobDetails[2]))
      {
        LOG.warn("Histoty file name matches, but no job id could be parsed.");
        continue;
      }

      final String jobId = jobDetails[2] + "_" + jobDetails[3] + "_" +
        jobDetails[4];

      if (LOG.isDebugEnabled()) {
        LOG.debug("Procesing " + jobId  + " from file: " + jobLogs[i]);
      }

      Path jobviewPath = new Path(
          centralLogViewDirName
        + Path.SEPARATOR
        + jobId);
      if (!jobviewPath.isAbsolute()) {
        jobviewPath = new Path(maprFS.getHomeDirectory(), jobviewPath);
      }

      final Path mappersPath  = new Path(jobviewPath, "mappers");
      final Path reducersPath = new Path(jobviewPath, "reducers");
      final Path hostsPath    = new Path(jobviewPath, "hosts");
      for (Path p : new Path[] {mappersPath, reducersPath, hostsPath}) {
        maprFS.mkdirs(p);
      }

      final JobInfo jobInfo = new JobInfo(jobId);

      DefaultJobHistoryParser.
        parseJobTasks(jobLogs[i].toString(), jobInfo, maprFS);

      final Map<String,Task> taskMap = jobInfo.getAllTasks();
      for (Task t : taskMap.values()) {
        final Set<Map.Entry<String,TaskAttempt>> attempts =
          t.getTaskAttempts().entrySet();
        for (Map.Entry<String,TaskAttempt> te : attempts) {
          final String taskAttemptId = te.getKey();
          final TaskAttempt task = te.getValue();
          final String valhost = task.get(Keys.HOSTNAME);

          if (valhost.isEmpty()) {
            if (LOG.isWarnEnabled()) {
                LOG.warn(
                  "Skipping TaskAttempt record " + taskAttemptId +
                  " due to missing key " + Keys.HOSTNAME);
            }
            continue;
          }

                                             /* getting rid of topology info */
          final int slashPos = valhost.lastIndexOf('/');
          final String host = valhost.substring(slashPos + 1);

          final Path attemptTargetPath = new Path(
              localVolumesPath + Path.SEPARATOR
            + host + Path.SEPARATOR
            + centralLogDirName  + Path.SEPARATOR
            + "mapred/userlogs" + Path.SEPARATOR
            + jobId + Path.SEPARATOR + taskAttemptId);

          final Path currentHostPath = new Path(hostsPath, host);
          maprFS.mkdirs(currentHostPath);
          final Path hostCentricLinkPath =
            new Path(currentHostPath, taskAttemptId);
          JobLogLinkUtils.createSymlinkIfAbsent(maprFS, attemptTargetPath, hostCentricLinkPath);
          Path mapRedLinkPath = null;
          if (task instanceof MapAttempt) {
            mapRedLinkPath = new Path(mappersPath, taskAttemptId);
          } else if (task instanceof ReduceAttempt) {
            mapRedLinkPath = new Path(reducersPath, taskAttemptId);
          }
          if (mapRedLinkPath == null) {
            if (LOG.isDebugEnabled()) {
              LOG.debug("Unknown task type encountered: " + taskAttemptId);
            }
          } else {
            JobLogLinkUtils.createSymlinkIfAbsent(maprFS, attemptTargetPath, mapRedLinkPath);
          }
        }
      }
    }
  }

  private void evalConf() throws IOException {
    jobsDonePath = new Path(conf.get(
      "mapred.job.tracker.history.completed.location",
      "/var/mapr/cluster/mapred/jobTracker/history/done"));

    localVolumesPath = conf.
      get("mapr.localvolumes.path", "/var/mapr/local");

    centralLogDirName = conf.
      get("mapr.centrallog.dir", "logs");

    centralLogViewDirName = conf.
      get(CENTRAL_LOGVIEWDIR_PROP, "/var/mapr/cluster/mapred/logview");

     maprFS = jobsDonePath.getFileSystem(conf);
  }

  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException {
    final OutputHierarchy out = new OutputHierarchy();
    CommandOutput output = new CommandOutput();
    output.setOutput(out);

    final String jobidArg = getParamTextValue(JobCommands.JOBID_PARAM, 0);
    final String jobid = jobidArg == null ? "" : jobidArg.trim();

    final String viewDirArg =
      getParamTextValue(JobCommands.JOB_LOGVIEWROOT_PARAM, 0);
    final String viewDir = viewDirArg == null ? "" : viewDirArg.trim();

    final String confFileArg =
      isParamPresent(JobCommands.JOB_CONF_PARAM) ?
        getParamTextValue(JobCommands.JOB_CONF_PARAM, 0) :
        null;
    final String confFile = confFileArg == null ? "" : confFileArg.trim();

    if (LOG.isDebugEnabled()) {
      LOG.debug(
        "Parsed arguments: " + jobid + ", " + viewDir + ", " + confFile);
    }

    if (jobid.isEmpty()) {
      out.addError(
        new OutputError(Errno.EINVAL, "Invalid job id: <" + jobidArg + ">"));
      return output;
    }

    if (jobid.startsWith("application")) {
      try {
        new SetupYarnAppLogLinks(jobid, viewDirArg).run();
      } catch (Exception e) {
        out.addError(new OutputError(
              Errno.EOPFAILED, "Failed to create job log links." + e.getMessage()));
      }

      return output;
    }

    if (!jobid.startsWith("job")) {
      out.addError(
        new OutputError(Errno.EINVAL,
            "Expected job pattern should begin with "
          + "\"job\"; pattern used : <" + jobidArg + ">"));
      return output;
    }

    if (!confFile.isEmpty()) {
          conf.addResource(new Path(confFile));
    }

    if (viewDir.isEmpty()) {
      out.addError(new OutputError(Errno.EINVAL,
        "Invalid directory name: <" + viewDirArg + ">"));
      return output;
    } else {
      conf.set(CENTRAL_LOGVIEWDIR_PROP, viewDir);
    }

    try {
      evalConf();
      createLogLinks(jobid);
    } catch (IOException e) {
      out.addError(new OutputError(
        Errno.EOPFAILED, "Failed to create job log links." + e.getMessage()));
    }

    return output;
  }
}
