package com.mapr.cli;


import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.apache.log4j.Logger;
import org.apache.hadoop.http.HttpServer;

import com.google.common.collect.ImmutableMap;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.baseutils.BaseUtilsHelper;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.URLProbingUtility;
import com.mapr.cli.common.NodesCommonUtils;
import com.mapr.cli.common.ServicesEnum;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;

import com.mapr.fs.proto.Common.ServiceData;
import com.google.protobuf.InvalidProtocolBufferException;

public class ServiceLinkCommand extends CLIBaseClass implements CLIInterface {

	  public static final String SERVICE_NAME = "name";
	  
	  private static final Logger LOG = Logger.getLogger(ServiceLinkCommand.class);
	  
	  public static final CLICommand serviceLinkCommand = 
		    new CLICommand(
		        "urls",
		        "getting service webserver link",
		        ServiceLinkCommand.class,
		        ExecutionTypeEnum.NATIVE,
		        new ImmutableMap.Builder<String,BaseInputParameter>()
		        .put(MapRCliUtil.CLUSTER_NAME_PARAM,
									new TextInputParameter(MapRCliUtil.CLUSTER_NAME_PARAM,
											"cluster name",
											CLIBaseClass.NOT_REQUIRED,
											null))
				.put(NodeServicesManagementCommand.ZK_CONNECTSTRING, new TextInputParameter(NodeServicesManagementCommand.ZK_CONNECTSTRING, "ZooKeeper Connect String: 'host:port,host:port,host:port,...'", CLIBaseClass.NOT_REQUIRED, null))
				.put(SERVICE_NAME, new TextInputParameter(SERVICE_NAME, "name of the service link is required for", CLIBaseClass.REQUIRED, null)) 
				.build(),null)
		    .setShortUsage("getting service webserver link");

	public ServiceLinkCommand(ProcessedInput input, CLICommand cliCommand) {
		super(input, cliCommand);
	}

	@Override
	public CommandOutput executeRealCommand() throws CLIProcessingException {
    String serviceName = getParamTextValue(SERVICE_NAME, 0);
    String clusterName = null;
    String zkConnectString = null;
 		OutputHierarchy out = new OutputHierarchy();
		CommandOutput co = new CommandOutput(out);
    if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
      clusterName = getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM,0);
    }

    if (isParamPresent(NodesCommonUtils.ZK_CONNECTSTRING)) {
      zkConnectString = getParamTextValue(NodesCommonUtils.ZK_CONNECTSTRING, 0);

    }
    String url = getUrl(out, serviceName, clusterName, zkConnectString, true);
    OutputNode node = new OutputNode();
    node.addChild(new OutputNode("url", url));
    out.addNode(node);
    return co;

  }

  public static String getUrl(OutputHierarchy out, String serviceName, String clusterName, String providedZkConnectString, Boolean runCheck) throws CLIProcessingException {

	  String zkConnectString = null;
	  if (clusterName != null) {
		  zkConnectString = CLDBRpcCommonUtils.getInstance().getZkConnect(clusterName);
	  } else {
		  zkConnectString = CLDBRpcCommonUtils.getInstance().getZkConnect();
	  }
	  if ((zkConnectString == null || zkConnectString.trim().isEmpty()) && providedZkConnectString != null) {
		  zkConnectString = providedZkConnectString;
	  }

	  if (zkConnectString == null) {
		  /**
		   * <MAPR_ERROR>
		   * Message:ZK Connect string is null. Check if ZK Connect string was provided correctly
		   * Function:NodeServicesManagementCommand.executeRealCommand()
		   * Meaning:The ZooKeeper connect string was not specified.
		   * Resolution:Check the command syntax and try again.
		   * </MAPR_ERROR>
		   */
		  out.addError(new OutputError(Errno.EZKCANTCONNECT, "ZK Connect string is null. Check if ZK Connect string was provided correctly"));
		  return null;
	  }

	  //String serviceName = serviceName;
	  Map<String, Properties> nodesProps = NodesCommonUtils.getServiceNodesProperties(zkConnectString, serviceName);
	  if (nodesProps == null) {
		  // most likely service with such name does not exist
		  String msg = "No service found with name: " + serviceName;
		  LOG.error(msg);
		  out.addError(new OutputError(Errno.EINVALMISC, msg));
		  return null;
	  }
	  ServiceData hostInfo = NodesCommonUtils.getServiceMasterData(zkConnectString, serviceName);
	  if (hostInfo == null) {
		  // this service is not even running
		  out.addError(new OutputError(Errno.INTERROR, "Service: " + serviceName + " is not running"));
		  return null;
	  }

	  if (!hostInfo.hasHost()) {
		  // which is kind of impossible
		  out.addError(new OutputError(Errno.INTERROR, "Service: " + serviceName + " has invalid node information: " + hostInfo.getHost()));
		  return null;
	  }

	  // now let's get UI stuff for the service
	  // if it is our service - get it from enum, otherwise from Properties
	  String url = null;
	  // Yarn Resourcemanager needs to be handled differently due to how its HA is implemented.
	  if (serviceName.equals("resourcemanager"))
		  url = BaseUtilsHelper.getUrlScheme() + MapRCliUtil.getRealRMWebAddress(zkConnectString, hostInfo);
	  else
		  url = createURL(hostInfo.getHost(), serviceName, nodesProps.get(hostInfo.getHost()));


	  if (url == null) {
		  out.addError(new OutputError(Errno.EINVALMISC, "Service: " + serviceName + " does not have URL credentials defined"));
		  return null;
	  }

	  if (runCheck) {

		  try {
			  if (URLProbingUtility.isURLReachable(url)) {
				  return url;
			  } else {
				  // need to get the rest of the nodes and check them out
				  Map<String, List<String>> serviceToNodeMap = NodesCommonUtils.serviceToNodesMap(zkConnectString);
				  List<String> nodes = serviceToNodeMap.get(serviceName);
				  for (String znode : nodes) {
					  String nodeUrl = createURL(znode, serviceName, nodesProps.get(znode));
					  if (nodeUrl != null) {
						  if (URLProbingUtility.isURLReachable(nodeUrl)) {
							  return nodeUrl;
						  }
					  }
				  }
				  // if we came here we could not find anything suitable
				  out.addError(new OutputError(Errno.EINVALMISC, "Service: " + serviceName + " does not have reachable URL"));
				  return null;
			  }
		  } catch (Exception e) {
			  // Might throw url processing exception
			  out.addError(new OutputError(Errno.EINVALMISC, "Service: " + serviceName + " does not have reachable URL"));
			  return null;
		  }
	  } else {
			// For certain cases we want to return null so it does not show up in the UI
			if (serviceName.equals("nodemanager")) {
				return null;
			} else {
				return url;
			}
	  }
  }

	private static String createURL(String hostName, String serviceName, Properties nodeProps) {
	   StringBuilder returnURL = new StringBuilder();

	  String uiPortProperty = BaseUtilsHelper.getUIPortProperty();

     returnURL.append(BaseUtilsHelper.getUrlScheme());
	 returnURL.append(hostName);
     returnURL.append(":");

	    try {
	      ServicesEnum serviceEnum = ServicesEnum.valueOf(serviceName);
	      // it is our service
	      if ( serviceEnum.getPort() == null || serviceEnum.getWebLink() == null ) {
	        return null;
	      }
	      // construct URL
	      returnURL.append(serviceEnum.getPort());
	      returnURL.append(serviceEnum.getWebLink());
	      return returnURL.toString();
	    } catch(IllegalArgumentException ex) {
	      // not our service get stuff from Properties
	      if ( nodeProps != null ) {
	        String uri = nodeProps.getProperty("service.uri");
	        String uiPort = nodeProps.getProperty(uiPortProperty);
	        if (uiPort == null) {
	          uiPort = nodeProps.getProperty(BaseUtilsHelper.getStandardUiPortProperty());
	        }
	        if ( uri != null ) {
	          if ( uiPort != null ) {
	            returnURL.append(uiPort);
	          }
	          returnURL.append(uri);
	          return returnURL.toString();
	        }
	      }
	    }
	    return null;
	}	
}
