package com.mapr.cli;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import org.apache.log4j.Logger;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.mapr.baseutils.Errno;
import com.mapr.baseutils.cldbutils.CLDBRpcCommonUtils;
import com.mapr.cli.common.NodesCommonUtils;
import com.mapr.cli.common.ServicesEnum;
import com.mapr.cliframework.base.CLIBaseClass;
import com.mapr.cliframework.base.CLICommand;
import com.mapr.cliframework.base.CLIInterface;
import com.mapr.cliframework.base.CLIProcessingException;
import com.mapr.cliframework.base.CLIUsageOnlyCommand;
import com.mapr.cliframework.base.TextCommandOutput;
import com.mapr.cliframework.base.CommandOutput;
import com.mapr.cliframework.base.ProcessedInput;
import com.mapr.cliframework.base.CLICommand.ExecutionTypeEnum;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputError;
import com.mapr.cliframework.base.CommandOutput.OutputHierarchy.OutputNode;
import com.mapr.cliframework.base.inputparams.BaseInputParameter;
import com.mapr.cliframework.base.inputparams.TextInputParameter;
import com.mapr.cliframework.util.FieldInfo;

import com.mapr.fs.proto.Common.ServiceData;
import com.mapr.fs.proto.Common.ExtendedInfo;
import com.google.protobuf.InvalidProtocolBufferException;

public class ServiceCommands extends CLIBaseClass implements CLIInterface {
  private static final Logger LOG = Logger.getLogger(ServiceCommands.class);
  private static final String INFO_SERVICE_CMD = "info";
  private static final String LIST_SERVICE_CMD = "list";
  private static final String NODE_PARAM_NAME = "node";
  private static final String OUTPUT_PARAM_NAME = "output";

  public static Map<String, BaseInputParameter> serviceListParams = new ImmutableMap.Builder<String, BaseInputParameter>()
      .put(
          ServiceCommands.NODE_PARAM_NAME,
          new TextInputParameter(ServiceCommands.NODE_PARAM_NAME, "Hostname of the node",
              CLIBaseClass.NOT_REQUIRED, "localhost")).put(
          MapRCliUtil.CLUSTER_NAME_PARAM,
          new TextInputParameter(MapRCliUtil.CLUSTER_NAME_PARAM, "cluster name",
              CLIBaseClass.NOT_REQUIRED, null)).put(
          NodesCommonUtils.ZK_CONNECTSTRING,
          new TextInputParameter(NodesCommonUtils.ZK_CONNECTSTRING,
              "ZooKeeper Connect String: 'host:port,host:port,host:port,...'",
              CLIBaseClass.NOT_REQUIRED, null)).put(
          ServiceCommands.OUTPUT_PARAM_NAME,
          new TextInputParameter(ServiceCommands.OUTPUT_PARAM_NAME,
              "terse|verbose. Default: verbose", CLIBaseClass.NOT_REQUIRED, "verbose")).build();

  static final CLICommand serviceListCommand = new CLICommand(LIST_SERVICE_CMD,
      "usage: service " + LIST_SERVICE_CMD + " -node <host name> -output <terse|verbose>", ServiceCommands.class,
      ExecutionTypeEnum.NATIVE, serviceListParams, null)
      .setShortUsage("service " + LIST_SERVICE_CMD + " -node <host name> -output <terse|verbose>");
  
  static final CLICommand infoCommand = new CLICommand(INFO_SERVICE_CMD, "Displays information for a service",
		  ServiceCommands.class, ExecutionTypeEnum.NATIVE,
		  new ImmutableMap.Builder<String, BaseInputParameter>()
		  .put(NodesCommonUtils.ZK_CONNECTSTRING,
				  new TextInputParameter(NodesCommonUtils.ZK_CONNECTSTRING,
						  "ZooKeeper Connect String: 'host:port,host:port,host:port,...'",
						  CLIBaseClass.NOT_REQUIRED, null))
						  .put(NodeServicesManagementCommand.SERVICE_NAME_PARAM,
								  new TextInputParameter(NodeServicesManagementCommand.SERVICE_NAME_PARAM,
										  "service name to perform action on",
										  CLIBaseClass.REQUIRED, null))
										  .build(), null).setShortUsage("service " + INFO_SERVICE_CMD + " -zkconnect <IP:Port> -name <service name>")
          .setUsageInVisible(true);

  public static final CLICommand serviceCommands = new CLICommand("service", "service [" + LIST_SERVICE_CMD + "|" + INFO_SERVICE_CMD + "]",
      CLIUsageOnlyCommand.class, ExecutionTypeEnum.NATIVE, new CLICommand[] { serviceListCommand, infoCommand })
      .setShortUsage("service [" + LIST_SERVICE_CMD + "|" + INFO_SERVICE_CMD + "]");

  public static Map<String, FieldInfo> fieldTable = new ImmutableMap.Builder<String, FieldInfo>()
      .put("name", new FieldInfo(1, "n", "name", String.class)).put("state",
          new FieldInfo(2, "s", "state", Integer.class)).put("version",
          new FieldInfo(3, "v", "version", String.class)).put("logpath",
          new FieldInfo(4, "lp", "logpath", String.class)).put("loglevel",
          new FieldInfo(5, "ll", "loglevel", Integer.class)).put("displayname",
          new FieldInfo(6, "dn", "displayname", String.class)).put("memallocated",
          new FieldInfo(7, "ma", "memallocated", String.class)).build();

  private enum State {
    NOT_CONFIGURED, CONFIGURED, RUNNING, STOPPED, FAILED, STANDBY
  };

  public ServiceCommands(ProcessedInput input, CLICommand cliCommand) {
    super(input, cliCommand);
  }

  @Override
  public CommandOutput executeRealCommand() throws CLIProcessingException {
    String command = cliCommand.getCommandName();
    if (command.equalsIgnoreCase(INFO_SERVICE_CMD)) {
    	return getServiceInfo();
    }
    else if(command.equalsIgnoreCase(LIST_SERVICE_CMD)) {
    	return getServiceList();
    }
    return new TextCommandOutput(("Service command failed: unknown command " + command + " received.").getBytes());
  }

  private State getStoppedFailedState(String service,
      Set<String> runningServices, List<String> nodeNames,
      List<String> nodeIps, Map<String, Properties> serviceNodesProperties) {

    List<String> nodes = new ArrayList<String>();
    if ( nodeNames != null && !nodeNames.isEmpty()) {
      nodes.addAll(nodeNames);
    }
    if ( nodeIps != null && !nodeIps.isEmpty() ) {
      nodes.addAll(nodeIps);
    }
    if ( nodes.isEmpty() ) {
      return null;
    }
    for ( String node : nodes ) {
      Properties props = serviceNodesProperties.get(node);
      if ( props != null ) {
        String actionPropValue = props.getProperty("last.action");
        if ( actionPropValue == null ) {
          continue;
        }
        if ( runningServices == null || runningServices.isEmpty() || !runningServices.contains(service) ) {
          if ( "failed".equalsIgnoreCase(actionPropValue) ) {
            return State.FAILED;
          }
          if ( "stop".equalsIgnoreCase(actionPropValue) ) {
            return State.STOPPED;
          }
        }
      }
    }
    return null;
  }

  private String getServiceMemory(String zkConnectString, String service, List<String> nodeHostNames) {
    for (String hostName : nodeHostNames) {
      Map<String, ServiceData> serviceProp = NodesCommonUtils.getServiceNodeData(zkConnectString, service);
      // Get kvstore if node has cldb running on it
      if (!serviceProp.containsKey(hostName) && service.equalsIgnoreCase("fileserver")) {
        serviceProp = NodesCommonUtils.getServiceNodeData(zkConnectString, "kvstore");
      }
      if (serviceProp.containsKey(hostName)) {
        ServiceData serviceMemory = serviceProp.get(hostName);
			  if ( serviceMemory == null ) {
				  return "";
			  }
			  LOG.info(serviceMemory.getHost() + " " + serviceMemory.getPort() + " " + serviceMemory.getMemory());
			  if(serviceMemory.hasMemory()) {
				  if(serviceMemory.getMemory() == 0.0) {
					  return "Auto";
				  }
				  else {
					  return serviceMemory.getMemory() + "";
				  }
			  }
		  }

	  }

	  return "";
  }

  /**
   * For a given service, determine the state of the service. 1. If a service is
   * in runningServices, then it's state is RUNNING. 2. If a service is not in
   * runningServices, but is in configuredServices, then its state is STOPPED.
   * 3. If a service is not in configuredServices, then its state is
   * NOT_CONFIGURED. 4. It is assumed that there is no state like CONFIGURED.
   *
   * @param service
   *          - name of the service
   * @param runningServices
   *          - set of running services
   * @param configuredServices
   *          - set of configured services
   * @return - {@link State}
   */
  private State getServiceState(String service, Set<String> runningServices,
      Set<String> configuredServices) {
    return runningServices.contains(service) ? State.RUNNING : configuredServices
        .contains(service) ? State.STOPPED : State.NOT_CONFIGURED;
  }

  /**
   * Fetches from zk a map of services configured on all the nodes. Looks up the
   * services configured on the passed in node. Returns the set of services.
   *
   * @param nodeHostName
   *          - hostname of the node
   * @param nodeIps
   *          - all the IPs of the node
   * @param zkConnectString
   *          - connection string to connect to zk
   * @return - a set of services configured on the node
   * @throws CLIProcessingException
   */
  public static Set<String> getConfiguredServices(List<String> nodeHostNames, List<String> nodeIps,
      String zkConnectString) throws CLIProcessingException {
    Map<String, List<String>> nodeToConfiguredServicesMap = NodesCommonUtils
        .findServicesConfiguredHierarchy(zkConnectString);
    Set<String> configuredServices = Sets.newHashSet();

    if (LOG.isDebugEnabled()) {
      LOG.debug("Configured services: " + nodeToConfiguredServicesMap.toString());
    }

    for ( String nodeHostName : nodeHostNames ) {
	    if (nodeHostName != null) {
	      List<String> serviceList = nodeToConfiguredServicesMap.get(nodeHostName);
	      if (serviceList != null && !serviceList.isEmpty()) {
	        configuredServices.addAll(serviceList);
	    }
    }

    if (nodeIps != null && !nodeIps.isEmpty())
      for (String nodeIp : nodeIps) {
        List<String> serviceListIp = nodeToConfiguredServicesMap.get(nodeIp);
        if (serviceListIp != null && !serviceListIp.isEmpty()) {
          configuredServices.addAll(serviceListIp);
        }
      }
    }
    return configuredServices;
  }

  /**
   * Fetches from zk a map of services running on all the nodes. Looks up the
   * services running on the passed in node. Returns the set of services.
   *
   * @param nodeHostName
   *          - hostname of the node
   * @param nodeIps
   *          - all the IPs of the node
   * @param zkConnectString
   *          - connection string to connect to zk
   * @return - a set of services running on the node
   * @throws CLIProcessingException
   */
  public static Set<String> getRunningServices(List<String> nodeHostNames, List<String> nodeIps,
      String zkConnectString) throws CLIProcessingException {
    Map<String, List<String>> nodeToRunningServicesMap = NodesCommonUtils
        .findServicesRunningHierarchy(zkConnectString);
    Set<String> runningServices = Sets.newHashSet();

    if (LOG.isDebugEnabled()) {
      LOG.debug("Running services: " + nodeToRunningServicesMap.toString());
    }

    for ( String nodeHostName : nodeHostNames ) {
	    if (nodeToRunningServicesMap.get(nodeHostName) != null
	        && !nodeToRunningServicesMap.get(nodeHostName).isEmpty()) {
	      runningServices.addAll(nodeToRunningServicesMap.get(nodeHostName));
	    }
    }

    for (String nodeIp : nodeIps) {
      if (nodeToRunningServicesMap.get(nodeIp) != null
          && !nodeToRunningServicesMap.get(nodeIp).isEmpty()) {
        runningServices.addAll(nodeToRunningServicesMap.get(nodeIp));
      }
    }
    return runningServices;
  }

  private String getZkConnectString() throws CLIProcessingException {
    String zkConnectString;
    if (isParamPresent(MapRCliUtil.CLUSTER_NAME_PARAM)) {
      zkConnectString = CLDBRpcCommonUtils.getInstance().getZkConnect(
          getParamTextValue(MapRCliUtil.CLUSTER_NAME_PARAM, 0));
    } else {
      zkConnectString = CLDBRpcCommonUtils.getInstance().getZkConnect();
    }
    if (zkConnectString == null || zkConnectString.trim().isEmpty()) {
      if (isParamPresent(NodesCommonUtils.ZK_CONNECTSTRING)) {
        zkConnectString = getParamTextValue(NodesCommonUtils.ZK_CONNECTSTRING, 0);
      }
    }

    return zkConnectString;
  }

  //Retrieves the Service Data ProtoBuf from ZooKeeper and returns each property
  private CommandOutput getServiceInfo() throws CLIProcessingException {
	  OutputHierarchy oh = new OutputHierarchy();
	  CommandOutput co = new CommandOutput(oh);
	  String zkConnect = getZkConnectString();
    if (zkConnect == null) {
      oh.addError(new OutputError(Errno.ERPCFAILED, "Could not connect to CLDB and no Zookeeper connect string " +
          "provided"));

      co.setOutput(oh);
      return co;
    }
	  String serviceName = getParamTextValue(NodeServicesManagementCommand.SERVICE_NAME_PARAM, 0);
    Map<String, ServiceData> serviceNodeData = NodesCommonUtils.getServiceNodeData(zkConnect, serviceName);

    if(serviceNodeData == null) {
      oh.addError(new OutputError(Errno.EINVAL, "Cannot find service " + serviceName));
      co.setOutput(oh);
      return co;
    }
    ServiceData masterNodeInfo = serviceNodeData.remove("master");
    String masterHostName = null;
    if(masterNodeInfo != null && masterNodeInfo.hasHost())
      masterHostName= masterNodeInfo.getHost();

    for(String nodeName : serviceNodeData.keySet()) {
      OutputNode node = new OutputNode();
      ServiceData serviceData = serviceNodeData.get(nodeName);
      if(serviceData != null) {
        node.addNode(new OutputNode("IsMaster", nodeName.equals(masterHostName) ? "Yes" : "No"));
        node.addNode(new OutputNode("Host", serviceData.hasHost() ? serviceData.getHost() : "Not set"));
        node.addNode(new OutputNode("Port", serviceData.hasPort() ? serviceData.getPort() : "Not set"));
        node.addNode(new OutputNode("Memory", serviceData.hasMemory() ? serviceData.getMemory() : "Not set"));
        node.addNode(new OutputNode("IsRunning", serviceData.hasIsRunning() ? serviceData.getIsRunning() : "Not set"));
        if(serviceData.getExtinfoCount() > 0) {
          OutputNode extInfoNode = new OutputNode("ExtendedInfo");
          for (ExtendedInfo extInfo : serviceData.getExtinfoList()) {
            extInfoNode.addChild(new OutputNode(extInfo.getKey(), extInfo.getValue()));
          }
          node.addChild(extInfoNode);
        }
        oh.addNode(node);
      }
    }

	  co.setOutput(oh);
	  return co;
  }

  private CommandOutput getServiceList() throws CLIProcessingException {
	    OutputHierarchy oh = new OutputHierarchy();
	    CommandOutput co = new CommandOutput(oh);

	    boolean terse = getParamTextValue(OUTPUT_PARAM_NAME, 0).equals("terse");
	    String nodeName = getParamTextValue(NODE_PARAM_NAME, 0);
	    if (nodeName == null || nodeName.isEmpty()) {
	      oh.addError(new OutputError(Errno.EINVAL, "Node's hostname provided is empty/null."));
	      return co;
	    }

	/*    try {
	      nodeName = InetAddress.getByName(nodeName).getCanonicalHostName();
	    } catch (UnknownHostException e) {
	      oh.addError(new OutputError(Errno.EINVAL, "Invalid hostname/ip provided for the node."));
	      return co;
	    }
	*/
	    List<String> nodeIps = NodesCommonUtils.convertHostToIpIncludingLocal(Lists.newArrayList(nodeName));
	    List<String> nodeNames = NodesCommonUtils.convertIpToHost(nodeIps);
	    if ( !nodeNames.contains(nodeName) ) {
	    	nodeNames.add(nodeName);
	    }
	    if (LOG.isDebugEnabled()) {
	      LOG.debug("node hostnames: " + nodeNames + ", IPs - " + nodeIps.toString());
	    }

	    String zkConnectString = getZkConnectString();
      if (zkConnectString == null) {
        oh.addError(new OutputError(Errno.ERPCFAILED, "Could not connect to CLDB and no Zookeeper connect string " +
            "provided"));

        co.setOutput(oh);
        return co;
      }
	    Set<String> runningServices = getRunningServices(nodeNames, nodeIps, zkConnectString);
	    Set<String> configuredServices = getConfiguredServices(nodeNames, nodeIps, zkConnectString);

	    Set<String> clusterWideConfiguredServices = NodesCommonUtils.findConfiguredServicesByServiceHierarchy(zkConnectString).keySet();
	    for ( String service : clusterWideConfiguredServices ) {
	        Map<String, Properties> serviceNodesProperties = NodesCommonUtils.getServiceNodesProperties(zkConnectString, service);
	        State state = getStoppedFailedState(service, runningServices, nodeNames, nodeIps, serviceNodesProperties);
	        String mem = "";
	        if ( state == null ) {
	         state = getServiceState(service, runningServices, configuredServices);
	         if ( state == State.RUNNING) {

	           mem = getServiceMemory(zkConnectString, service, nodeNames);
	         }

	         if ( state == State.STOPPED ) {
	           // most likely it is standby - though not conclusively
	           state = State.STANDBY;
	         }
	        }
	        OutputNode node = new OutputNode();
	        node.addChild(new OutputNode(fieldTable.get("name").getName(terse), service));
	        node.addChild(new OutputNode(fieldTable.get("state").getName(terse), state.ordinal()));
	        if (!mem.equals("")) {
	          node.addChild(new OutputNode(fieldTable.get("memallocated").getName(terse), mem));
	        }
	        try {
				ServicesEnum serviceEnum = ServicesEnum.valueOf(service);
		        if (LOG.isDebugEnabled()) {
		            LOG.debug("node: " + nodeName + "service : " + service + ", state = " + state
		                + ", logpath = " + serviceEnum.getLogPath());
		          }
		          node.addChild(new OutputNode(fieldTable.get("logpath").getName(terse), serviceEnum.getLogPath()));
		          node.addChild(new OutputNode(fieldTable.get("displayname").getName(terse), serviceEnum.getDisplayName()));
	        } catch (IllegalArgumentException ex) {
				// not our service get stuff from Properties
	        	Map<String, Properties> nodesProps = NodesCommonUtils.getServiceNodesProperties(zkConnectString, service);
	        	if ( nodesProps != null && !nodesProps.isEmpty()) {
	        		List<String> nodeNameIps = new ArrayList<String>(nodeNames);
	        		nodeNameIps.addAll(nodeIps);
	        		Properties props = null;
	        		for ( String nodeIp : nodeNameIps ) {
	        		  props = nodesProps.get(nodeIp);
	        		  if ( props != null ) {
	        		    break;
	        		  }
	        		}
	            // Try to get properties from secondary location
		       		if ( props == null ) {
	              props = nodesProps.values().toArray(new Properties[0])[0];
	            }
		    			String logsLocation = props.getProperty("service.logs.location");
		    			if ( logsLocation != null ) {
		    				node.addChild(new OutputNode(fieldTable.get("logpath").getName(terse), logsLocation));
		    			} else {
		    				node.addChild(new OutputNode(fieldTable.get("logpath").getName(terse), "undefined"));
		    			}
		    			String displayName = props.getProperty("service.displayname", service);
		    			node.addChild(new OutputNode(fieldTable.get("displayname").getName(terse), displayName));

	        	}

	        }
	        oh.addNode(node);
	    }

	    return co;
  }

}
